import { refreshToken } from './auth';

const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export interface SellerProfile {
  id: number;
  user_id: string;
  seller_id: string;
  full_name: string;
  email: string;
  phone_number: string;
  date_of_birth?: string;
  gender?: string;
  ownership_type?: string;
  property_count: number;
  property_types: string[];
  years_of_experience: number;
  total_portfolio_value_range?: string;
  selling_method?: string;
  selling_timeline?: string;
  price_flexibility: boolean;
  preferred_communication: string[];
  availability_hours?: string;
  bio?: string;
  profile_image_url?: string;
  portfolio_summary?: string;
  marketing_consent: boolean;
  status: 'active' | 'inactive' | 'suspended';
  verification_level: 'basic' | 'standard' | 'premium' | 'verified';
  kyc_status: 'pending' | 'verified' | 'rejected';
  total_properties_listed: number;
  total_properties_sold: number;
  total_revenue_generated: number;
  average_selling_time_days: number;
  created_at: string;
  updated_at: string;
}

export interface SellerProfileCreate {
  full_name: string;
  email: string;
  phone_number: string;
  date_of_birth?: string;
  gender?: string;
  ownership_type?: string;
  property_count?: number;
  property_types?: string[];
  years_of_experience?: number;
  total_portfolio_value_range?: string;
  selling_method?: string;
  selling_timeline?: string;
  price_flexibility?: boolean;
  preferred_communication?: string[];
  availability_hours?: string;
  bio?: string;
  profile_image_url?: string;
  portfolio_summary?: string;
  marketing_consent?: boolean;
}

const makeRequest = async (url: string, options: RequestInit = {}) => {
  const sellerToken = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
  if (!sellerToken) throw new Error('No seller token');

  const response = await fetch(url, {
    ...options,
    headers: {
      'Authorization': `Bearer ${sellerToken}`,
      'Content-Type': 'application/json',
      ...options.headers,
    },
  });

  if (response.status === 401) {
    const newToken = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
    const retryResponse = await fetch(url, {
      ...options,
      headers: {
        'Authorization': `Bearer ${newToken}`,
        'Content-Type': 'application/json',
        ...options.headers,
      },
    });
    if (!retryResponse.ok) throw new Error('Request failed');
    return retryResponse;
  }

  if (!response.ok) {
    const errorData = await response.json().catch(() => ({}));
    console.error('API Error:', response.status, JSON.stringify(errorData, null, 2));
    throw new Error(errorData.message || errorData.detail?.[0]?.msg || 'Request failed');
  }
  return response;
};

export const getSellerProfile = async (): Promise<SellerProfile> => {
  const response = await makeRequest(`${API_BASE_URL}/seller-profile/me`);
  return await response.json();
};

export const updateSellerProfile = async (profileData: Partial<SellerProfile>): Promise<SellerProfile> => {
  const response = await makeRequest(`${API_BASE_URL}/seller-profile/me`, {
    method: 'PUT',
    body: JSON.stringify(profileData),
  });
  return await response.json();
};

export const completeSellerProfile = async (profileData: { profile_data: SellerProfileCreate }): Promise<SellerProfile> => {
  console.log('Sending profile data:', JSON.stringify(profileData, null, 2));
  const response = await makeRequest(`${API_BASE_URL}/seller-profile/complete`, {
    method: 'POST',
    body: JSON.stringify(profileData),
  });
  return await response.json();
};

export const getSellerProfileCompletionStatus = async (): Promise<any> => {
  const response = await makeRequest(`${API_BASE_URL}/seller-profile/completion-status`);
  return await response.json();
};