import { Transaction, TransactionCreate, TimelineEvent, RefundCreate } from '@/types/transaction';

const API_BASE = 'http://localhost:8090/api';

const getHeaders = () => {
  const token = localStorage.getItem('access_token');
  if (!token) {
    throw new Error('Authentication required. Please log in.');
  }
  return {
    'Content-Type': 'application/json',
    'Authorization': `Bearer ${token}`
  };
};

export const transactionApi = {
  create: async (data: TransactionCreate): Promise<Transaction> => {
    const response = await fetch(`${API_BASE}/transactions/`, {
      method: 'POST',
      headers: getHeaders(),
      body: JSON.stringify(data)
    });
    if (!response.ok) throw new Error('Failed to create transaction');
    return response.json();
  },

  getById: async (transactionId: string): Promise<Transaction> => {
    const response = await fetch(`${API_BASE}/transactions/${transactionId}`, {
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to fetch transaction');
    return response.json();
  },

  list: async (params?: {
    user_id?: string;
    status?: string;
    limit?: number;
    offset?: number;
  }): Promise<Transaction[]> => {
    try {
      const queryParams = new URLSearchParams(params as any).toString();
      const response = await fetch(`${API_BASE}/transactions/?${queryParams}`, {
        headers: getHeaders()
      });
      
      if (response.status === 401) {
        throw new Error('Authentication failed. Please log in again.');
      }
      
      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`Failed to fetch transactions: ${errorText}`);
      }
      
      return response.json();
    } catch (error) {
      if (error.message.includes('Authentication')) {
        // Clear invalid tokens and redirect to login
        localStorage.removeItem('access_token');
        localStorage.removeItem('refresh_token');
        window.location.href = '/login';
      }
      throw error;
    }
  },

  getTimeline: async (transactionId: string): Promise<TimelineEvent[]> => {
    const response = await fetch(`${API_BASE}/transactions/${transactionId}/timeline`, {
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to fetch timeline');
    return response.json();
  },

  initiateRefund: async (transactionId: string, data: Omit<RefundCreate, 'transaction_id'>): Promise<any> => {
    const response = await fetch(`${API_BASE}/transactions/${transactionId}/refund`, {
      method: 'POST',
      headers: getHeaders(),
      body: JSON.stringify({
        transaction_id: transactionId,
        ...data
      })
    });
    if (!response.ok) throw new Error('Failed to initiate refund');
    return response.json();
  }
};
