import { 
  UnitInventory, 
  ProjectInventory, 
  UnitVisualizationRequest, 
  UnitStatusUpdate, 
  UnitDetails, 
  UnitAvailabilityStats,
  StatusColors,
  VisualizationTypeOption,
  VisualizationType,
  UnitStatus
} from '../types/unitInventory';

const API_BASE_URL = 'http://localhost:8090';

class UnitInventoryAPI {
  private getAuthHeaders() {
    const token = localStorage.getItem('access_token');
    return {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${token}`
    };
  }

  async getProjects(): Promise<{ id: string; name: string; location: string; status: string }[]> {
    try {
      console.log('🔄 Trying to fetch real projects list');
      const response = await fetch(`${API_BASE_URL}/api/unit-inventory/projects`);
      
      if (response.ok) {
        const realProjects = await response.json();
        console.log('✅ Real projects loaded:', realProjects);
        return realProjects;
      } else {
        console.log(`⚠️ Projects API failed (${response.status}), using mock projects`);
        return this.getMockProjects();
      }
    } catch (error) {
      console.log('❌ Projects API connection failed, using mock projects:', error);
      return this.getMockProjects();
    }
  }

  private getMockProjects() {
    return [
      {
        id: 'demo_skyline',
        name: 'Skyline Towers',
        location: 'Whitefield, Bangalore',
        status: 'Active'
      },
      {
        id: 'demo_greenvalley',
        name: 'Green Valley Villas',
        location: 'Koramangala, Bangalore',
        status: 'Active'
      },
      {
        id: 'demo_techpark',
        name: 'Tech Park Commercial',
        location: 'Electronic City, Bangalore',
        status: 'Active'
      }
    ];
  }

  async getProjectInventory(projectId: string): Promise<ProjectInventory> {
    try {
      console.log(`🔄 Trying to fetch real data for project: ${projectId}`);
      const response = await fetch(
        `${API_BASE_URL}/api/unit-inventory/projects/${projectId}/inventory`
      );

      if (response.ok) {
        const realData = await response.json();
        console.log('✅ Real backend data loaded:', realData);
        return realData;
      } else {
        console.log(`⚠️ Backend API failed (${response.status}), using mock data`);
        return this.getMockProjectInventory(projectId);
      }
    } catch (error) {
      console.log('❌ Backend connection failed, using mock data:', error);
      return this.getMockProjectInventory(projectId);
    }
  }

  private getMockProjectInventory(projectId: string): ProjectInventory {
    const mockUnits: UnitInventory[] = [
      {
        id: 1,
        project_id: projectId,
        tower: 'A',
        floor: 1,
        unit_number: 'A-101',
        unit_type: '2 BHK',
        area: 1200,
        price: 8500000,
        status: 'Available',
        carpet_area: 1000,
        facing: 'East',
        amenities: ['Parking', 'Gym', 'Swimming Pool'],
        created_by: 'demo',
        created_at: new Date().toISOString()
      },
      {
        id: 2,
        project_id: projectId,
        tower: 'A',
        floor: 1,
        unit_number: 'A-102',
        unit_type: '2 BHK',
        area: 1200,
        price: 8500000,
        status: 'Booked',
        carpet_area: 1000,
        facing: 'West',
        amenities: ['Parking', 'Gym', 'Swimming Pool'],
        booked_by: 'John Doe',
        created_by: 'demo',
        created_at: new Date().toISOString()
      },
      {
        id: 3,
        project_id: projectId,
        tower: 'A',
        floor: 2,
        unit_number: 'A-201',
        unit_type: '3 BHK',
        area: 1500,
        price: 12000000,
        status: 'Sold',
        carpet_area: 1300,
        facing: 'North',
        amenities: ['Parking', 'Gym', 'Swimming Pool', 'Balcony'],
        created_by: 'demo',
        created_at: new Date().toISOString()
      },
      {
        id: 4,
        project_id: projectId,
        tower: 'B',
        floor: 1,
        unit_number: 'B-101',
        unit_type: '1 BHK',
        area: 800,
        price: 6000000,
        status: 'Available',
        carpet_area: 650,
        facing: 'South',
        amenities: ['Parking', 'Gym'],
        created_by: 'demo',
        created_at: new Date().toISOString()
      },
      {
        id: 5,
        project_id: projectId,
        tower: 'B',
        floor: 2,
        unit_number: 'B-201',
        unit_type: '2 BHK',
        area: 1100,
        price: 8000000,
        status: 'On Hold',
        carpet_area: 950,
        facing: 'East',
        amenities: ['Parking', 'Gym', 'Swimming Pool'],
        created_by: 'demo',
        created_at: new Date().toISOString()
      }
    ];

    const projectNames: Record<string, string> = {
      'demo_skyline': 'Skyline Towers',
      'demo_greenvalley': 'Green Valley Villas',
      'demo_techpark': 'Tech Park Commercial'
    };

    return {
      project_id: projectId,
      project_name: projectNames[projectId] || 'Demo Project',
      total_units: mockUnits.length,
      available_units: mockUnits.filter(u => u.status === 'Available').length,
      booked_units: mockUnits.filter(u => u.status === 'Booked').length,
      sold_units: mockUnits.filter(u => u.status === 'Sold').length,
      on_hold_units: mockUnits.filter(u => u.status === 'On Hold').length,
      blocked_units: mockUnits.filter(u => u.status === 'Blocked').length,
      towers: ['A', 'B'],
      floors: [1, 2],
      units: mockUnits,
      tower_summary: {
        'A': { total: 3, available: 1, booked: 1, sold: 1, on_hold: 0, blocked: 0 },
        'B': { total: 2, available: 1, booked: 0, sold: 0, on_hold: 1, blocked: 0 }
      },
      floor_summary: {
        1: { total: 3, available: 2, booked: 1, sold: 0, on_hold: 0, blocked: 0 },
        2: { total: 2, available: 0, booked: 0, sold: 1, on_hold: 1, blocked: 0 }
      }
    };
  }

  async getUnitVisualization(
    projectId: string,
    visualizationType: VisualizationType = VisualizationType.FLOOR_WISE,
    towerFilter?: string,
    floorFilter?: number,
    statusFilter?: UnitStatus[]
  ): Promise<any> {
    try {
      console.log(`🔄 Trying to fetch visualization data for project: ${projectId}`);
      const params = new URLSearchParams({
        visualization_type: visualizationType
      });

      if (towerFilter) params.append('tower_filter', towerFilter);
      if (floorFilter !== undefined) params.append('floor_filter', floorFilter.toString());
      if (statusFilter && statusFilter.length > 0) {
        statusFilter.forEach(status => params.append('status_filter', status));
      }

      const response = await fetch(
        `${API_BASE_URL}/api/unit-inventory/projects/${projectId}/visualization?${params}`,
        {
          method: 'POST'
        }
      );

      if (response.ok) {
        const realData = await response.json();
        console.log('✅ Real visualization data loaded:', realData);
        return realData;
      } else {
        console.log(`⚠️ Visualization API failed (${response.status}), using mock data`);
        const inventory = this.getMockProjectInventory(projectId);
        return this.formatMockVisualization(inventory, visualizationType);
      }
    } catch (error) {
      console.log('❌ Visualization API connection failed, using mock data:', error);
      const inventory = this.getMockProjectInventory(projectId);
      return this.formatMockVisualization(inventory, visualizationType);
    }
  }

  private formatMockVisualization(inventory: ProjectInventory, type: VisualizationType) {
    if (type === VisualizationType.FLOOR_WISE) {
      const floorData: Record<string, UnitInventory[]> = {};
      inventory.units.forEach(unit => {
        const key = `Floor ${unit.floor}`;
        if (!floorData[key]) floorData[key] = [];
        floorData[key].push(unit);
      });
      return { type: 'floor_wise', data: floorData };
    }
    
    if (type === VisualizationType.TOWER_WISE) {
      const towerData: Record<string, UnitInventory[]> = {};
      inventory.units.forEach(unit => {
        const key = `Tower ${unit.tower}`;
        if (!towerData[key]) towerData[key] = [];
        towerData[key].push(unit);
      });
      return { type: 'tower_wise', data: towerData };
    }
    
    return { type: 'grid_layout', data: inventory.units };
  }

  async updateUnitStatus(unitId: number, statusUpdate: Omit<UnitStatusUpdate, 'unit_id'>): Promise<UnitInventory> {
    try {
      const response = await fetch(
        `${API_BASE_URL}/api/unit-inventory/units/${unitId}/status`,
        {
          method: 'PUT',
          headers: this.getAuthHeaders(),
          body: JSON.stringify(statusUpdate)
        }
      );

      if (!response.ok) {
        throw new Error(`Failed to update unit status: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Error updating unit status:', error);
      throw error;
    }
  }

  async getUnitDetails(unitId: number): Promise<UnitDetails> {
    try {
      const response = await fetch(
        `${API_BASE_URL}/api/unit-inventory/units/${unitId}/details`,
        {
          headers: this.getAuthHeaders()
        }
      );

      if (!response.ok) {
        throw new Error(`Failed to fetch unit details: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Error fetching unit details:', error);
      throw error;
    }
  }

  async getAvailabilityStats(projectId: string): Promise<UnitAvailabilityStats> {
    try {
      const response = await fetch(
        `${API_BASE_URL}/api/unit-inventory/projects/${projectId}/stats`,
        {
          headers: this.getAuthHeaders()
        }
      );

      if (!response.ok) {
        throw new Error(`Failed to fetch availability stats: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Error fetching availability stats:', error);
      throw error;
    }
  }

  async getProjectUnits(
    projectId: string,
    filters?: {
      tower?: string;
      floor?: number;
      status?: UnitStatus;
      unit_type?: string;
    }
  ): Promise<{ units: UnitInventory[]; total_count: number; filters_applied: any }> {
    try {
      const params = new URLSearchParams();
      
      if (filters?.tower) params.append('tower', filters.tower);
      if (filters?.floor !== undefined) params.append('floor', filters.floor.toString());
      if (filters?.status) params.append('status', filters.status);
      if (filters?.unit_type) params.append('unit_type', filters.unit_type);

      const response = await fetch(
        `${API_BASE_URL}/api/unit-inventory/projects/${projectId}/units?${params}`,
        {
          headers: this.getAuthHeaders()
        }
      );

      if (!response.ok) {
        throw new Error(`Failed to fetch project units: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Error fetching project units:', error);
      throw error;
    }
  }

  async getStatusColors(): Promise<StatusColors> {
    try {
      const response = await fetch(
        `${API_BASE_URL}/api/unit-inventory/status-colors`
      );

      if (!response.ok) {
        throw new Error(`Failed to fetch status colors: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Error fetching status colors:', error);
      throw error;
    }
  }

  async getVisualizationTypes(): Promise<{ visualization_types: VisualizationTypeOption[] }> {
    try {
      const response = await fetch(
        `${API_BASE_URL}/api/unit-inventory/visualization-types`
      );

      if (!response.ok) {
        throw new Error(`Failed to fetch visualization types: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Error fetching visualization types:', error);
      throw error;
    }
  }
}

export const unitInventoryAPI = new UnitInventoryAPI();