import axios from 'axios';
import { API_BASE_URL } from '../config/api';

const getAuthHeaders = () => {
  const token = localStorage.getItem('access_token');
  return token ? { Authorization: `Bearer ${token}` } : {};
};

export interface UsageMetrics {
  total_users: number;
  active_users_24h: number;
  active_users_7d: number;
  active_users_30d: number;
  total_sessions: number;
  avg_session_duration: number;
  total_events: number;
}

export interface FunnelStep {
  step_name: string;
  event_type: string;
  user_count: number;
  conversion_rate: number;
  drop_off_rate: number;
}

export interface FunnelAnalysis {
  funnel_name: string;
  total_entered: number;
  total_completed: number;
  overall_conversion: number;
  steps: FunnelStep[];
}

export interface UserEngagement {
  user_id: string;
  full_name: string | null;
  email: string | null;
  role: string;
  total_events: number;
  last_active: string;
  engagement_score: number;
  preferred_device: string | null;
}

export interface DropOffAnalysis {
  page_url: string;
  total_visits: number;
  exit_count: number;
  exit_rate: number;
  avg_time_on_page: number;
}

export interface UsageByRole {
  role: string;
  user_count: number;
  total_events: number;
  avg_events_per_user: number;
  active_percentage: number;
}

export interface RecentActivity {
  user_name: string;
  user_email: string;
  event_type: string;
  page_url: string;
  device_type: string;
  created_at: string;
  role: string;
}

export const usageAnalyticsApi = {
  getMetrics: async (days: number = 30, orgId?: string): Promise<UsageMetrics> => {
    const params = new URLSearchParams({ days: days.toString() });
    if (orgId) params.append('org_id', orgId);
    const response = await axios.get(`${API_BASE_URL}/usage-analytics/metrics?${params}`, {
      headers: getAuthHeaders()
    });
    return response.data;
  },

  getFunnelAnalysis: async (funnelType: string = 'property_view', days: number = 30, orgId?: string): Promise<FunnelAnalysis> => {
    const params = new URLSearchParams({ funnel_type: funnelType, days: days.toString() });
    if (orgId) params.append('org_id', orgId);
    const response = await axios.get(`${API_BASE_URL}/usage-analytics/funnel?${params}`, {
      headers: getAuthHeaders()
    });
    return response.data;
  },

  getUserEngagement: async (limit: number = 50, orgId?: string): Promise<UserEngagement[]> => {
    const params = new URLSearchParams({ limit: limit.toString() });
    if (orgId) params.append('org_id', orgId);
    const response = await axios.get(`${API_BASE_URL}/usage-analytics/engagement?${params}`, {
      headers: getAuthHeaders()
    });
    return response.data;
  },

  getDropOffAnalysis: async (days: number = 30, orgId?: string): Promise<DropOffAnalysis[]> => {
    const params = new URLSearchParams({ days: days.toString() });
    if (orgId) params.append('org_id', orgId);
    const response = await axios.get(`${API_BASE_URL}/usage-analytics/dropoff?${params}`, {
      headers: getAuthHeaders()
    });
    return response.data;
  },

  getUsageByRole: async (days: number = 30, orgId?: string): Promise<UsageByRole[]> => {
    const params = new URLSearchParams({ days: days.toString() });
    if (orgId) params.append('org_id', orgId);
    const response = await axios.get(`${API_BASE_URL}/usage-analytics/by-role?${params}`, {
      headers: getAuthHeaders()
    });
    return response.data;
  },

  getRecentActivity: async (limit: number = 50, role?: string, eventType?: string): Promise<RecentActivity[]> => {
    const params = new URLSearchParams({ limit: limit.toString() });
    if (role && role !== 'all') params.append('role', role);
    if (eventType && eventType !== 'all') params.append('event_type', eventType);
    const response = await axios.get(`${API_BASE_URL}/usage-analytics/recent-activity?${params}`, {
      headers: getAuthHeaders()
    });
    return response.data;
  },
};
