const API_BASE = import.meta.env.VITE_API_BASE || '/api';

export interface UserStats {
  saved_properties: number;
  viewed_properties: number;
  inquiries_sent: number;
  wishlisted_properties: number;
  profile_completion: number;
}

export interface Activity {
  id: string;
  activity_type: string;
  property_id: number;
  timestamp: string;
  details: Record<string, any>;
}

export interface SavedProperty {
  id: string;
  property_id: number;
  saved_at: string;
  property?: {
    id: number;
    title: string;
    location: string;
    beds: number;
    baths: number;
    area: string;
    price: string;
    imageUrl?: string;
    isUrgent?: boolean;
    isBidding?: boolean;
  };
}

export interface DashboardOverview {
  user_stats: UserStats;
  recent_activities: Activity[];
  recommended_properties: any[];
}

export interface TrackActivityRequest {
  activity_type: string;
  property_id: number;
  details?: Record<string, any>;
}

export interface Inquiry {
  inquiry_id: string;
  buyer_id: string;
  owner_id: string;
  property_id: number;
  reason_to_buy: string;
  buying_timeline: string;
  interested_in_loan: boolean;
  interested_in_visit: boolean;
  status: string;
  created_at: string;
  property_title: string;
  property_address: string;
  property_price: number;
  buyer_name: string;
  buyer_email: string;
  buyer_phone: string;
  owner_email: string;
  owner_phone: string;
  owner_name: string;
}

export interface InquiriesResponse {
  inquiries: Inquiry[];
  total_count: number;
  page: number;
  limit: number;
}

export const userDashboardAPI = {
  async getOverview(userId: string): Promise<DashboardOverview> {
    const response = await fetch(`${API_BASE}/user-dashboard/overview?user_id=${userId}`);
    if (!response.ok) throw new Error('Failed to fetch overview');
    return response.json();
  },

  async getActivities(userId: string, page = 1, limit = 20): Promise<Activity[]> {
    const response = await fetch(`${API_BASE}/user-dashboard/activities?user_id=${userId}&page=${page}&limit=${limit}`);
    if (!response.ok) throw new Error('Failed to fetch activities');
    return response.json();
  },

  async trackActivity(userId: string, activity: TrackActivityRequest): Promise<string> {
    const response = await fetch(`${API_BASE}/user-dashboard/activities?user_id=${userId}`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(activity)
    });
    if (!response.ok) throw new Error('Failed to track activity');
    return response.json();
  },

  async getSavedProperties(userId: string): Promise<SavedProperty[]> {
    const response = await fetch(`${API_BASE}/user-dashboard/saved-properties?user_id=${userId}`);
    if (!response.ok) throw new Error('Failed to fetch saved properties');
    return response.json();
  },

  async saveProperty(userId: string, propertyId: number): Promise<string> {
    console.log(`Saving property ${propertyId} for user ${userId}`);
    console.log(`API URL: ${API_BASE}/user-dashboard/saved-properties/${propertyId}?user_id=${userId}`);
    const response = await fetch(`${API_BASE}/user-dashboard/saved-properties/${propertyId}?user_id=${userId}`, {
      method: 'POST'
    });
    console.log('Save property response status:', response.status);
    if (!response.ok) {
      const errorText = await response.text();
      console.error('Save property error:', errorText);
      throw new Error(`Failed to save property: ${response.status} ${errorText}`);
    }
    const result = await response.json();
    console.log('Save property result:', result);
    return result;
  },

  async removeSavedProperty(userId: string, propertyId: number): Promise<string> {
    console.log(`Removing saved property ${propertyId} for user ${userId}`);
    const response = await fetch(`${API_BASE}/user-dashboard/saved-properties/${propertyId}?user_id=${userId}`, {
      method: 'DELETE'
    });
    console.log('Remove property response status:', response.status);
    if (!response.ok) {
      const errorText = await response.text();
      console.error('Remove property error:', errorText);
      throw new Error(`Failed to remove saved property: ${response.status} ${errorText}`);
    }
    const result = await response.json();
    console.log('Remove property result:', result);
    return result;
  },

  async getInquiries(userId: string, page = 1, limit = 20): Promise<InquiriesResponse> {
    const response = await fetch(`${API_BASE}/user-dashboard/inquiries?user_id=${userId}&page=${page}&limit=${limit}`);
    if (!response.ok) throw new Error('Failed to fetch inquiries');
    return response.json();
  },

  async getSearchHistory(userId: string, limit = 10): Promise<any[]> {
    try {
      const response = await fetch(`${API_BASE}/search-history/${userId}?limit=${limit}`);
      if (!response.ok) {
        console.warn(`Search history API returned ${response.status}`);
        return [];
      }
      const data = await response.json();
      return Array.isArray(data) ? data : [];
    } catch (error) {
      console.warn('Search history API unavailable:', error);
      return [];
    }
  },

  async getSearchRecommendations(userId: string): Promise<any[]> {
    const response = await fetch(`${API_BASE}/search-history/recommendations/${userId}`);
    if (!response.ok) throw new Error('Failed to fetch search recommendations');
    return response.json();
  },

  async getMarketStats(userId: string, budgetMin: number, budgetMax: number, locations: string[]): Promise<any> {
    const locationsParam = locations.join(',');
    const response = await fetch(`${API_BASE}/user-dashboard/market-stats?user_id=${userId}&budget_min=${budgetMin}&budget_max=${budgetMax}&locations=${encodeURIComponent(locationsParam)}`);
    if (!response.ok) {
      console.warn('Market stats API failed, using fallback');
      return null;
    }
    return response.json();
  },

  async getMatchedProperties(userId: string, limit: number = 10): Promise<any> {
    try {
      const response = await fetch(`${API_BASE}/user-dashboard/matched-properties?user_id=${userId}&limit=${limit}`);
      if (!response.ok) {
        console.warn('Matched properties API failed');
        return { matched_properties: [], total_matches: 0, has_search_history: false };
      }
      return response.json();
    } catch (error) {
      console.error('Error fetching matched properties:', error);
      return { matched_properties: [], total_matches: 0, has_search_history: false };
    }
  }
};