// API request utility removed - using direct fetch calls

const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export interface VisitBooking {
  id?: string;
  property_id: number;
  user_id?: number;
  booking_reference?: string;
  status?: 'scheduled' | 'confirmed' | 'in-progress' | 'completed' | 'cancelled';
  
  // Basic Visit Details
  purpose_of_visit: string;
  visit_date: string;
  visit_start_time: string;
  visit_duration: number;
  
  // Visitor Information
  full_name: string;
  mobile_number: string;
  email?: string;
  number_of_visitors: string;
  
  // Travel & Arrival Details
  mode_of_travel: string;
  expected_arrival_time?: string;
  need_parking: boolean;
  
  // Spots/Stops Inside Property
  spots_to_visit: string[];
  specific_block_unit?: string;
  
  // Meeting Preferences
  want_to_meet_agent: boolean;
  preferred_meeting_time?: string;
  meeting_purpose?: string;
  preferred_representative?: string;
  
  // Special Requirements
  accessibility_needs?: string;
  language_preference: string;
  photography_allowed: boolean;
  need_brochures: boolean;
  
  // Notifications & Reminders
  sms_reminder: boolean;
  whatsapp_reminder: boolean;
  notify_agent: boolean;
  
  // Additional Notes
  custom_requests?: string;
  specific_instructions?: string;
  clarifications?: string;
  
  // Timestamps
  created_at?: string;
  updated_at?: string;
}

export interface Property {
  id: number;
  title: string;
  address: string;
  city: string;
  price?: string;
  type?: string;
  bedrooms?: number;
}

// Get Property Details
export const getPropertyDetails = async (propertyId: number): Promise<Property> => {
  const token = localStorage.getItem('access_token');
  console.log('🏠 Fetching property details for:', propertyId);
  
  const response = await fetch(`${API_BASE}/visit-bookings/properties/${propertyId}`, {
    headers: {
      'Authorization': `Bearer ${token}`,
      'Content-Type': 'application/json'
    }
  });
  
  console.log('📡 Property details response:', response.status);
  
  if (!response.ok) {
    console.error('❌ Failed to fetch property details:', response.status);
    throw new Error('Failed to fetch property details');
  }
  
  const data = await response.json();
  console.log('🏠 Property data:', data);
  return data.property || data;
};

// Create Visit Booking
export const createVisitBooking = async (bookingData: VisitBooking): Promise<any> => {
  const token = localStorage.getItem('access_token');
  const user = JSON.parse(localStorage.getItem('user') || '{}');
  
  const payload = {
    ...bookingData,
    user_id: user.user_id || user.id
  };
  
  console.log('🚀 Sending to API:', `${API_BASE}/visit-bookings/`);
  console.log('📦 Payload:', payload);
  console.log('🔑 Token:', token ? 'Present' : 'Missing');
  
  const response = await fetch(`${API_BASE}/visit-bookings/`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${token}`
    },
    body: JSON.stringify(payload)
  });
  
  console.log('📡 Response status:', response.status);
  
  if (!response.ok) {
    const errorText = await response.text();
    console.error('❌ API Error:', response.status, errorText);
    throw new Error(`Failed to create visit booking: ${response.status} - ${errorText}`);
  }
  
  const result = await response.json();
  console.log('✅ API Success:', result);
  return result;
};

// Get Visit Booking
export const getVisitBooking = async (bookingId: string): Promise<VisitBooking> => {
  const token = localStorage.getItem('token');
  const response = await fetch(`${API_BASE}/api/visit-bookings/${bookingId}`, {
    headers: {
      'Authorization': `Bearer ${token}`
    }
  });
  if (!response.ok) throw new Error('Failed to fetch visit booking');
  return response.json();
};

// Update Visit Booking
export const updateVisitBooking = async (bookingId: string, bookingData: Partial<VisitBooking>): Promise<any> => {
  const token = localStorage.getItem('token');
  const response = await fetch(`${API_BASE}/api/visit-bookings/${bookingId}`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${token}`
    },
    body: JSON.stringify(bookingData)
  });
  if (!response.ok) throw new Error('Failed to update visit booking');
  return response.json();
};

// Get User Bookings (sent by user)
export const getUserBookings = async (userId: number): Promise<VisitBooking[]> => {
  const token = localStorage.getItem('access_token');
  console.log('🔍 Fetching user bookings for:', userId);
  
  const response = await fetch(`${API_BASE}/visit-bookings/user/${userId}`, {
    headers: {
      'Authorization': `Bearer ${token}`,
      'Content-Type': 'application/json'
    }
  });
  
  console.log('📡 User bookings response:', response.status);
  
  if (!response.ok) {
    console.error('❌ Failed to fetch user bookings:', response.status);
    throw new Error('Failed to fetch user bookings');
  }
  
  const data = await response.json();
  console.log('📊 User bookings data:', data);
  return data.bookings || data.results || data || [];
};

// Get Received Bookings (for properties owned by user)
export const getReceivedBookings = async (userId: number): Promise<VisitBooking[]> => {
  const token = localStorage.getItem('access_token');
  console.log('🔍 Fetching received bookings for property owner:', userId);
  
  // This should be a separate endpoint that gets bookings for properties owned by the user
  const response = await fetch(`${API_BASE}/visit-bookings/received/${userId}`, {
    headers: {
      'Authorization': `Bearer ${token}`,
      'Content-Type': 'application/json'
    }
  });
  
  console.log('📡 Received bookings response:', response.status);
  
  if (!response.ok) {
    console.error('❌ Failed to fetch received bookings:', response.status);
    // Fallback: filter from localStorage based on property ownership
    const localBookings = JSON.parse(localStorage.getItem('visit_bookings') || '[]');
    return localBookings.filter((booking: any) => booking.property_owner_id === userId);
  }
  
  const data = await response.json();
  console.log('📊 Received bookings data:', data);
  return data.bookings || data.results || data || [];
};

// Update Booking Status
export const updateBookingStatus = async (bookingId: string, status: string): Promise<any> => {
  const token = localStorage.getItem('access_token');
  const response = await fetch(`${API_BASE}/visit-bookings/${bookingId}/status`, {
    method: 'PATCH',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${token}`
    },
    body: JSON.stringify({ status })
  });
  if (!response.ok) throw new Error('Failed to update booking status');
  return response.json();
};

// Get All Bookings (Admin)
export const getAllBookings = async (): Promise<VisitBooking[]> => {
  const token = localStorage.getItem('access_token');
  console.log('🔍 Fetching all bookings for admin');
  
  const response = await fetch(`${API_BASE}/visit-bookings/`, {
    headers: {
      'Authorization': `Bearer ${token}`,
      'Content-Type': 'application/json'
    }
  });
  
  console.log('📡 All bookings response:', response.status);
  
  if (!response.ok) {
    console.error('❌ Failed to fetch all bookings:', response.status);
    throw new Error('Failed to fetch all bookings');
  }
  
  const data = await response.json();
  console.log('📊 All bookings data:', data);
  return data.bookings || data.results || data || [];
};