const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

// Utility function to convert file to base64
export const fileToBase64 = (file: File): Promise<string> => {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.readAsDataURL(file);
    reader.onload = () => {
      const result = reader.result as string;
      // Remove the data:mime;base64, prefix
      const base64 = result.split(',')[1];
      resolve(base64);
    };
    reader.onerror = error => reject(error);
  });
};

export interface VisitScheduleRequest {
  user_id?: string;
  property_id: number;
  full_name: string;
  email: string;
  mobile: string;
  preferred_contact: 'Call' | 'Email' | 'WhatsApp';
  buyer_type: 'Individual Buyer' | 'Investor' | 'Builder' | 'Agent';
  preferred_date: string;
  preferred_time: string;
  alternate_date?: string;
  alternate_time?: string;
  number_of_visitors: number;
  visit_mode: 'In-person Visit' | 'Virtual Tour' | 'Video Call';
  pickup_assistance: boolean;
  visit_reason: string;
  comments?: string;
  id_document?: string;
  video_call_enabled?: boolean;
}

export interface VisitSchedule {
  id: number;
  user_id: string;
  property_id: number;
  status: 'pending' | 'admin_approved' | 'seller_approved' | 'admin_rejected' | 'seller_rejected' | 'approved' | 'rejected' | 'scheduled' | 'confirmed' | 'completed' | 'cancelled' | 'rescheduled';
  created_at: string;
  updated_at: string;
  full_name: string;
  email: string;
  mobile: string;
  preferred_contact: string;
  buyer_type: string;
  preferred_date: string;
  preferred_time: string;
  alternate_date?: string;
  alternate_time?: string;
  number_of_visitors: number;
  visit_mode: string;
  pickup_assistance: boolean;
  visit_reason: string;
  comments?: string;
  id_document?: string;
  video_call_room_id?: string;
  video_call_enabled: boolean;
  video_call_started_at?: string;
  video_call_ended_at?: string;
  property_title: string;
  property_address: string;
  property_city: string;
}

export interface VisitSchedulesResponse {
  visits: VisitSchedule[];
  total_count: number;
  page: number;
  limit: number;
}

// Create a new visit schedule
export const createVisitSchedule = async (visitData: VisitScheduleRequest): Promise<VisitSchedule> => {
  const token = localStorage.getItem('access_token');
  
  const response = await fetch(`${API_BASE}/visit-schedules/`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      ...(token && { 'Authorization': `Bearer ${token}` })
    },
    body: JSON.stringify(visitData)
  });

  if (!response.ok) {
    const errorText = await response.text();
    let errorMessage = 'Failed to create visit schedule';
    try {
      const error = JSON.parse(errorText);
      errorMessage = JSON.stringify(error.detail || error.message || error);
    } catch {
      errorMessage = errorText || errorMessage;
    }
    console.error('Visit schedule API error:', errorText);
    throw new Error(errorMessage);
  }

  return response.json();
};

// Get user visit schedules
export const getUserVisitSchedules = async (
  userId: string,
  page: number = 1,
  limit: number = 10,
  status?: string
): Promise<VisitSchedulesResponse> => {
  const token = localStorage.getItem('access_token');
  
  const params = new URLSearchParams({
    user_id: userId,
    page: page.toString(),
    limit: limit.toString(),
    ...(status && { status })
  });

  const response = await fetch(`${API_BASE}/visit-schedules/?${params}`, {
    headers: {
      ...(token && { 'Authorization': `Bearer ${token}` })
    }
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Failed to fetch visit schedules');
  }

  return response.json();
};

// Update visit schedule status (for admin/seller approval)
export const updateVisitScheduleStatus = async (
  visitId: number,
  status: string,
  comments?: string
): Promise<VisitSchedule> => {
  const token = localStorage.getItem('access_token');
  
  const response = await fetch(`${API_BASE}/visit-schedules/${visitId}/status`, {
    method: 'PATCH',
    headers: {
      'Content-Type': 'application/json',
      ...(token && { 'Authorization': `Bearer ${token}` })
    },
    body: JSON.stringify({ status, comments })
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Failed to update visit schedule');
  }

  return response.json();
};

// Get seller visits by status
export const getSellerVisits = async (
  sellerId: string,
  page: number = 1,
  limit: number = 10,
  status?: string
): Promise<VisitSchedulesResponse> => {
  const token = localStorage.getItem('access_token');
  
  const params = new URLSearchParams({
    page: page.toString(),
    limit: limit.toString(),
    ...(status && { status })
  });

  const response = await fetch(`${API_BASE}/visit-schedules/seller/${sellerId}/visits?${params}`, {
    headers: {
      ...(token && { 'Authorization': `Bearer ${token}` })
    }
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Failed to fetch seller visits');
  }

  return response.json();
};

// Get seller pending visits
export const getSellerPendingVisits = async (
  sellerId: string,
  page: number = 1,
  limit: number = 10
): Promise<VisitSchedulesResponse> => {
  return getSellerVisits(sellerId, page, limit, 'admin_approved');
};

// Seller approve visit
export const sellerApproveVisit = async (
  visitId: number,
  sellerId: string
): Promise<VisitSchedule> => {
  const token = localStorage.getItem('access_token');
  
  const response = await fetch(`${API_BASE}/visit-schedules/${visitId}/seller-approve?seller_id=${sellerId}`, {
    method: 'POST',
    headers: {
      ...(token && { 'Authorization': `Bearer ${token}` })
    }
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Failed to approve visit');
  }

  return response.json();
};

// Seller reject visit
export const sellerRejectVisit = async (
  visitId: number,
  sellerId: string
): Promise<VisitSchedule> => {
  const token = localStorage.getItem('access_token');
  
  const response = await fetch(`${API_BASE}/visit-schedules/${visitId}/seller-reject?seller_id=${sellerId}`, {
    method: 'POST',
    headers: {
      ...(token && { 'Authorization': `Bearer ${token}` })
    }
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Failed to reject visit');
  }

  return response.json();
};

// Get seller approved visits
export const getSellerApprovedVisits = async (
  sellerId: string,
  page: number = 1,
  limit: number = 10
): Promise<VisitSchedulesResponse> => {
  return getSellerVisits(sellerId, page, limit, 'seller_approved');
};

// Get seller rejected visits
export const getSellerRejectedVisits = async (
  sellerId: string,
  page: number = 1,
  limit: number = 10
): Promise<VisitSchedulesResponse> => {
  return getSellerVisits(sellerId, page, limit, 'seller_rejected');
};

// Check if user has pending visit for a property
export const checkPendingVisit = async (
  userId: string,
  propertyId: number
): Promise<{ hasPending: boolean; visit?: VisitSchedule }> => {
  const token = localStorage.getItem('access_token');
  
  const response = await fetch(`${API_BASE}/visit-schedules/check-pending?user_id=${userId}&property_id=${propertyId}`, {
    headers: {
      ...(token && { 'Authorization': `Bearer ${token}` })
    }
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Failed to check pending visits');
  }

  return response.json();
};

// Video call utility functions
export const getVideoCallUrl = (roomId: string): string => {
  return `https://meet.jit.si/${roomId}`;
};

export const isVideoCallAvailable = (visit: VisitSchedule): boolean => {
  return visit.video_call_enabled && 
         visit.video_call_room_id !== null && 
         visit.status === 'seller_approved';
};

export const canJoinVideoCall = (visit: VisitSchedule): boolean => {
  if (!isVideoCallAvailable(visit)) return false;
  
  const now = new Date();
  const visitDate = new Date(visit.preferred_date);
  const timeDiff = visitDate.getTime() - now.getTime();
  const hoursDiff = timeDiff / (1000 * 3600);
  
  // Allow joining 30 minutes before and 2 hours after scheduled time
  return hoursDiff >= -2 && hoursDiff <= 0.5;
};