const API_BASE_URL = 'http://localhost:8000';

// Zoning Data Types
export interface ZoningData {
  id: number;
  property_id: number;
  zone_classification: string;
  permitted_uses: string[];
  max_far: number;
  max_building_height: number;
  min_plot_size: number;
  max_plot_coverage: number;
  setback_requirements: Record<string, any>;
  parking_requirements: Record<string, any>;
  environmental_restrictions: Record<string, any>;
  infrastructure_availability: Record<string, any>;
  future_development_plans: Record<string, any>;
  development_restrictions: string;
  data_source: string;
  last_updated: string;
  created_at: string;
  updated_at: string;
}

export interface ZoningDataListResponse {
  zoning_data: ZoningData[];
  total_count: number;
  page: number;
  limit: number;
}

// Zoning Regulations Types
export interface ZoningRegulation {
  id: number;
  city: string;
  zone_type: string;
  regulation_type: string;
  regulation_name: string;
  description: string;
  effective_date: string;
  expiry_date: string | null;
  regulation_data: Record<string, any>;
  is_active: boolean;
  created_at: string;
  updated_at: string;
}

export interface ZoningRegulationListResponse {
  regulations: ZoningRegulation[];
  total_count: number;
  page: number;
  limit: number;
}

// Zoning Zones Types
export interface ZoningZone {
  id: number;
  city: string;
  zone_code: string;
  zone_name: string;
  zone_description: string | null;
  boundary_coordinates: any;
  area_sqkm: number | null;
  total_properties: number;
  created_at: string;
  updated_at: string;
}

export interface ZoningZoneListResponse {
  zones: ZoningZone[];
  total_count: number;
  page: number;
  limit: number;
}

// Development Potential Types
export interface DevelopmentPotentialRequest {
  property_id: number;
  proposed_area?: number;
}

export interface DevelopmentPotentialResponse {
  property_id: number;
  current_zoning: string;
  max_allowable_far: number;
  max_buildable_area: number;
  potential_floors: number;
  permitted_uses: string[];
  development_efficiency: number;
  estimated_construction_cost: number;
  estimated_property_value: number;
  risk_factors: string[];
  recommendations: string[];
}

// Compliance Check Types
export interface ComplianceCheckRequest {
  property_id: number;
  proposed_development: Record<string, any>;
  building_type: string;
  built_up_area: number;
  num_floors: number;
  building_height?: number;
}

export interface ComplianceCheckResponse {
  property_id: number;
  is_compliant: boolean;
  compliance_score: number;
  violations: Array<{
    type: string;
    description: string;
    severity: string;
  }>;
  warnings: string[];
  recommendations: string[];
  required_approvals: string[];
  estimated_timeline: string;
}

// Zoning Insights Types
export interface ZoningInsightsResponse {
  property_id: number;
  zoning_data: ZoningData | null;
  nearby_zones: ZoningZone[];
  applicable_regulations: ZoningRegulation[];
  development_potential: DevelopmentPotentialResponse | null;
  compliance_status: ComplianceCheckResponse | null;
  last_updated: string;
}

// API Functions

// Zoning Data
export const getZoningDataByPropertyId = async (propertyId: number): Promise<ZoningData> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/zoning/data/property/${propertyId}`, {
      headers: { 'Content-Type': 'application/json' }
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch zoning data: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error fetching zoning data:', error);
    throw error;
  }
};

export const getAllZoningData = async (params: {
  page?: number;
  limit?: number;
  city?: string;
  zone_classification?: string;
} = {}): Promise<ZoningDataListResponse> => {
  try {
    const queryParams = new URLSearchParams();

    Object.entries(params).forEach(([key, value]) => {
      if (value !== undefined && value !== null && value !== '') {
        queryParams.append(key, value.toString());
      }
    });

    const url = `${API_BASE_URL}/api/zoning/data${queryParams.toString() ? '?' + queryParams.toString() : ''}`;
    const response = await fetch(url, {
      headers: { 'Content-Type': 'application/json' }
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch zoning data: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error fetching zoning data:', error);
    throw error;
  }
};

export const createZoningData = async (zoningData: any): Promise<ZoningData> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/zoning/data`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      },
      body: JSON.stringify(zoningData)
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to create zoning data: ${response.status} - ${errorText}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error creating zoning data:', error);
    throw error;
  }
};

export const updateZoningData = async (propertyId: number, zoningData: any): Promise<ZoningData> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/zoning/data/property/${propertyId}`, {
      method: 'PUT',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      },
      body: JSON.stringify(zoningData)
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to update zoning data: ${response.status} - ${errorText}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error updating zoning data:', error);
    throw error;
  }
};

// Zoning Regulations
export const getZoningRegulations = async (params: {
  city: string;
  zone_type?: string;
  page?: number;
  limit?: number;
}): Promise<ZoningRegulationListResponse> => {
  try {
    const queryParams = new URLSearchParams();

    Object.entries(params).forEach(([key, value]) => {
      if (value !== undefined && value !== null && value !== '') {
        queryParams.append(key, value.toString());
      }
    });

    const url = `${API_BASE_URL}/api/zoning/regulations?${queryParams.toString()}`;
    const response = await fetch(url, {
      headers: { 'Content-Type': 'application/json' }
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch zoning regulations: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error fetching zoning regulations:', error);
    throw error;
  }
};

export const createZoningRegulation = async (regulationData: any): Promise<ZoningRegulation> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/zoning/regulations`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      },
      body: JSON.stringify(regulationData)
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to create zoning regulation: ${response.status} - ${errorText}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error creating zoning regulation:', error);
    throw error;
  }
};

// Zoning Zones
export const getZoningZones = async (params: {
  city: string;
  page?: number;
  limit?: number;
}): Promise<ZoningZoneListResponse> => {
  try {
    const queryParams = new URLSearchParams();

    Object.entries(params).forEach(([key, value]) => {
      if (value !== undefined && value !== null && value !== '') {
        queryParams.append(key, value.toString());
      }
    });

    const url = `${API_BASE_URL}/api/zoning/zones?${queryParams.toString()}`;
    const response = await fetch(url, {
      headers: { 'Content-Type': 'application/json' }
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch zoning zones: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error fetching zoning zones:', error);
    throw error;
  }
};

export const createZoningZone = async (zoneData: any): Promise<ZoningZone> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/zoning/zones`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      },
      body: JSON.stringify(zoneData)
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to create zoning zone: ${response.status} - ${errorText}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error creating zoning zone:', error);
    throw error;
  }
};

// Business Logic APIs
export const calculateDevelopmentPotential = async (request: DevelopmentPotentialRequest): Promise<DevelopmentPotentialResponse> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/zoning/development-potential`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      },
      body: JSON.stringify(request)
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to calculate development potential: ${response.status} - ${errorText}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error calculating development potential:', error);
    throw error;
  }
};

export const checkCompliance = async (request: ComplianceCheckRequest): Promise<ComplianceCheckResponse> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/zoning/compliance-check`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      },
      body: JSON.stringify(request)
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to check compliance: ${response.status} - ${errorText}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error checking compliance:', error);
    throw error;
  }
};

export const getZoningInsights = async (propertyId: number): Promise<ZoningInsightsResponse> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/zoning/insights/${propertyId}`, {
      headers: { 'Content-Type': 'application/json' }
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch zoning insights: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error fetching zoning insights:', error);
    throw error;
  }
};

// Main API object
export const zoningAPI = {
  // Zoning Data
  getZoningDataByPropertyId,
  getAllZoningData,
  createZoningData,
  updateZoningData,

  // Zoning Regulations
  getZoningRegulations,
  createZoningRegulation,

  // Zoning Zones
  getZoningZones,
  createZoningZone,

  // Business Logic
  calculateDevelopmentPotential,
  checkCompliance,
  getZoningInsights
};