// src/components/AIDescriptionGenerator.tsx
import React, { useState } from 'react';
import { toast } from 'sonner';
import { aiDescriptionApi, PropertyData } from '../utils/aiDescriptionApi';

interface AIDescriptionGeneratorProps {
  propertyData: any;
  onDescriptionSelect: (description: string) => void;
  isOpen: boolean;
  onClose: () => void;
}

interface DescriptionOption {
  id: number;
  tone: string;
  description: string;
}

export const AIDescriptionGenerator: React.FC<AIDescriptionGeneratorProps> = ({
  propertyData,
  onDescriptionSelect,
  isOpen,
  onClose
}) => {
  const [descriptions, setDescriptions] = useState<DescriptionOption[]>([]);
  const [isGenerating, setIsGenerating] = useState(false);
  const [selectedDescription, setSelectedDescription] = useState<string>('');

  const generateDescriptions = async () => {
    setIsGenerating(true);
    
    try {
      const cleanData = {
        title: propertyData.title || 'Property',
        propertyType: propertyData.propertyType || '',
        listingType: propertyData.listingType || '',
        bedrooms: propertyData.bedrooms ? String(propertyData.bedrooms) : '',
        bathrooms: propertyData.bathrooms ? String(propertyData.bathrooms) : '',
        builtUpArea: propertyData.builtUpArea ? String(propertyData.builtUpArea) : '',
        city: propertyData.city || '',
        locality: propertyData.locality || '',
        facing: propertyData.facing || '',
        furnishing: propertyData.furnishing || '',
        propertyAge: propertyData.propertyAge || '',
        amenities: propertyData.amenities || [],
        features: propertyData.features || [],
        highlights: propertyData.highlights || [],
        salePrice: propertyData.salePrice ? String(propertyData.salePrice) : '',
        monthlyRent: propertyData.monthlyRent ? String(propertyData.monthlyRent) : '',
        phone: propertyData.contactPhone || '',
        email: propertyData.contactEmail || ''
      };
      
      const result = await aiDescriptionApi.generateMultipleDescriptions(cleanData);
      
      if (result.success && result.descriptions) {
        setDescriptions(result.descriptions);
      } else {
        toast.error(result.error || 'Failed to generate descriptions');
      }
      
    } catch (error) {
      console.error('Error generating descriptions:', error);
      toast.error('Failed to generate descriptions. Please try again.');
    } finally {
      setIsGenerating(false);
    }
  };

  const handleSelectDescription = () => {
    if (selectedDescription && selectedDescription.trim()) {
      console.log('Applying selected description:', selectedDescription);
      onDescriptionSelect(selectedDescription);
      onClose();
      toast.success('Description applied successfully!');
      // Reset state
      setSelectedDescription('');
      setDescriptions([]);
    } else {
      toast.error('Please select a description first');
    }
  };

  React.useEffect(() => {
    if (isOpen) {
      if (descriptions.length === 0) {
        generateDescriptions();
      }
      // Reset selection when modal opens
      setSelectedDescription('');
    }
  }, [isOpen]);

  React.useEffect(() => {
    // Reset selection when descriptions change
    setSelectedDescription('');
  }, [descriptions]);

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-xl shadow-2xl max-w-3xl w-full max-h-[70vh] flex flex-col">
        {/* Header */}
        <div className="bg-gradient-to-r from-[#0056D2] to-blue-600 text-white p-4 flex-shrink-0">
          <div className="flex items-center justify-between">
            <div>
              <h2 className="text-xl font-bold">AI Property Description Generator</h2>
              <p className="text-blue-100 mt-1 text-sm">Choose from AI-generated descriptions tailored for your property</p>
            </div>
            <button
              onClick={onClose}
              className="text-white hover:text-purple-200 transition-colors"
            >
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
              </svg>
            </button>
          </div>
        </div>

        {/* Content */}
        <div className="flex-1 overflow-y-auto p-6 min-h-0" style={{maxHeight: 'calc(70vh - 140px)'}}>
          {isGenerating ? (
            <div className="flex flex-col items-center justify-center py-12">
              <div className="relative">
                <div className="w-16 h-16 border-4 border-purple-200 border-t-purple-600 rounded-full animate-spin"></div>
                <div className="absolute inset-0 flex items-center justify-center">
                  <svg className="w-6 h-6 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                  </svg>
                </div>
              </div>
              <h3 className="text-xl font-semibold text-gray-800 mt-4">Generating Descriptions</h3>
              <p className="text-gray-600 mt-2 text-center max-w-md">
                Our AI is crafting professional property descriptions based on your property details...
              </p>
            </div>
          ) : descriptions.length > 0 ? (
            <div className="space-y-4">
              <div className="text-center mb-4">
                <h3 className="text-lg font-semibold text-gray-800">Select Your Preferred Description</h3>
                <p className="text-gray-600 mt-1 text-sm">Choose the description that best represents your property</p>
              </div>

              <div className="space-y-4">
                {descriptions.map((option) => (
                  <div
                    key={option.id}
                    className={`border-2 rounded-lg cursor-pointer transition-all duration-200 ${
                      selectedDescription === option.description
                        ? 'border-[#0056D2] bg-blue-50 shadow-md'
                        : 'border-gray-200 hover:border-[#0056D2] hover:shadow-sm'
                    }`}
                    onClick={() => {
                      console.log('Selecting description:', option.id, option.tone);
                      setSelectedDescription(option.description);
                    }}
                  >
                    <div className="p-4">
                      <div className="flex items-center justify-between mb-3">
                        <div className="flex items-center">
                          <div className={`w-5 h-5 rounded-full border-2 mr-3 flex items-center justify-center ${
                            selectedDescription === option.description
                              ? 'border-[#0056D2] bg-[#0056D2]'
                              : 'border-gray-300 bg-white'
                          }`}>
                            {selectedDescription === option.description && (
                              <svg className="w-3 h-3 text-white" fill="currentColor" viewBox="0 0 20 20">
                                <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                              </svg>
                            )}
                          </div>
                          <span className="font-medium text-gray-800 capitalize">
                            {option.tone} Style
                          </span>
                        </div>
                        <span className="text-xs text-gray-500 bg-gray-100 px-2 py-1 rounded-full">
                          {option.description.length} characters
                        </span>
                      </div>
                      <div className="max-h-32 overflow-y-auto text-gray-700 leading-relaxed text-sm bg-gray-50 p-3 rounded border scrollbar-thin scrollbar-thumb-gray-300 scrollbar-track-gray-100">
                        {option.description}
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          ) : (
            <div className="text-center py-12">
              <svg className="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
              </svg>
              <h3 className="text-lg font-semibold text-gray-800">No Descriptions Generated</h3>
              <p className="text-gray-600 mt-2">Unable to generate descriptions. Please try again.</p>
              <button
                onClick={generateDescriptions}
                className="mt-4 px-4 py-2 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors"
              >
                Try Again
              </button>
            </div>
          )}
        </div>

        {/* Footer */}
        <div className="bg-gray-50 px-4 py-3 flex items-center justify-between border-t flex-shrink-0">
          <div className="flex items-center text-sm text-gray-600">
            <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
            {selectedDescription ? `Selected: ${selectedDescription.substring(0, 50)}...` : 'Select a description to continue'}
          </div>
          <div className="flex space-x-3">
            <button
              onClick={onClose}
              className="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
            >
              Cancel
            </button>
            <button
              onClick={handleSelectDescription}
              disabled={!selectedDescription}
              className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-semibold shadow-md hover:shadow-lg disabled:opacity-50 disabled:cursor-not-allowed"
            >
              Use This Description
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};