import { useEffect, useState, useRef } from 'react';
import { arAssetsApi, ARAsset } from '../api/arAssets';
import { Camera, Loader2, RotateCcw, Move3D } from 'lucide-react';
import * as THREE from 'three';
import { GLTFLoader } from 'three/examples/jsm/loaders/GLTFLoader.js';
import { OrbitControls } from 'three/examples/jsm/controls/OrbitControls.js';

export default function ARViewer() {
  const [assets, setAssets] = useState<ARAsset[]>([]);
  const [selectedAsset, setSelectedAsset] = useState<ARAsset | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [isARSupported, setIsARSupported] = useState(false);
  const [isInAR, setIsInAR] = useState(false);
  const [modelLoading, setModelLoading] = useState(false);
  
  const containerRef = useRef<HTMLDivElement>(null);
  const sceneRef = useRef<THREE.Scene | null>(null);
  const rendererRef = useRef<THREE.WebGLRenderer | null>(null);
  const cameraRef = useRef<THREE.PerspectiveCamera | null>(null);
  const controlsRef = useRef<OrbitControls | null>(null);
  const currentModelRef = useRef<THREE.Group | null>(null);
  const animationIdRef = useRef<number | null>(null);

  // Initialize Three.js scene
  const initScene = () => {
    if (!containerRef.current) return;

    // Scene
    const scene = new THREE.Scene();
    scene.background = new THREE.Color(0xf9fafb);
    sceneRef.current = scene;

    // Camera
    const camera = new THREE.PerspectiveCamera(
      75,
      containerRef.current.clientWidth / containerRef.current.clientHeight,
      0.1,
      1000
    );
    camera.position.set(2, 2, 2);
    cameraRef.current = camera;

    // Renderer
    const renderer = new THREE.WebGLRenderer({ antialias: true, alpha: true });
    renderer.setSize(containerRef.current.clientWidth, containerRef.current.clientHeight);
    renderer.shadowMap.enabled = true;
    renderer.shadowMap.type = THREE.PCFSoftShadowMap;
    renderer.xr.enabled = true;
    containerRef.current.appendChild(renderer.domElement);
    rendererRef.current = renderer;

    // Controls
    const controls = new OrbitControls(camera, renderer.domElement);
    controls.enableDamping = true;
    controls.dampingFactor = 0.05;
    controlsRef.current = controls;

    // Lighting
    const ambientLight = new THREE.AmbientLight(0xffffff, 0.6);
    scene.add(ambientLight);

    const directionalLight = new THREE.DirectionalLight(0xffffff, 0.8);
    directionalLight.position.set(5, 5, 5);
    directionalLight.castShadow = true;
    scene.add(directionalLight);

    // Ground plane for AR
    const groundGeometry = new THREE.PlaneGeometry(10, 10);
    const groundMaterial = new THREE.MeshLambertMaterial({ 
      color: 0xcccccc, 
      transparent: true, 
      opacity: 0.3 
    });
    const ground = new THREE.Mesh(groundGeometry, groundMaterial);
    ground.rotation.x = -Math.PI / 2;
    ground.receiveShadow = true;
    scene.add(ground);

    // Animation loop
    const animate = () => {
      animationIdRef.current = requestAnimationFrame(animate);
      controls.update();
      renderer.render(scene, camera);
    };
    animate();

    // Handle resize
    const handleResize = () => {
      if (!containerRef.current || !camera || !renderer) return;
      camera.aspect = containerRef.current.clientWidth / containerRef.current.clientHeight;
      camera.updateProjectionMatrix();
      renderer.setSize(containerRef.current.clientWidth, containerRef.current.clientHeight);
    };
    window.addEventListener('resize', handleResize);

    return () => {
      window.removeEventListener('resize', handleResize);
    };
  };

  // Load 3D model
  const loadModel = async (asset: ARAsset) => {
    if (!sceneRef.current) return;
    
    setModelLoading(true);
    
    // Remove previous model
    if (currentModelRef.current) {
      sceneRef.current.remove(currentModelRef.current);
    }

    try {
      const loader = new GLTFLoader();
      
      // Configure loader to handle textures properly
      const manager = new THREE.LoadingManager();
      manager.setURLModifier((url) => {
        // Convert blob URLs to proxy URLs
        if (url.startsWith('blob:')) {
          return url;
        }
        // Handle relative texture paths
        if (!url.startsWith('http') && !url.startsWith('/')) {
          return `/minio/nal-ar-assets/furniture/${asset.sub_category}/${url}`;
        }
        return url.replace('http://localhost:9000', '/minio');
      });
      
      loader.manager = manager;
      
      // Use Vite proxy to bypass CORS
      const proxyUrl = asset.file_url.replace('http://localhost:9000', '/minio');
      
      const gltf = await new Promise<any>((resolve, reject) => {
        loader.load(
          proxyUrl,
          (gltf) => {
            // Process materials to handle missing textures
            gltf.scene.traverse((child: any) => {
              if (child.isMesh && child.material) {
                // Create fallback material if textures fail
                if (child.material.map && !child.material.map.image) {
                  child.material.map = null;
                  child.material.color.setHex(0x888888);
                }
                child.material.needsUpdate = true;
              }
            });
            resolve(gltf);
          },
          (progress) => {
            console.log('Loading progress:', (progress.loaded / progress.total * 100) + '%');
          },
          (error) => {
            console.error('GLTFLoader error with proxy:', error);
            reject(error);
          }
        );
      });

      const model = gltf.scene;
      model.scale.set(
        asset.default_scale[0],
        asset.default_scale[1], 
        asset.default_scale[2]
      );
      
      // Enable shadows
      model.traverse((child: any) => {
        if (child.isMesh) {
          child.castShadow = true;
          child.receiveShadow = true;
        }
      });

      sceneRef.current.add(model);
      currentModelRef.current = model;
      setModelLoading(false);
      setError(null);
    } catch (err) {
      console.error('Failed to load model:', err);
      setError(`Failed to load 3D model: ${err}`);
      setModelLoading(false);
    }
  };

  // Check WebXR support
  const checkARSupport = async () => {
    if ('xr' in navigator) {
      try {
        const supported = await (navigator as any).xr.isSessionSupported('immersive-ar');
        setIsARSupported(supported);
      } catch {
        setIsARSupported(false);
      }
    }
  };

  // Start AR session
  const startAR = async () => {
    if (!rendererRef.current || !isARSupported) return;

    try {
      const session = await (navigator as any).xr.requestSession('immersive-ar', {
        requiredFeatures: ['hit-test'],
        optionalFeatures: ['dom-overlay'],
        domOverlay: { root: document.body }
      });
      
      await rendererRef.current.xr.setSession(session);
      setIsInAR(true);
    } catch (err) {
      console.error('Failed to start AR:', err);
      // Fallback for non-AR devices
      alert('AR not supported on this device. Viewing in 3D mode.');
    }
  };

  // Initialize
  useEffect(() => {
    checkARSupport();
    
    // Fetch assets
    arAssetsApi
      .getAll('furniture')
      .then((data) => {
        setAssets(data);
        if (data.length > 0) {
          setSelectedAsset(data[0]);
        }
        setLoading(false);
      })
      .catch((err) => {
        setError(err.message);
        setLoading(false);
      });
  }, []);

  // Initialize scene when container is ready
  useEffect(() => {
    if (!loading && containerRef.current) {
      const cleanup = initScene();
      return cleanup;
    }
  }, [loading]);

  // Load model when asset changes
  useEffect(() => {
    if (selectedAsset && sceneRef.current) {
      loadModel(selectedAsset);
    }
  }, [selectedAsset]);

  // Cleanup
  useEffect(() => {
    return () => {
      if (animationIdRef.current) {
        cancelAnimationFrame(animationIdRef.current);
      }
      if (rendererRef.current) {
        rendererRef.current.dispose();
      }
    };
  }, []);

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <Loader2 className="w-8 h-8 animate-spin text-blue-600" />
      </div>
    );
  }

  if (error) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="text-center">
          <p className="text-red-600 mb-2">Error loading AR assets</p>
          <p className="text-sm text-gray-600">{error}</p>
        </div>
      </div>
    );
  }

  return (
    <div className="flex h-screen bg-gray-50">
      {/* Asset List - Left Sidebar */}
      <div className="w-96 bg-white shadow-lg overflow-y-auto">
        <div className="p-6 border-b">
          <h2 className="text-2xl font-bold text-gray-800">AR Furniture</h2>
          <p className="text-sm text-gray-600 mt-1">Select an item to view in AR</p>
          {isARSupported && (
            <div className="mt-2 px-2 py-1 bg-green-100 text-green-800 text-xs rounded">
              AR Ready
            </div>
          )}
        </div>

        <div className="p-4 space-y-3">
          {assets.map((asset) => (
            <div
              key={asset.id}
              onClick={() => setSelectedAsset(asset)}
              className={`p-4 rounded-lg cursor-pointer transition-all ${
                selectedAsset?.id === asset.id
                  ? 'bg-blue-50 border-2 border-blue-500'
                  : 'bg-gray-50 border-2 border-transparent hover:border-gray-300'
              }`}
            >
              <h3 className="font-semibold text-gray-800">{asset.name}</h3>
              <p className="text-xs text-gray-500 mt-1">
                {asset.real_width}m × {asset.real_height}m × {asset.real_depth}m
              </p>
              <p className="text-xs text-gray-400 mt-1 capitalize">{asset.placement_type}</p>
            </div>
          ))}
        </div>
      </div>

      {/* AR Viewer - Main Area */}
      <div className="flex-1 flex flex-col">
        {selectedAsset ? (
          <>
            {/* Header */}
            <div className="bg-white shadow-sm p-4 border-b">
              <h1 className="text-xl font-bold text-gray-800">{selectedAsset.name}</h1>
              <div className="flex gap-4 mt-2 text-sm text-gray-600">
                <span>Size: {selectedAsset.real_width}m × {selectedAsset.real_height}m × {selectedAsset.real_depth}m</span>
                <span>•</span>
                <span className="capitalize">Placement: {selectedAsset.placement_type}</span>
              </div>
            </div>

            {/* Three.js Viewer */}
            <div className="flex-1 relative">
              <div ref={containerRef} className="w-full h-full" />
              
              {modelLoading && (
                <div className="absolute inset-0 flex items-center justify-center bg-gray-50 bg-opacity-75">
                  <Loader2 className="w-8 h-8 animate-spin text-blue-600" />
                </div>
              )}

              {/* Controls */}
              <div className="absolute bottom-8 left-1/2 transform -translate-x-1/2 flex gap-4">
                {isARSupported && (
                  <button
                    onClick={startAR}
                    className="bg-blue-600 hover:bg-blue-700 text-white px-6 py-3 rounded-full shadow-lg flex items-center gap-2 font-semibold transition-all"
                  >
                    <Camera className="w-5 h-5" />
                    {isInAR ? 'Exit AR' : 'View in AR'}
                  </button>
                )}
                
                <button
                  onClick={() => {
                    if (controlsRef.current && cameraRef.current) {
                      controlsRef.current.reset();
                      cameraRef.current.position.set(2, 2, 2);
                    }
                  }}
                  className="bg-gray-600 hover:bg-gray-700 text-white px-4 py-3 rounded-full shadow-lg flex items-center gap-2 transition-all"
                >
                  <RotateCcw className="w-5 h-5" />
                </button>
              </div>

              {/* Instructions */}
              <div className="absolute top-4 right-4 bg-white rounded-lg shadow-md p-4 max-w-xs">
                <h4 className="font-semibold text-gray-800 mb-2">Controls:</h4>
                <ul className="text-sm text-gray-600 space-y-1">
                  <li>• Drag to rotate</li>
                  <li>• Scroll to zoom</li>
                  <li>• Right-drag to pan</li>
                  {isARSupported && <li>• Click AR for immersive view</li>}
                </ul>
              </div>
            </div>
          </>
        ) : (
          <div className="flex-1 flex items-center justify-center">
            <p className="text-gray-500">Select an item from the list</p>
          </div>
        )}
      </div>
    </div>
  );
}
