import React, { useState } from 'react';
import { MapPin, Search, Navigation } from 'lucide-react';

interface AddressAutocompleteProps {
  value: string;
  onChange: (address: string) => void;
  onLocationSelect?: (location: { lat: number; lng: number; address: string }) => void;
  placeholder?: string;
}

const mockAddresses = [
  { address: 'Koramangala, Bangalore, Karnataka', lat: 12.9352, lng: 77.6245 },
  { address: 'Whitefield, Bangalore, Karnataka', lat: 12.9698, lng: 77.7500 },
  { address: 'Indiranagar, Bangalore, Karnataka', lat: 12.9784, lng: 77.6408 },
  { address: 'HSR Layout, Bangalore, Karnataka', lat: 12.9116, lng: 77.6370 },
  { address: 'Electronic City, Bangalore, Karnataka', lat: 12.8456, lng: 77.6603 }
];

export const AddressAutocomplete: React.FC<AddressAutocompleteProps> = ({
  value,
  onChange,
  onLocationSelect,
  placeholder = "Enter property address..."
}) => {
  const [suggestions, setSuggestions] = useState<typeof mockAddresses>([]);
  const [showSuggestions, setShowSuggestions] = useState(false);
  const [isGettingLocation, setIsGettingLocation] = useState(false);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const inputValue = e.target.value;
    onChange(inputValue);

    if (inputValue.length > 2) {
      const filtered = mockAddresses.filter(addr =>
        addr.address.toLowerCase().includes(inputValue.toLowerCase())
      );
      setSuggestions(filtered);
      setShowSuggestions(true);
    } else {
      setShowSuggestions(false);
    }
  };

  const handleSuggestionClick = (suggestion: typeof mockAddresses[0]) => {
    onChange(suggestion.address);
    setShowSuggestions(false);
    if (onLocationSelect) {
      onLocationSelect({
        lat: suggestion.lat,
        lng: suggestion.lng,
        address: suggestion.address
      });
    }
  };

  const getCurrentLocation = () => {
    if (!navigator.geolocation) {
      alert('Geolocation is not supported by this browser.');
      return;
    }

    setIsGettingLocation(true);
    
    navigator.geolocation.getCurrentPosition(
      async (position) => {
        const { latitude, longitude } = position.coords;
        
        try {
          // Mock reverse geocoding with more realistic address format
          const mockAddress = `Current Location (${latitude.toFixed(6)}, ${longitude.toFixed(6)})`;
          
          onChange(mockAddress);
          
          if (onLocationSelect) {
            onLocationSelect({
              lat: latitude,
              lng: longitude,
              address: mockAddress
            });
          }
          
          // Show success message
          const successMsg = document.createElement('div');
          successMsg.className = 'fixed top-4 right-4 bg-green-500 text-white px-4 py-2 rounded-lg shadow-lg z-50';
          successMsg.textContent = '✓ Location detected successfully!';
          document.body.appendChild(successMsg);
          setTimeout(() => {
            document.body.removeChild(successMsg);
          }, 3000);
        } catch (error) {
          console.error('Error getting address:', error);
          alert('Could not get address for your location');
        } finally {
          setIsGettingLocation(false);
        }
      },
      (error) => {
        setIsGettingLocation(false);
        switch (error.code) {
          case error.PERMISSION_DENIED:
            alert('Location access denied. Please enable location permissions.');
            break;
          case error.POSITION_UNAVAILABLE:
            alert('Location information is unavailable.');
            break;
          case error.TIMEOUT:
            alert('Location request timed out.');
            break;
          default:
            alert('An unknown error occurred while getting location.');
            break;
        }
      },
      {
        enableHighAccuracy: true,
        timeout: 10000,
        maximumAge: 60000
      }
    );
  };

  return (
    <div className="relative">
      <div className="relative">
        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
        <input
          type="text"
          value={value}
          onChange={handleInputChange}
          placeholder={placeholder}
          className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
          onFocus={() => value.length > 2 && setShowSuggestions(true)}
          onBlur={() => setTimeout(() => setShowSuggestions(false), 200)}
        />

      </div>

      {showSuggestions && suggestions.length > 0 && (
        <div className="absolute z-10 w-full mt-1 bg-white border border-gray-300 rounded-lg shadow-lg max-h-60 overflow-y-auto">
          {suggestions.map((suggestion, index) => (
            <button
              key={index}
              type="button"
              className="w-full px-4 py-3 text-left hover:bg-gray-50 flex items-center border-b border-gray-100 last:border-b-0"
              onClick={() => handleSuggestionClick(suggestion)}
            >
              <MapPin className="w-4 h-4 mr-3 text-gray-400" />
              <span className="text-gray-900">{suggestion.address}</span>
            </button>
          ))}
        </div>
      )}
      
      {/* Use Current Location Button */}
      <button
        type="button"
        onClick={getCurrentLocation}
        disabled={isGettingLocation}
        className="mt-2 flex items-center space-x-2 text-sm text-[#2B256D] hover:text-[#1f1a5a] transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
      >
        {isGettingLocation ? (
          <>
            <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-[#2B256D]"></div>
            <span>Getting your location...</span>
          </>
        ) : (
          <>
            <Navigation className="w-4 h-4" />
            <span>Use my current location</span>
          </>
        )}
      </button>
    </div>
  );
};