import React, { useState, useEffect } from 'react';
import axios from 'axios';

interface Lead {
  id: string;
  name: string;
  email: string;
  phone: string;
  message?: string;
  budget?: string;
  preferred_contact: string;
  score?: number;
  category?: string;
  property_title?: string;
  status: string;
  created_at: string;
}

export const AgentLeads: React.FC = () => {
  const [leads, setLeads] = useState<Lead[]>([]);
  const [loading, setLoading] = useState(true);
  const [sortBy, setSortBy] = useState('score');
  const [filterScore, setFilterScore] = useState('all');

  useEffect(() => {
    fetchLeads();
  }, []);

  const fetchLeads = async () => {
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      // The instruction provided a specific change for this line.
      // Note: The provided instruction `fetch(`${API_BASE_URL}/api/leads-management/leads/agent/${user.id}`, {user_id})`
      // has a syntax error with `{user_id}` as the second argument for `fetch`.
      // Assuming the intent was to pass `user.id` as a path parameter and the original `axios` call
      // was meant to be replaced with `axios.get` to the new endpoint.
      // If `fetch` was strictly intended, the `response.data.leads` would also need adjustment.
      // Sticking to `axios` for consistency with `response.data.leads` and `axios.patch` below,
      // and interpreting the path change.
      // Use fallback ID if user.id is missing to prevent 404/500 in dev
      const userId = user.id || user.user_id || '00000000-0000-0000-0000-000000000000';
      const response = await fetch(`/api/leads-management/leads/agent/${userId}`);
      if (!response.ok) throw new Error('Failed to fetch leads');
      const data = await response.json();

      // Map backend data to frontend Lead interface
      const mappedLeads: Lead[] = data.map((item: any) => ({
        id: item.id.toString(), // Assuming backend `id` maps to frontend `id`
        name: item.buyer_name || item.name,
        email: item.buyer_email || item.email,
        phone: item.buyer_phone || item.phone,
        score: item.score || 0,
        category: item.priority || 'Medium', // Map priority to category
        message: item.message || 'No details provided.', // Assuming 'explanation' maps to 'message'
        status: item.status || 'new', // Assuming a default status if not provided
        created_at: item.created_at || new Date().toISOString(),
        preferred_contact: item.preferred_contact || 'call', // Assuming a default
        budget: item.budget,
        property_title: item.property_title,
      }));
      setLeads(mappedLeads);
    } catch (error) {
      console.error('Error fetching leads:', error);
      // Hardcoded data for demo
      setLeads([
        {
          id: '1',
          name: 'John Smith',
          email: 'john@example.com',
          phone: '+91 9876543210',
          message: 'Interested in 3BHK apartment',
          budget: '80 Lakhs',
          preferred_contact: 'call',
          score: 85,
          category: 'High',
          property_title: '3BHK Luxury Apartment',
          status: 'new',
          created_at: new Date().toISOString()
        },
        {
          id: '2',
          name: 'Sarah Johnson',
          email: 'sarah@example.com',
          phone: '+91 9876543211',
          message: 'Looking for investment property',
          budget: '50 Lakhs',
          preferred_contact: 'whatsapp',
          score: 65,
          category: 'Medium',
          property_title: '2BHK Modern Flat',
          status: 'contacted',
          created_at: new Date().toISOString()
        }
      ]);
    } finally {
      setLoading(false);
    }
  };

  const updateLeadStage = async (leadId: string, newStatus: string) => {
    try {
      await axios.patch(`/api/leads/${leadId}/stage`, { status: newStatus });
      setLeads(leads.map(lead =>
        lead.id === leadId ? { ...lead, status: newStatus } : lead
      ));
    } catch (error) {
      console.error('Error updating lead stage:', error);
      // Update locally for demo
      setLeads(leads.map(lead =>
        lead.id === leadId ? { ...lead, status: newStatus } : lead
      ));
    }
  };

  const handleContact = (lead: Lead, method: string) => {
    switch (method) {
      case 'call':
        window.open(`tel:${lead.phone}`);
        // Move archived leads back to contacted
        if (lead.status === 'lost') {
          updateLeadStage(lead.id, 'contacted');
        }
        break;
      case 'sms':
        window.open(`sms:${lead.phone}`);
        break;
      case 'whatsapp':
        window.open(`https://wa.me/${lead.phone.replace(/[^0-9]/g, '')}`);
        // Move archived leads back to contacted
        if (lead.status === 'lost') {
          updateLeadStage(lead.id, 'contacted');
        }
        break;
      case 'email':
        window.open(`mailto:${lead.email}`);
        break;
    }
  };

  const getScoreColor = (score?: number) => {
    if (!score) return 'bg-gray-100 text-gray-800';
    if (score >= 80) return 'bg-red-100 text-red-800';
    if (score >= 50) return 'bg-yellow-100 text-yellow-800';
    return 'bg-gray-100 text-gray-800';
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'new': return 'bg-blue-100 text-blue-800';
      case 'contacted': return 'bg-yellow-100 text-yellow-800';
      case 'qualified': return 'bg-purple-100 text-purple-800';
      case 'negotiation': return 'bg-orange-100 text-orange-800';
      case 'converted': return 'bg-green-100 text-green-800';
      case 'lost': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const sortedLeads = leads
    .filter(lead => filterScore === 'all' ||
      (filterScore === 'high' && (lead.score || 0) >= 80) ||
      (filterScore === 'medium' && (lead.score || 0) >= 50 && (lead.score || 0) < 80) ||
      (filterScore === 'low' && (lead.score || 0) < 50))
    .sort((a, b) => {
      if (sortBy === 'score') return (b.score || 0) - (a.score || 0);
      if (sortBy === 'name') return a.name.localeCompare(b.name);
      return new Date(b.created_at).getTime() - new Date(a.created_at).getTime();
    });

  if (loading) {
    return <div className="p-6">Loading leads...</div>;
  }

  return (
    <div className="p-6">
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-2xl font-bold">Agent Leads</h1>
        <div className="flex gap-4">
          <select
            value={sortBy}
            onChange={(e) => setSortBy(e.target.value)}
            className="px-3 py-2 border rounded-lg"
          >
            <option value="score">Sort by Score</option>
            <option value="name">Sort by Name</option>
            <option value="date">Sort by Date</option>
          </select>

          <select
            value={filterScore}
            onChange={(e) => setFilterScore(e.target.value)}
            className="px-3 py-2 border rounded-lg"
          >
            <option value="all">All Scores</option>
            <option value="high">High Score (80+)</option>
            <option value="medium">Medium Score (50-79)</option>
            <option value="low">Low Score (Less than 50)</option>
          </select>
        </div>
      </div>

      <div className="grid gap-4">
        {sortedLeads.map(lead => (
          <div key={lead.id} className="bg-white border rounded-lg p-4 shadow-sm">
            <div className="flex justify-between items-start mb-3">
              <div>
                <h3 className="font-semibold text-lg">{lead.name}</h3>
                <p className="text-gray-600">{lead.property_title}</p>
                <p className="text-sm text-gray-500">{lead.email} • {lead.phone}</p>
              </div>

              <div className="flex items-center gap-2">
                <span className={`px-3 py-1 rounded-full text-sm font-medium ${getScoreColor(lead.score)}`}>
                  Score: {lead.score || 0}
                </span>
              </div>
            </div>

            {lead.message && (
              <p className="text-gray-700 mb-3">{lead.message}</p>
            )}

            {lead.budget && (
              <p className="text-sm text-gray-600 mb-3">Budget: {lead.budget}</p>
            )}

            <div className="flex justify-between items-center mb-3">
              <div className="flex gap-2">
                <button
                  onClick={() => handleContact(lead, 'call')}
                  className="p-2 bg-green-100 text-green-600 rounded-lg hover:bg-green-200"
                  title="Call"
                >
                  📞
                </button>
                <button
                  onClick={() => handleContact(lead, 'sms')}
                  className="p-2 bg-blue-100 text-blue-600 rounded-lg hover:bg-blue-200"
                  title="SMS"
                >
                  💬
                </button>
                <button
                  onClick={() => handleContact(lead, 'whatsapp')}
                  className="p-2 bg-green-100 text-green-600 rounded-lg hover:bg-green-200"
                  title="WhatsApp"
                >
                  🟢
                </button>
                <button
                  onClick={() => handleContact(lead, 'email')}
                  className="p-2 bg-gray-100 text-gray-600 rounded-lg hover:bg-gray-200"
                  title="Email"
                >
                  📨
                </button>
              </div>

              <span className="text-xs text-gray-500">
                {new Date(lead.created_at).toLocaleDateString()}
              </span>
            </div>

            <div className="flex items-center gap-2">
              <span className="text-sm font-medium">Prospect Stage:</span>
              <select
                value={lead.status}
                onChange={(e) => updateLeadStage(lead.id, e.target.value)}
                className={`px-2 py-1 rounded text-xs border ${getStatusColor(lead.status)}`}
              >
                <option value="new">New</option>
                <option value="contacted">Contacted</option>
                <option value="qualified">Qualified</option>
                <option value="negotiation">Negotiation</option>
                <option value="converted">Converted</option>
                <option value="lost">Lost</option>
              </select>
            </div>
          </div>
        ))}
      </div>

      {sortedLeads.length === 0 && (
        <div className="text-center py-12">
          <p className="text-gray-500">No leads found</p>
        </div>
      )}
    </div>
  );
};