import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { Wind, AlertCircle } from 'lucide-react';

interface AirQualityData {
  aqi: number | null;
  air_quality_score: number;
  temperature?: number;
  humidity?: number;
  location?: {
    city?: string;
    state?: string;
    country?: string;
  };
  is_real_data?: boolean;
  data_source?: string;
}

interface AirQualityScoreProps {
  propertyId: number;
}

const AirQualityScore: React.FC<AirQualityScoreProps> = ({ propertyId }) => {
  const [airQuality, setAirQuality] = useState<AirQualityData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchAirQuality = async () => {
      try {
        setLoading(true);
        console.log('🌬️ Fetching air quality for property:', propertyId);
        const url = `http://localhost:8090/api/properties/${propertyId}/air-quality`;
        console.log('🔗 Air Quality API URL:', url);
        
        const response = await fetch(url);
        console.log('📡 Air Quality API Response:', response.status, response.statusText);
        
        if (!response.ok) {
          console.error('❌ Air quality API error:', response.status, response.statusText);
          const errorText = await response.text();
          console.error('❌ Error details:', errorText);
          
          // Show specific error message
          if (response.status === 400) {
            console.warn('⚠️ Property coordinates not available - this is why you\'re seeing demo data');
          } else if (response.status === 404) {
            console.warn('⚠️ Property not found - this is why you\'re seeing demo data');
          } else if (response.status === 500) {
            console.warn('⚠️ Backend server error - this is why you\'re seeing demo data');
          }
          
          throw new Error(`Failed to fetch air quality data: ${response.status}`);
        }
        
        const data = await response.json();
        console.log('✅ Air Quality Data:', data);
        setAirQuality(data);
      } catch (err) {
        console.error('❌ Air Quality Error:', err);
        console.log('🔍 REASON FOR DEMO DATA: API call failed - likely property has no coordinates or backend is down');
        console.log('🔧 TO FIX: Ensure property has latitude/longitude and backend is running on port 8090');
        
        // Fallback to demo data if API fails
        setAirQuality({
          aqi: 85,
          air_quality_score: 3,
          temperature: 28,
          humidity: 65,
          location: {
            city: 'Bangalore',
            state: 'Karnataka',
            country: 'India'
          },
          is_real_data: false,
          data_source: 'Demo Data'
        });
      } finally {
        setLoading(false);
      }
    };

    fetchAirQuality();
  }, [propertyId]);

  const getScoreLabel = (score: number): string => {
    switch (score) {
      case 5: return 'Excellent';
      case 4: return 'Good';
      case 3: return 'Moderate';
      case 2: return 'Poor';
      case 1: return 'Very Poor';
      default: return 'Unknown';
    }
  };

  const getScoreColor = (score: number): string => {
    switch (score) {
      case 5: return 'bg-green-500';
      case 4: return 'bg-lime-500';
      case 3: return 'bg-yellow-500';
      case 2: return 'bg-orange-500';
      case 1: return 'bg-red-500';
      default: return 'bg-gray-500';
    }
  };

  const renderStars = (score: number) => {
    return Array.from({ length: 5 }, (_, i) => (
      <span key={i} className={i < score ? 'text-yellow-400' : 'text-gray-300'}>
        ★
      </span>
    ));
  };

  if (loading) {
    return (
      <Card className="w-full">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Wind className="h-5 w-5" />
            Air Quality
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="animate-pulse">
            <div className="h-4 bg-gray-200 rounded w-3/4 mb-2"></div>
            <div className="h-4 bg-gray-200 rounded w-1/2"></div>
          </div>
        </CardContent>
      </Card>
    );
  }



  return (
    <Card className="w-full bg-white border-0 shadow-xl rounded-2xl overflow-hidden">
      <CardHeader className="bg-gradient-to-r from-blue-500 to-indigo-600 pb-4">
        <CardTitle className="flex items-center gap-3">
          <div className="p-2 bg-white/20 rounded-full">
            <Wind className="h-5 w-5 text-white" />
          </div>
          <span className="text-xl font-bold text-black">Air Quality & Weather</span>
        </CardTitle>
      </CardHeader>
      <CardContent className="p-0">
        <div className="grid grid-cols-2 divide-x divide-gray-100">
          {/* Left Side - Air Quality (Google Weather Style) */}
          <div className="p-6 bg-gradient-to-br from-blue-50 to-blue-100">
            <div className="text-center mb-4">
              {/* Large AQI Number */}
              {airQuality?.aqi && (
                <div className="mb-2">
                  <div className="text-4xl font-bold text-gray-800 mb-1">
                    {airQuality.aqi}
                  </div>
                  <div className="text-sm text-gray-600 mb-3">AQI (PM2.5)</div>
                </div>
              )}
              
              {/* Stars and Quality Badge */}
              <div className="flex items-center justify-center gap-2 mb-3">
                <span className="text-2xl">
                  {renderStars(airQuality?.air_quality_score || 0)}
                </span>
              </div>
              
              <Badge className="bg-gradient-to-r from-blue-500 to-blue-600 text-white px-4 py-1 text-sm font-semibold rounded-full">
                {getScoreLabel(airQuality?.air_quality_score || 0)}
              </Badge>
            </div>
            
            {/* Location & Source */}
            <div className="space-y-2 text-center">
              {airQuality?.location?.city && (
                <div className="flex items-center justify-center gap-1">
                  <span className="text-sm">📍</span>
                  <span className="text-sm font-medium text-gray-700">
                    Nearest Station: {airQuality.location.city}
                  </span>
                </div>
              )}
              
              {airQuality?.data_source && (
                <div className="flex items-center justify-center gap-1">
                  <span className="text-xs">
                    {airQuality.is_real_data ? '🟢' : '🟡'}
                  </span>
                  <span className="text-xs text-gray-500">
                    {airQuality.is_real_data ? 'Live Data' : 'Demo'}
                  </span>
                </div>
              )}
            </div>
          </div>
          
          {/* Right Side - Weather (Google Weather Style) */}
          <div className="p-6 bg-gradient-to-br from-orange-50 to-yellow-100">
            <div className="space-y-6">
              {/* Temperature */}
              {airQuality?.temperature && (
                <div className="text-center">
                  <div className="text-5xl mb-2">🌡️</div>
                  <div className="text-3xl font-bold text-gray-800 mb-1">
                    {airQuality.temperature}°
                  </div>
                  <div className="text-sm text-gray-600">Temperature</div>
                </div>
              )}
              
              {/* Humidity */}
              {airQuality?.humidity && (
                <div className="text-center">
                  <div className="text-4xl mb-2">💧</div>
                  <div className="text-2xl font-bold text-gray-800 mb-1">
                    {airQuality.humidity}%
                  </div>
                  <div className="text-sm text-gray-600">Humidity</div>
                </div>
              )}
            </div>
          </div>
        </div>
        
        {/* Bottom Info Bar */}
        <div className="bg-gray-50 px-6 py-3 border-t border-gray-100">
          <div className="flex items-center justify-between text-xs text-gray-500">
            <span>Real-time environmental data</span>
            {airQuality?.data_source && (
              <span>Powered by {airQuality.data_source}</span>
            )}
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

export default AirQualityScore;