import React, { useState } from 'react';
import { AuctionSession, auctionUtils } from '../api/auctions';
import { useAuction } from '../hooks/useAuction';

interface AuctionCardProps {
  auction: AuctionSession;
  onViewDetails?: (sessionId: string) => void;
}

export const AuctionCard: React.FC<AuctionCardProps> = ({ auction, onViewDetails }) => {
  const [showBidForm, setShowBidForm] = useState(false);
  const [bidAmount, setBidAmount] = useState('');
  
  const {
    currentBid,
    timeRemaining,
    isRegistered,
    isLoading,
    error,
    placeBid,
    registerForAuction,
    validateBid
  } = useAuction(auction.session_id);

  const handlePlaceBid = async (e: React.FormEvent) => {
    e.preventDefault();
    const amount = parseFloat(bidAmount);
    
    const validation = validateBid(amount);
    if (!validation.isValid) {
      alert(validation.message);
      return;
    }

    try {
      await placeBid(amount);
      setBidAmount('');
      setShowBidForm(false);
    } catch (error) {
      console.error('Failed to place bid:', error);
    }
  };

  const handleRegister = async () => {
    try {
      await registerForAuction();
    } catch (error) {
      console.error('Failed to register:', error);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'upcoming': return 'bg-blue-100 text-blue-800';
      case 'paused': return 'bg-yellow-100 text-yellow-800';
      case 'completed': return 'bg-gray-100 text-gray-800';
      case 'cancelled': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <div className="bg-white rounded-lg shadow-md p-6 border border-gray-200">
      {/* Header */}
      <div className="flex justify-between items-start mb-4">
        <div>
          <h3 className="text-lg font-semibold text-gray-900 mb-1">{auction.title}</h3>
          <span className={`inline-block px-2 py-1 rounded-full text-xs font-medium ${getStatusColor(auction.status)}`}>
            {auction.status.toUpperCase()}
          </span>
        </div>
        <button
          onClick={() => onViewDetails?.(auction.session_id)}
          className="text-blue-600 hover:text-blue-800 text-sm font-medium"
        >
          View Details
        </button>
      </div>

      {/* Auction Info */}
      <div className="space-y-3 mb-4">
        <div className="flex justify-between">
          <span className="text-gray-600">Starting Bid:</span>
          <span className="font-medium">{auctionUtils.formatCurrency(auction.starting_bid)}</span>
        </div>
        
        <div className="flex justify-between">
          <span className="text-gray-600">Current Bid:</span>
          <span className="font-semibold text-green-600">
            {auctionUtils.formatCurrency(currentBid || auction.current_bid)}
          </span>
        </div>

        <div className="flex justify-between">
          <span className="text-gray-600">Registration Fee:</span>
          <span className="font-medium">{auctionUtils.formatCurrency(auction.registration_fee)}</span>
        </div>

        {auction.status === 'active' && (
          <div className="flex justify-between">
            <span className="text-gray-600">Time Remaining:</span>
            <span className="font-medium text-red-600">{timeRemaining}</span>
          </div>
        )}

        <div className="flex justify-between">
          <span className="text-gray-600">Total Bids:</span>
          <span className="font-medium">{auction.total_bids}</span>
        </div>
      </div>

      {/* Error Display */}
      {error && (
        <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-md">
          <p className="text-red-600 text-sm">{error}</p>
        </div>
      )}

      {/* Action Buttons */}
      <div className="space-y-2">
        {auction.status === 'active' && (
          <>
            {!isRegistered ? (
              <button
                onClick={handleRegister}
                disabled={isLoading}
                className="w-full bg-blue-600 text-white py-2 px-4 rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {isLoading ? 'Registering...' : `Register (${auctionUtils.formatCurrency(auction.registration_fee)})`}
              </button>
            ) : (
              <>
                {!showBidForm ? (
                  <button
                    onClick={() => setShowBidForm(true)}
                    className="w-full bg-green-600 text-white py-2 px-4 rounded-md hover:bg-green-700"
                  >
                    Place Bid
                  </button>
                ) : (
                  <form onSubmit={handlePlaceBid} className="space-y-2">
                    <input
                      type="number"
                      value={bidAmount}
                      onChange={(e) => setBidAmount(e.target.value)}
                      placeholder={`Min: ${auctionUtils.formatCurrency((currentBid || auction.current_bid) + auction.minimum_increment)}`}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      min={(currentBid || auction.current_bid) + auction.minimum_increment}
                      step={auction.minimum_increment}
                      required
                    />
                    <div className="flex space-x-2">
                      <button
                        type="submit"
                        disabled={isLoading}
                        className="flex-1 bg-green-600 text-white py-2 px-4 rounded-md hover:bg-green-700 disabled:opacity-50"
                      >
                        {isLoading ? 'Placing...' : 'Submit Bid'}
                      </button>
                      <button
                        type="button"
                        onClick={() => setShowBidForm(false)}
                        className="flex-1 bg-gray-300 text-gray-700 py-2 px-4 rounded-md hover:bg-gray-400"
                      >
                        Cancel
                      </button>
                    </div>
                  </form>
                )}
              </>
            )}
          </>
        )}

        {auction.status === 'upcoming' && (
          <button
            onClick={handleRegister}
            disabled={isLoading || isRegistered}
            className="w-full bg-blue-600 text-white py-2 px-4 rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {isRegistered ? 'Registered' : isLoading ? 'Registering...' : 'Pre-Register'}
          </button>
        )}
      </div>

      {/* Auction Details */}
      <div className="mt-4 pt-4 border-t border-gray-200">
        <p className="text-sm text-gray-600 line-clamp-2">{auction.description}</p>
        <div className="flex justify-between items-center mt-2 text-xs text-gray-500">
          <span>{auction.registered_bidders} registered bidders</span>
          <span>ID: {auction.session_id.slice(0, 8)}...</span>
        </div>
      </div>
    </div>
  );
};

export default AuctionCard;