import React, { useState } from 'react';
import { auctionAPI, userAuctionAPI, adminAuctionAPI } from '../api/auctions';

const AuctionDemo: React.FC = () => {
  const [results, setResults] = useState<any>(null);
  const [loading, setLoading] = useState(false);

  const testAPI = async (apiCall: () => Promise<any>, description: string) => {
    setLoading(true);
    try {
      console.log(`Testing: ${description}`);
      const result = await apiCall();
      console.log(`✅ ${description} - Success:`, result);
      setResults({ success: true, description, data: result });
    } catch (error) {
      console.error(`❌ ${description} - Error:`, error);
      setResults({ success: false, description, error: error instanceof Error ? error.message : 'Unknown error' });
    } finally {
      setLoading(false);
    }
  };

  const tests = [
    {
      name: 'Get Active Auctions',
      action: () => testAPI(() => auctionAPI.getActiveAuctions(), 'Fetch Active Auctions')
    },
    {
      name: 'Get Upcoming Auctions',
      action: () => testAPI(() => auctionAPI.getUpcomingAuctions(), 'Fetch Upcoming Auctions')
    },
    {
      name: 'Get All Auctions (Admin)',
      action: () => testAPI(() => adminAuctionAPI.getAllAuctions(), 'Fetch All Auctions (Admin)')
    },
    {
      name: 'Create Test Auction',
      action: () => testAPI(() => adminAuctionAPI.createAuction({
        property_id: 'test-property-123',
        title: 'Test Auction Property',
        description: 'This is a test auction for API integration',
        starting_bid: 1000000,
        minimum_increment: 50000,
        registration_fee: 10000,
        start_time: new Date(Date.now() + 60000).toISOString(), // 1 minute from now
        end_time: new Date(Date.now() + 3600000).toISOString() // 1 hour from now
      }), 'Create Test Auction')
    }
  ];

  return (
    <div className="max-w-4xl mx-auto p-6">
      <h1 className="text-3xl font-bold text-gray-900 mb-6">Auction API Integration Demo</h1>
      
      <div className="bg-white rounded-lg shadow-md p-6 mb-6">
        <h2 className="text-xl font-semibold mb-4">API Configuration</h2>
        <div className="bg-gray-50 p-4 rounded-md">
          <p><strong>Backend URL:</strong> http://localhost:8090/api</p>
          <p><strong>Auction API:</strong> /auctions</p>
          <p><strong>Admin API:</strong> /auction-management</p>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
        {tests.map((test, index) => (
          <button
            key={index}
            onClick={test.action}
            disabled={loading}
            className="bg-blue-600 text-white px-4 py-3 rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {loading ? 'Testing...' : test.name}
          </button>
        ))}
      </div>

      {results && (
        <div className="bg-white rounded-lg shadow-md p-6">
          <h3 className="text-lg font-semibold mb-4">
            {results.success ? '✅ Success' : '❌ Error'}: {results.description}
          </h3>
          
          <div className="bg-gray-50 p-4 rounded-md overflow-auto">
            <pre className="text-sm">
              {results.success 
                ? JSON.stringify(results.data, null, 2)
                : results.error
              }
            </pre>
          </div>
        </div>
      )}

      <div className="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-blue-900 mb-3">Integration Status</h3>
        <div className="space-y-2 text-sm">
          <div className="flex items-center">
            <span className="w-3 h-3 bg-green-500 rounded-full mr-2"></span>
            <span>Auction API endpoints integrated</span>
          </div>
          <div className="flex items-center">
            <span className="w-3 h-3 bg-green-500 rounded-full mr-2"></span>
            <span>React hooks for auction management</span>
          </div>
          <div className="flex items-center">
            <span className="w-3 h-3 bg-green-500 rounded-full mr-2"></span>
            <span>WebSocket support for real-time updates</span>
          </div>
          <div className="flex items-center">
            <span className="w-3 h-3 bg-green-500 rounded-full mr-2"></span>
            <span>Admin auction management interface</span>
          </div>
          <div className="flex items-center">
            <span className="w-3 h-3 bg-green-500 rounded-full mr-2"></span>
            <span>User bidding interface</span>
          </div>
        </div>
      </div>

      <div className="mt-6 bg-yellow-50 border border-yellow-200 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-yellow-900 mb-3">Next Steps</h3>
        <ul className="list-disc list-inside space-y-1 text-sm text-yellow-800">
          <li>Start your backend server on port 8090</li>
          <li>Test the API endpoints using the buttons above</li>
          <li>Navigate to /auctions to see the auction list page</li>
          <li>Use admin dashboard to create and manage auctions</li>
          <li>Implement authentication for user-specific features</li>
        </ul>
      </div>
    </div>
  );
};

export default AuctionDemo;