import React, { useState, useEffect } from 'react';

interface AuctionRegistrationModalProps {
  isOpen: boolean;
  onClose: () => void;
  onSubmit: (details: any) => void;
  loading: boolean;
  registrationFee: number;
}

export const AuctionRegistrationModal: React.FC<AuctionRegistrationModalProps> = ({
  isOpen, onClose, onSubmit, loading, registrationFee
}) => {
  const [formData, setFormData] = useState({
    full_name: '',
    email: '',
    phone: '',
    address: '',
    pan_number: '',
    aadhar_number: '',
    occupation: '',
    annual_income: '',
    terms_accepted: false
  });
  const [errors, setErrors] = useState<Record<string, string>>({});

  const validateField = (name: string, value: string) => {
    switch (name) {
      case 'full_name':
        if (!value.trim()) return 'Full name is required';
        if (value.trim().length < 3) return 'Name must be at least 3 characters';
        if (!/^[a-zA-Z\s]+$/.test(value)) return 'Name can only contain letters';
        break;
      case 'email':
        if (!value.trim()) return 'Email is required';
        if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value)) return 'Invalid email format';
        break;
      case 'phone':
        if (!value.trim()) return 'Phone is required';
        if (!/^[6-9]\d{9}$/.test(value)) return 'Invalid phone (10 digits, start with 6-9)';
        break;
      case 'pan_number':
        if (!value.trim()) return 'PAN is required';
        if (!/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/.test(value)) return 'Invalid PAN format (e.g., ABCDE1234F)';
        break;
      case 'aadhar_number':
        if (value && !/^\d+$/.test(value)) return 'Aadhar can only contain digits';
        if (value && value.length !== 12) return 'Aadhar must be exactly 12 digits';
        break;
      case 'address':
        if (!value.trim()) return 'Address is required';
        if (value.trim().length < 10) return 'Address must be at least 10 characters';
        break;
      case 'annual_income':
        if (value && (isNaN(Number(value)) || Number(value) < 0)) return 'Invalid income';
        break;
    }
    return '';
  };

  const handleChange = (name: string, value: string) => {
    setFormData(prev => ({ ...prev, [name]: value }));
    const error = validateField(name, value);
    setErrors(prev => ({ ...prev, [name]: error }));
  };

  // Auto-fill user data from localStorage when modal opens
  useEffect(() => {
    if (!isOpen) return;
    
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const fullName = `${user.first_name || ''} ${user.last_name || ''}`.trim();
    
    setFormData(prev => ({
      ...prev,
      email: user.email || prev.email,
      full_name: fullName || user.name || user.full_name || prev.full_name
    }));
  }, [isOpen]);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    // Validate all fields
    const newErrors: Record<string, string> = {};
    Object.keys(formData).forEach(key => {
      if (key !== 'terms_accepted') {
        const error = validateField(key, formData[key as keyof typeof formData] as string);
        if (error) newErrors[key] = error;
      }
    });
    
    if (!formData.terms_accepted) {
      alert('Please accept terms and conditions');
      return;
    }
    
    if (Object.keys(newErrors).length > 0) {
      setErrors(newErrors);
      alert('Please fix all errors before submitting');
      return;
    }
    
    onSubmit(formData);
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-lg w-full max-w-4xl">
        {/* Header */}
        <div className="p-4 border-b">
          <div className="flex justify-between items-center">
            <h2 className="text-xl font-bold">Auction Registration</h2>
            <button onClick={onClose} className="text-gray-500 hover:text-gray-700">
              <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
              </svg>
            </button>
          </div>
          <p className="text-gray-600 text-sm mt-1">Registration Fee: ₹{registrationFee.toLocaleString()}</p>
        </div>

        {/* Form */}
        <div className="p-4 max-h-96 overflow-y-auto">
          <form onSubmit={handleSubmit} className="space-y-3">
            <div className="grid grid-cols-3 gap-3">
              <div>
                <label className="block text-xs font-medium mb-1">Full Name *</label>
                <input
                  type="text"
                  required
                  value={formData.full_name}
                  onChange={(e) => handleChange('full_name', e.target.value)}
                  className={`w-full p-2 text-sm border rounded focus:ring-1 focus:ring-blue-500 ${errors.full_name ? 'border-red-500' : ''}`}
                  placeholder="Full name"
                />
                {errors.full_name && <p className="text-xs text-red-500 mt-1">{errors.full_name}</p>}
              </div>
              <div>
                <label className="block text-xs font-medium mb-1">Email *</label>
                <input
                  type="email"
                  required
                  value={formData.email}
                  onChange={(e) => handleChange('email', e.target.value)}
                  className={`w-full p-2 text-sm border rounded focus:ring-1 focus:ring-blue-500 ${errors.email ? 'border-red-500' : ''}`}
                  placeholder="Email"
                />
                {errors.email && <p className="text-xs text-red-500 mt-1">{errors.email}</p>}
              </div>
              <div>
                <label className="block text-xs font-medium mb-1">Phone *</label>
                <input
                  type="tel"
                  required
                  value={formData.phone}
                  onChange={(e) => handleChange('phone', e.target.value)}
                  className={`w-full p-2 text-sm border rounded focus:ring-1 focus:ring-blue-500 ${errors.phone ? 'border-red-500' : ''}`}
                  placeholder="10-digit mobile"
                  maxLength={10}
                />
                {errors.phone && <p className="text-xs text-red-500 mt-1">{errors.phone}</p>}
              </div>
              <div>
                <label className="block text-xs font-medium mb-1">PAN *</label>
                <input
                  type="text"
                  required
                  value={formData.pan_number}
                  onChange={(e) => handleChange('pan_number', e.target.value.toUpperCase())}
                  className={`w-full p-2 text-sm border rounded focus:ring-1 focus:ring-blue-500 ${errors.pan_number ? 'border-red-500' : ''}`}
                  placeholder="ABCDE1234F"
                  maxLength={10}
                />
                {errors.pan_number && <p className="text-xs text-red-500 mt-1">{errors.pan_number}</p>}
              </div>
              <div>
                <label className="block text-xs font-medium mb-1">Aadhar</label>
                <input
                  type="text"
                  value={formData.aadhar_number}
                  onChange={(e) => handleChange('aadhar_number', e.target.value)}
                  className={`w-full p-2 text-sm border rounded focus:ring-1 focus:ring-blue-500 ${errors.aadhar_number ? 'border-red-500' : ''}`}
                  placeholder="12-digit Aadhar"
                  maxLength={12}
                />
                {errors.aadhar_number && <p className="text-xs text-red-500 mt-1">{errors.aadhar_number}</p>}
              </div>
              <div>
                <label className="block text-xs font-medium mb-1">Occupation</label>
                <input
                  type="text"
                  value={formData.occupation}
                  onChange={(e) => handleChange('occupation', e.target.value)}
                  className="w-full p-2 text-sm border rounded focus:ring-1 focus:ring-blue-500"
                  placeholder="Occupation"
                />
              </div>
            </div>

            <div>
              <label className="block text-xs font-medium mb-1">Address *</label>
              <textarea
                required
                value={formData.address}
                onChange={(e) => handleChange('address', e.target.value)}
                className={`w-full p-2 text-sm border rounded focus:ring-1 focus:ring-blue-500 ${errors.address ? 'border-red-500' : ''}`}
                rows={2}
                placeholder="Complete address"
              />
              {errors.address && <p className="text-xs text-red-500 mt-1">{errors.address}</p>}
            </div>

            <div>
              <label className="block text-xs font-medium mb-1">Annual Income (₹)</label>
              <input
                type="number"
                value={formData.annual_income}
                onChange={(e) => handleChange('annual_income', e.target.value)}
                className={`w-full p-2 text-sm border rounded focus:ring-1 focus:ring-blue-500 ${errors.annual_income ? 'border-red-500' : ''}`}
                placeholder="Annual income"
              />
              {errors.annual_income && <p className="text-xs text-red-500 mt-1">{errors.annual_income}</p>}
            </div>

            <div className="bg-gray-50 p-3 rounded">
              <div className="flex items-start space-x-2">
                <input
                  type="checkbox"
                  id="terms"
                  checked={formData.terms_accepted}
                  onChange={(e) => setFormData({...formData, terms_accepted: e.target.checked})}
                  className="mt-1"
                />
                <label htmlFor="terms" className="text-xs text-gray-700">
                  I accept the terms and conditions for auction participation *
                </label>
              </div>
            </div>
          </form>
        </div>

        {/* Footer */}
        <div className="p-4 border-t bg-gray-50 flex-shrink-0">
          <div className="flex gap-3">
            <button
              type="button"
              onClick={onClose}
              className="flex-1 px-4 py-3 border-2 border-gray-300 text-gray-700 rounded hover:bg-gray-100 font-medium"
            >
              Cancel
            </button>
            <button
              onClick={handleSubmit}
              disabled={loading || !formData.terms_accepted}
              className="flex-1 px-6 py-3 rounded-lg flex items-center justify-center gap-2 font-semibold transition-colors shadow-lg"
              style={{ backgroundColor: loading ? '#9ca3af' : '#2563eb', color: 'white', minHeight: '48px' }}
            >
              {loading ? (
                <>
                  <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white"></div>
                  <span>Submitting...</span>
                </>
              ) : (
                <>
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v2a2 2 0 002 2z" />
                  </svg>
                  <span>Submit & Proceed to Payment</span>
                </>
              )}
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};