"use client"

import React, { useState } from 'react'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Eye, EyeOff, Mail, Lock, X, Check } from 'lucide-react'

interface AuthModalProps {
  onClose: () => void
  onSuccess?: (userData: any) => void
}

const AuthModal: React.FC<AuthModalProps> = ({ onClose, onSuccess }) => {
  const [activeTab, setActiveTab] = useState<'login' | 'signup'>('login')
  const [showPassword, setShowPassword] = useState(false)
  const [rememberMe, setRememberMe] = useState(false)
  const [loginData, setLoginData] = useState({ emailMobile: '', password: '' })
  const [signupData, setSignupData] = useState({
    userRole: '', fullName: '', email: '', mobile: '', password: '', confirmPassword: '',
    hasAgency: '', agencyName: '', reraNumber: '', experience: '', officeAddress: '',
    documents: null as FileList | null, profilePhoto: null as FileList | null, areasOfOperation: '', propertyTypes: [] as string[],
    specialization: '', website: ''
  })

  const handleLoginChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setLoginData({ ...loginData, [e.target.name]: e.target.value })
  }

  const handleSignupChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value, type } = e.target
    const target = e.target as HTMLInputElement
    
    if (type === 'file') {
      setSignupData({ ...signupData, [name]: target.files })
    } else if (type === 'checkbox' && name === 'propertyTypes') {
      const updatedTypes = target.checked 
        ? [...signupData.propertyTypes, value]
        : signupData.propertyTypes.filter((type: string) => type !== value)
      setSignupData({ ...signupData, propertyTypes: updatedTypes })
    } else {
      setSignupData({ ...signupData, [name]: value })
    }
  }

  const handleLoginSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    const userData = {
      fullName: 'User',
      email: loginData.emailMobile,
      userRole: 'buyer'
    }
    onSuccess?.(userData)
  }

  const handleSignupSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    if (signupData.password !== signupData.confirmPassword) {
      alert('Passwords do not match')
      return
    }
    const userData = {
      fullName: signupData.fullName,
      email: signupData.email,
      userRole: signupData.userRole
    }
    onSuccess?.(userData)
  }

  const handleSocialLogin = (provider: string) => {
    const userData = {
      fullName: 'User',
      email: 'user@example.com',
      userRole: 'buyer'
    }
    onSuccess?.(userData)
  }

  return (
    <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-2xl shadow-2xl w-full max-w-5xl h-[90vh] max-h-[700px] overflow-hidden relative">
        {/* Close Button */}
        <button 
          onClick={onClose} 
          className="absolute top-6 right-6 z-10 bg-white/90 rounded-full p-2 shadow-lg hover:bg-white transition-colors"
        >
          <X className="h-5 w-5 text-gray-600" />
        </button>
        
        <div className="flex h-full">
          {/* Left Panel - Brand/Message */}
          <div className="hidden md:flex md:w-1/2 bg-gradient-to-br from-[#0056D2] via-[#0056D2]/90 to-[#00BFA6] p-12 flex-col justify-center text-white relative overflow-hidden">
            {/* Background Pattern */}
            <div className="absolute inset-0 opacity-10">
              <div className="absolute top-10 left-10 w-32 h-32 border border-white/20 rounded-full"></div>
              <div className="absolute bottom-20 right-10 w-24 h-24 border border-white/20 rounded-full"></div>
              <div className="absolute top-1/2 left-1/4 w-16 h-16 border border-white/20 rounded-full"></div>
            </div>
            
            <div className="relative z-10">
              {/* Logo */}
              <div className="mb-8">
                <img src="/src/assets/logo.jpg" alt="NAL India" className="h-12 w-auto mb-4" />
                <h1 className="text-3xl font-bold">NAL India</h1>
              </div>
              
              {/* Main Message */}
              <div className="mb-8">
                <h2 className="text-2xl font-semibold mb-4 leading-tight">
                  Find with Confidence,<br />Invest with Trust.
                </h2>
                <p className="text-lg text-white/90 mb-8">
                  NAL India ensures every property is verified, secure, and future-ready.
                </p>
              </div>
              
              {/* Feature Highlights */}
              <div className="space-y-4">
                {[
                  'Verified Properties',
                  'Trusted Agents', 
                  'Transparent Deals'
                ].map((feature, index) => (
                  <div key={index} className="flex items-center space-x-3">
                    <div className="bg-white/20 rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <span className="text-white/90">{feature}</span>
                  </div>
                ))}
              </div>
            </div>
          </div>
          
          {/* Right Panel - Form Section */}
          <div className="w-full md:w-1/2 p-8 md:p-12 flex flex-col justify-center">
            <div className="w-full max-w-md mx-auto">
              {/* Mobile Logo */}
              <div className="md:hidden mb-8 text-center">
                <img src="/src/assets/logo.jpg" alt="NAL India" className="h-10 w-auto mx-auto mb-2" />
                <h1 className="text-2xl font-bold text-[#0056D2]">NAL India</h1>
              </div>
              
              <Tabs value={activeTab} onValueChange={(value: string) => setActiveTab(value as 'login' | 'signup')} className="w-full">
                <TabsList className="grid w-full grid-cols-2 mb-8">
                  <TabsTrigger value="login" className="text-sm font-medium">Login</TabsTrigger>
                  <TabsTrigger value="signup" className="text-sm font-medium">Sign Up</TabsTrigger>
                </TabsList>
                
                <TabsContent value="login" className="space-y-6">
                  <div className="text-center mb-6">
                    <h2 className="text-2xl font-semibold text-gray-900 mb-2">Welcome Back</h2>
                    <p className="text-gray-600">Sign in to your account to continue</p>
                  </div>
                  
                  <form onSubmit={handleLoginSubmit} className="space-y-4">
                    <div className="space-y-2">
                      <Label htmlFor="login-email" className="text-sm font-medium text-gray-700">Email / Mobile Number</Label>
                      <div className="relative">
                        <Mail className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
                        <Input
                          id="login-email"
                          name="emailMobile"
                          type="text"
                          placeholder="Enter your email or mobile number"
                          value={loginData.emailMobile}
                          onChange={handleLoginChange}
                          className="pl-10 h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                          required
                        />
                      </div>
                    </div>
                    
                    <div className="space-y-2">
                      <Label htmlFor="login-password" className="text-sm font-medium text-gray-700">Password</Label>
                      <div className="relative">
                        <Lock className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
                        <Input
                          id="login-password"
                          name="password"
                          type={showPassword ? 'text' : 'password'}
                          placeholder="Enter your password"
                          value={loginData.password}
                          onChange={handleLoginChange}
                          className="pl-10 pr-10 h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                          required
                        />
                        <Button
                          type="button"
                          variant="ghost"
                          size="sm"
                          className="absolute right-0 top-0 h-full px-3 py-2 hover:bg-transparent"
                          onClick={() => setShowPassword(!showPassword)}
                        >
                          {showPassword ? (
                            <EyeOff className="h-4 w-4 text-gray-400" />
                          ) : (
                            <Eye className="h-4 w-4 text-gray-400" />
                          )}
                        </Button>
                      </div>
                    </div>
                    
                    <div className="flex items-center justify-between">
                      <div className="flex items-center">
                        <input
                          id="remember-me"
                          type="checkbox"
                          checked={rememberMe}
                          onChange={(e) => setRememberMe(e.target.checked)}
                          className="h-4 w-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2]"
                        />
                        <Label htmlFor="remember-me" className="ml-2 text-sm text-gray-600">Remember me</Label>
                      </div>
                      <Button variant="link" className="p-0 h-auto text-sm text-[#0056D2] hover:text-[#0056D2]/80">
                        Forgot Password?
                      </Button>
                    </div>
                    
                    <Button type="submit" className="w-full h-12 bg-[#0056D2] hover:bg-[#0056D2]/90 text-white font-medium rounded-lg">
                      Sign In
                    </Button>
                    
                    <div className="text-center">
                      <Button variant="link" className="text-sm text-[#0056D2] hover:text-[#0056D2]/80">
                        Login with OTP
                      </Button>
                    </div>
                  </form>
                </TabsContent>
                
                <TabsContent value="signup" className="space-y-6">
                  <div className="text-center mb-6">
                    <h2 className="text-2xl font-semibold text-gray-900 mb-2">Create Account</h2>
                    <p className="text-gray-600">Join NAL India to get started</p>
                  </div>
                  
                  <div className="max-h-[400px] overflow-y-auto pr-2">
                    <form onSubmit={handleSignupSubmit} className="space-y-4">
                      <div>
                        <Label className="block text-sm font-medium text-gray-700 mb-3">I am a</Label>
                        <div className="grid grid-cols-3 gap-2">
                          {['buyer', 'seller', 'agent'].map((role) => (
                            <button
                              key={role}
                              type="button"
                              onClick={() => setSignupData({ ...signupData, userRole: role })}
                              className={`py-2 px-3 rounded-lg text-sm font-medium transition-all ${
                                signupData.userRole === role
                                  ? 'bg-[#0056D2] text-white shadow-md'
                                  : 'border border-gray-300 text-gray-700 hover:border-[#0056D2] hover:text-[#0056D2]'
                              }`}
                            >
                              {role.charAt(0).toUpperCase() + role.slice(1)}
                            </button>
                          ))}
                        </div>
                      </div>

                      {signupData.userRole === 'agent' && (
                        <div>
                          <Label className="block text-sm font-medium text-gray-700 mb-3">Do you belong to an Agency?</Label>
                          <div className="grid grid-cols-2 gap-2">
                            {['yes', 'no'].map((option) => (
                              <button
                                key={option}
                                type="button"
                                onClick={() => setSignupData({ ...signupData, hasAgency: option })}
                                className={`py-2 px-4 rounded-lg text-sm font-medium transition-all ${
                                  signupData.hasAgency === option
                                    ? 'bg-[#0056D2] text-white shadow-md'
                                    : 'border border-gray-300 text-gray-700 hover:border-[#0056D2] hover:text-[#0056D2]'
                                }`}
                              >
                                {option.charAt(0).toUpperCase() + option.slice(1)}
                              </button>
                            ))}
                          </div>
                        </div>
                      )}

                      <Input
                        name="fullName"
                        placeholder="Full Name"
                        value={signupData.fullName}
                        onChange={handleSignupChange}
                        className="h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                        required
                      />
                      <Input
                        name="email"
                        type="email"
                        placeholder="Email Address"
                        value={signupData.email}
                        onChange={handleSignupChange}
                        className="h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                        required
                      />
                      <Input
                        name="mobile"
                        type="tel"
                        placeholder="Mobile Number"
                        value={signupData.mobile}
                        onChange={handleSignupChange}
                        className="h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                        required
                      />
                      <Input
                        name="password"
                        type="password"
                        placeholder="Create Password"
                        value={signupData.password}
                        onChange={handleSignupChange}
                        className="h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                        required
                      />

                      {signupData.userRole === 'agent' && signupData.hasAgency === 'yes' && (
                        <div className="space-y-4 pt-4 border-t border-gray-200">
                          <h4 className="font-semibold text-gray-900">Professional Details</h4>
                          <Input
                            name="agencyName"
                            placeholder="Agency / Company Name"
                            value={signupData.agencyName}
                            onChange={handleSignupChange}
                            className="h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                          />
                          <Input
                            name="reraNumber"
                            placeholder="RERA Number / License ID"
                            value={signupData.reraNumber}
                            onChange={handleSignupChange}
                            className="h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                          />
                          <div className="grid grid-cols-2 gap-3">
                            <Input
                              name="experience"
                              type="number"
                              placeholder="Years of Experience"
                              value={signupData.experience}
                              onChange={handleSignupChange}
                              className="h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                              min="0"
                            />
                            <select
                              name="specialization"
                              value={signupData.specialization}
                              onChange={handleSignupChange}
                              className="h-12 px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                            >
                              <option value="">Specialization</option>
                              <option value="rentals">Rentals</option>
                              <option value="sales">Sales</option>
                              <option value="luxury">Luxury</option>
                            </select>
                          </div>
                          <Input
                            name="officeAddress"
                            placeholder="Office Address / City"
                            value={signupData.officeAddress}
                            onChange={handleSignupChange}
                            className="h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                          />
                          <Input
                            name="areasOfOperation"
                            placeholder="Areas of Operation"
                            value={signupData.areasOfOperation}
                            onChange={handleSignupChange}
                            className="h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                          />
                          
                          <div>
                            <Label className="block text-sm font-medium text-gray-700 mb-2">Property Types</Label>
                            <div className="flex gap-4">
                              {['residential', 'commercial', 'land'].map((type) => (
                                <label key={type} className="flex items-center gap-2 cursor-pointer">
                                  <input
                                    type="checkbox"
                                    name="propertyTypes"
                                    value={type}
                                    checked={signupData.propertyTypes.includes(type as string)}
                                    onChange={handleSignupChange}
                                    className="w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2]"
                                  />
                                  <span className="text-sm text-gray-700">{type.charAt(0).toUpperCase() + type.slice(1)}</span>
                                </label>
                              ))}
                            </div>
                          </div>
                          
                          <div className="grid grid-cols-1 gap-3">
                            <div>
                              <Label className="block text-sm font-medium text-gray-700 mb-1">Documents Upload</Label>
                              <input
                                type="file"
                                name="documents"
                                onChange={handleSignupChange}
                                multiple
                                accept=".pdf,.jpg,.jpeg,.png"
                                className="w-full px-3 py-2 border border-gray-300 rounded-md text-sm h-12"
                              />
                            </div>
                            <div>
                              <Label className="block text-sm font-medium text-gray-700 mb-1">Profile Photo</Label>
                              <input
                                type="file"
                                name="profilePhoto"
                                onChange={handleSignupChange}
                                accept=".jpg,.jpeg,.png"
                                className="w-full px-3 py-2 border border-gray-300 rounded-md text-sm h-12"
                              />
                            </div>
                          </div>
                          
                          <Input
                            name="website"
                            type="url"
                            placeholder="Website / Social Links (Optional)"
                            value={signupData.website}
                            onChange={handleSignupChange}
                            className="h-12 border-gray-300 focus:border-[#0056D2] focus:ring-[#0056D2]"
                          />
                        </div>
                      )}

                      <Button type="submit" className="w-full h-12 bg-[#0056D2] hover:bg-[#0056D2]/90 text-white font-medium rounded-lg">
                        Sign Up
                      </Button>
                    </form>
                  </div>
                </TabsContent>
              </Tabs>
              
              {/* Switch between Login/Signup */}
              <div className="text-center mt-6">
                <p className="text-sm text-gray-600">
                  {activeTab === 'login' ? (
                    <>
                      Don't have an account?{' '}
                      <Button
                        variant="link"
                        className="p-0 h-auto font-medium text-[#0056D2] hover:text-[#0056D2]/80"
                        onClick={() => setActiveTab('signup')}
                      >
                        Sign up
                      </Button>
                    </>
                  ) : (
                    <>
                      Already have an account?{' '}
                      <Button
                        variant="link"
                        className="p-0 h-auto font-medium text-[#0056D2] hover:text-[#0056D2]/80"
                        onClick={() => setActiveTab('login')}
                      >
                        Sign in
                      </Button>
                    </>
                  )}
                </p>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}

export default AuthModal