import React, { useState } from 'react';
import { AzureBlobMediaGallery } from './AzureBlobMediaGallery';
import { useAzureBlobMedia } from '../hooks/useAzureBlobMedia';
import { toast } from 'sonner';

/**
 * Example component demonstrating Azure Blob Storage integration
 * This shows how to use all 4 Azure Blob APIs:
 * 1. POST /api/property-media/upload - Upload Property Media
 * 2. GET /api/property-media/{property_id} - Get Property Media  
 * 3. GET /api/property-media/{media_id}/download - Generate Download URL
 * 4. PUT /api/property-media/{media_id} - Replace Media
 */
export const AzureBlobExample: React.FC = () => {
  const [propertyId, setPropertyId] = useState('1'); // Example property ID
  const [selectedFiles, setSelectedFiles] = useState<FileList | null>(null);

  const {
    images,
    videos,
    documents,
    loading,
    uploading,
    error,
    uploadImages,
    uploadVideos,
    uploadDocuments,
    getDownloadUrl,
    replaceMediaFile,
    deleteMediaFile,
    loadPropertyMedia,
    totalMediaCount,
    hasMedia
  } = useAzureBlobMedia(propertyId);

  // Example: Upload multiple images
  const handleImageUpload = async () => {
    if (!selectedFiles || selectedFiles.length === 0) {
      toast.error('Please select files to upload');
      return;
    }

    const imageFiles = Array.from(selectedFiles).filter(file => 
      file.type.startsWith('image/')
    );

    if (imageFiles.length === 0) {
      toast.error('Please select image files');
      return;
    }

    const result = await uploadImages(propertyId, imageFiles);
    if (result.success) {
      toast.success(`Uploaded ${imageFiles.length} images to Azure Blob Storage`);
      setSelectedFiles(null);
    }
  };

  // Example: Upload videos
  const handleVideoUpload = async () => {
    if (!selectedFiles || selectedFiles.length === 0) {
      toast.error('Please select video files to upload');
      return;
    }

    const videoFiles = Array.from(selectedFiles).filter(file => 
      file.type.startsWith('video/')
    );

    if (videoFiles.length === 0) {
      toast.error('Please select video files');
      return;
    }

    const result = await uploadVideos(propertyId, videoFiles);
    if (result.success) {
      toast.success(`Uploaded ${videoFiles.length} videos to Azure Blob Storage`);
      setSelectedFiles(null);
    }
  };

  // Example: Upload documents
  const handleDocumentUpload = async () => {
    if (!selectedFiles || selectedFiles.length === 0) {
      toast.error('Please select document files to upload');
      return;
    }

    const documentFiles = Array.from(selectedFiles).filter(file => 
      file.type === 'application/pdf' || 
      file.type.includes('document') ||
      file.type.includes('text')
    );

    if (documentFiles.length === 0) {
      toast.error('Please select document files (PDF, DOC, etc.)');
      return;
    }

    const documentTypes = documentFiles.map(() => 'general'); // Default type
    const result = await uploadDocuments(propertyId, documentFiles, documentTypes);
    if (result.success) {
      toast.success(`Uploaded ${documentFiles.length} documents to Azure Blob Storage`);
      setSelectedFiles(null);
    }
  };

  // Example: Download media with secure URL
  const handleDownloadExample = async (mediaId: string) => {
    const result = await getDownloadUrl(mediaId);
    if (result.success && result.url) {
      // Open download URL in new tab
      window.open(result.url, '_blank');
      toast.success('Download URL generated successfully');
    }
  };

  return (
    <div className="max-w-6xl mx-auto p-6 space-y-8">
      <div className="bg-white rounded-lg shadow-lg p-6">
        <h1 className="text-2xl font-bold text-gray-900 mb-4">
          Azure Blob Storage Integration Example
        </h1>
        <p className="text-gray-600 mb-6">
          This example demonstrates the integration of all 4 Azure Blob Storage APIs for property media management.
        </p>

        {/* Property ID Input */}
        <div className="mb-6">
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Property ID
          </label>
          <input
            type="text"
            value={propertyId}
            onChange={(e) => setPropertyId(e.target.value)}
            className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="Enter property ID"
          />
        </div>

        {/* Media Statistics */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
          <div className="bg-blue-50 p-4 rounded-lg text-center">
            <div className="text-2xl font-bold text-blue-600">{images.length}</div>
            <div className="text-sm text-blue-800">Images</div>
          </div>
          <div className="bg-green-50 p-4 rounded-lg text-center">
            <div className="text-2xl font-bold text-green-600">{videos.length}</div>
            <div className="text-sm text-green-800">Videos</div>
          </div>
          <div className="bg-purple-50 p-4 rounded-lg text-center">
            <div className="text-2xl font-bold text-purple-600">{documents.length}</div>
            <div className="text-sm text-purple-800">Documents</div>
          </div>
          <div className="bg-gray-50 p-4 rounded-lg text-center">
            <div className="text-2xl font-bold text-gray-600">{totalMediaCount}</div>
            <div className="text-sm text-gray-800">Total Files</div>
          </div>
        </div>

        {/* File Upload Section */}
        <div className="bg-gray-50 p-6 rounded-lg mb-6">
          <h3 className="text-lg font-semibold mb-4">Upload Files to Azure Blob Storage</h3>
          
          <div className="mb-4">
            <input
              type="file"
              multiple
              onChange={(e) => setSelectedFiles(e.target.files)}
              className="w-full px-3 py-2 border border-gray-300 rounded-md"
              accept="image/*,video/*,.pdf,.doc,.docx"
            />
          </div>

          <div className="flex flex-wrap gap-3">
            <button
              onClick={handleImageUpload}
              disabled={uploading || !selectedFiles}
              className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {uploading ? 'Uploading...' : 'Upload as Images'}
            </button>
            
            <button
              onClick={handleVideoUpload}
              disabled={uploading || !selectedFiles}
              className="px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {uploading ? 'Uploading...' : 'Upload as Videos'}
            </button>
            
            <button
              onClick={handleDocumentUpload}
              disabled={uploading || !selectedFiles}
              className="px-4 py-2 bg-purple-600 text-white rounded-md hover:bg-purple-700 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {uploading ? 'Uploading...' : 'Upload as Documents'}
            </button>

            <button
              onClick={() => loadPropertyMedia(propertyId)}
              disabled={loading}
              className="px-4 py-2 bg-gray-600 text-white rounded-md hover:bg-gray-700 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {loading ? 'Loading...' : 'Refresh Media'}
            </button>
          </div>
        </div>

        {/* Error Display */}
        {error && (
          <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
            <div className="text-red-800 font-medium">Error:</div>
            <div className="text-red-700">{error}</div>
          </div>
        )}

        {/* API Integration Status */}
        <div className="bg-green-50 border border-green-200 rounded-lg p-4 mb-6">
          <h4 className="font-semibold text-green-900 mb-2">Azure Blob API Integration Status</h4>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
            <div className="flex items-center space-x-2">
              <span className="w-2 h-2 bg-green-500 rounded-full"></span>
              <span>POST /api/property-media/upload - Upload Media ✓</span>
            </div>
            <div className="flex items-center space-x-2">
              <span className="w-2 h-2 bg-green-500 rounded-full"></span>
              <span>GET /api/property-media/{'{property_id}'} - Get Media ✓</span>
            </div>
            <div className="flex items-center space-x-2">
              <span className="w-2 h-2 bg-green-500 rounded-full"></span>
              <span>GET /api/property-media/{'{media_id}'}/download - Download URL ✓</span>
            </div>
            <div className="flex items-center space-x-2">
              <span className="w-2 h-2 bg-green-500 rounded-full"></span>
              <span>PUT /api/property-media/{'{media_id}'} - Replace Media ✓</span>
            </div>
          </div>
        </div>
      </div>

      {/* Azure Blob Media Gallery */}
      <AzureBlobMediaGallery 
        propertyId={propertyId}
        editable={true}
        showUpload={true}
      />

      {/* Usage Instructions */}
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-blue-900 mb-4">How to Use Azure Blob Integration</h3>
        <div className="space-y-3 text-sm text-blue-800">
          <div>
            <strong>1. Upload Files:</strong> Select files and click the appropriate upload button. Files are automatically stored in Azure Blob Storage with secure URLs.
          </div>
          <div>
            <strong>2. View Media:</strong> All uploaded media is displayed in the gallery below with thumbnails and metadata.
          </div>
          <div>
            <strong>3. Download Files:</strong> Click the download button on any media item to generate a secure download URL.
          </div>
          <div>
            <strong>4. Replace Files:</strong> Use the replace button to update existing media files while maintaining the same ID.
          </div>
          <div>
            <strong>5. Delete Files:</strong> Remove files from both the database and Azure Blob Storage.
          </div>
        </div>
      </div>
    </div>
  );
};

export default AzureBlobExample;