import React, { useState } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { 
  Upload, 
  Image, 
  Video, 
  FileText, 
  Download, 
  Replace, 
  Trash2, 
  Eye, 
  X,
  Play,
  Pause,
  RotateCcw
} from 'lucide-react';
import { useAzureBlobMedia } from '../hooks/useAzureBlobMedia';
import { toast } from 'sonner';

interface AzureBlobMediaGalleryProps {
  propertyId: string;
  editable?: boolean;
  showUpload?: boolean;
  className?: string;
}

export const AzureBlobMediaGallery: React.FC<AzureBlobMediaGalleryProps> = ({
  propertyId,
  editable = false,
  showUpload = false,
  className = ''
}) => {
  const {
    images,
    videos,
    documents,
    loading,
    uploading,
    error,
    uploadImages,
    uploadVideos,
    uploadDocuments,
    getDownloadUrl,
    replaceMediaFile,
    deleteMediaFile,
    loadPropertyMedia
  } = useAzureBlobMedia(propertyId);

  const [activeTab, setActiveTab] = useState<'images' | 'videos'>('images');
  const [selectedMedia, setSelectedMedia] = useState<string | null>(null);
  const [isPreviewOpen, setIsPreviewOpen] = useState(false);
  const [selectedCategory, setSelectedCategory] = useState<'regular' | 'drone' | '3d'>('regular');

  // Handle file uploads
  const handleFileUpload = async (files: FileList | null, type: 'images' | 'videos' | 'documents') => {
    if (!files || files.length === 0) return;

    const fileArray = Array.from(files);
    
    try {
      switch (type) {
        case 'images':
          await uploadImages(propertyId, fileArray, selectedCategory);
          break;
        case 'videos':
          await uploadVideos(propertyId, fileArray, selectedCategory);
          break;

      }
    } catch (error) {
      console.error('Upload error:', error);
    }
  };

  // Handle media download
  const handleDownload = async (mediaId: string, fileName?: string) => {
    try {
      const result = await getDownloadUrl(mediaId);
      if (result.success && result.url) {
        // For videos, we need to fetch the blob and create a download link
        if (activeTab === 'videos') {
          const response = await fetch(result.url);
          
          if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
          }
          
          const blob = await response.blob();
          
          // Validate blob before creating object URL
          if (!blob || blob.size === 0) {
            throw new Error('Invalid or empty file received');
          }
          
          const url = window.URL.createObjectURL(blob);
          
          const link = document.createElement('a');
          link.href = url;
          link.download = fileName || `video_${mediaId}.mp4`;
          document.body.appendChild(link);
          link.click();
          document.body.removeChild(link);
          
          // Clean up the object URL
          window.URL.revokeObjectURL(url);
        } else {
          // For images, direct download should work
          const link = document.createElement('a');
          link.href = result.url;
          link.download = fileName || `media_${mediaId}`;
          link.target = '_blank';
          document.body.appendChild(link);
          link.click();
          document.body.removeChild(link);
        }
        
        toast.success('Download started');
      }
    } catch (error) {
      console.error('Download error:', error);
      toast.error('Download failed');
    }
  };

  // Handle media replacement
  const handleReplace = async (mediaId: string, type: 'image' | 'video' | 'document') => {
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = type === 'image' ? 'image/*' : type === 'video' ? 'video/*' : '*/*';
    
    input.onchange = async (e) => {
      const file = (e.target as HTMLInputElement).files?.[0];
      if (file) {
        await replaceMediaFile(mediaId, file, type);
      }
    };
    
    input.click();
  };

  // Handle media deletion
  const handleDelete = async (mediaId: string, type: 'image' | 'video' | 'document') => {
    if (window.confirm('Are you sure you want to delete this media file?')) {
      await deleteMediaFile(mediaId, type);
    }
  };

  // Get current media array based on active tab
  const getCurrentMedia = () => {
    switch (activeTab) {
      case 'images': return images;
      case 'videos': return videos;
      default: return [];
    }
  };

  const currentMedia = getCurrentMedia();

  if (loading) {
    return (
      <div className={`flex items-center justify-center p-8 ${className}`}>
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <span className="ml-2 text-gray-600">Loading media...</span>
      </div>
    );
  }

  return (
    <div className={`bg-white rounded-lg border border-gray-200 ${className}`}>
      {/* Header with tabs */}
      <div className="border-b border-gray-200 p-4">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold text-gray-900">Property Media</h3>
          {editable && (
            <button
              onClick={() => loadPropertyMedia(propertyId)}
              className="p-2 text-gray-500 hover:text-gray-700 transition-colors"
              title="Refresh media"
            >
              <RotateCcw className="w-4 h-4" />
            </button>
          )}
        </div>

        {/* Tab navigation */}
        <div className="flex space-x-1 bg-gray-100 p-1 rounded-lg">
          <button
            onClick={() => setActiveTab('images')}
            className={`flex items-center space-x-2 px-3 py-2 rounded-md text-sm font-medium transition-colors ${
              activeTab === 'images'
                ? 'bg-white text-blue-600 shadow-sm'
                : 'text-gray-600 hover:text-gray-900'
            }`}
          >
            <Image className="w-4 h-4" />
            <span>Images ({images.length})</span>
          </button>
          
          <button
            onClick={() => setActiveTab('videos')}
            className={`flex items-center space-x-2 px-3 py-2 rounded-md text-sm font-medium transition-colors ${
              activeTab === 'videos'
                ? 'bg-white text-blue-600 shadow-sm'
                : 'text-gray-600 hover:text-gray-900'
            }`}
          >
            <Video className="w-4 h-4" />
            <span>Videos ({videos.length})</span>
          </button>
          

        </div>
      </div>

      {/* Upload section */}
      {showUpload && (
        <div className="p-4 border-b border-gray-200">
          {/* Category Selection */}
          <div className="mb-4">
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Media Category
            </label>
            <select
              value={selectedCategory}
              onChange={(e) => setSelectedCategory(e.target.value as 'regular' | 'drone' | '3d')}
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            >
              <option value="regular">📷 Regular Photos/Videos</option>
              <option value="drone">🚁 Drone View (Aerial)</option>
              <option value="3d">🎮 3D View (Virtual Tour)</option>
            </select>
          </div>
          
          <div className="border-2 border-dashed border-gray-300 rounded-lg p-4 text-center hover:border-blue-400 transition-colors">
            <input
              type="file"
              multiple
              accept={
                activeTab === 'images' ? 'image/*' : 'video/*'
              }
              onChange={(e) => handleFileUpload(e.target.files, activeTab)}
              className="hidden"
              id={`upload-${activeTab}`}
              disabled={uploading}
            />
            <label htmlFor={`upload-${activeTab}`} className="cursor-pointer">
              {uploading ? (
                <div className="flex items-center justify-center">
                  <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600"></div>
                  <span className="ml-2 text-gray-600">Uploading...</span>
                </div>
              ) : (
                <>
                  <Upload className="w-8 h-8 mx-auto mb-2 text-gray-400" />
                  <div className="text-sm font-medium text-gray-900 mb-1">
                    Upload {activeTab} ({selectedCategory === 'regular' ? 'Regular' : selectedCategory === 'drone' ? 'Drone View' : '3D View'})
                  </div>
                  <div className="text-xs text-gray-500">
                    {activeTab === 'images' && 'JPG, PNG, WebP up to 5MB each'}
                    {activeTab === 'videos' && 'MP4, MOV, AVI up to 50MB each'}
                  </div>
                </>
              )}
            </label>
          </div>
        </div>
      )}

      {/* Media grid */}
      <div className="p-4">
        {error && (
          <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg text-red-700 text-sm">
            {error}
          </div>
        )}

        {currentMedia.length === 0 ? (
          <div className="text-center py-8">
            <div className="text-gray-400 mb-2">
              {activeTab === 'images' && <Image className="w-12 h-12 mx-auto" />}
              {activeTab === 'videos' && <Video className="w-12 h-12 mx-auto" />}
            </div>
            <p className="text-gray-500">No {activeTab} uploaded yet</p>
          </div>
        ) : (
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
            {currentMedia.map((media) => (
              <motion.div
                key={media.id}
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
                className="relative group bg-gray-50 rounded-lg overflow-hidden border border-gray-200 hover:border-gray-300 transition-colors"
              >
                {/* Media preview */}
                <div className="aspect-square relative">
                  {activeTab === 'images' && (
                    <img
                      src={media.url}
                      alt={media.file_name || 'Property image'}
                      className="w-full h-full object-cover"
                      onClick={() => {
                        setSelectedMedia(media.id);
                        setIsPreviewOpen(true);
                      }}
                    />
                  )}
                  
                  {activeTab === 'videos' && (
                    <div 
                      className="w-full h-full bg-gray-800 flex items-center justify-center cursor-pointer"
                      onClick={() => {
                        setSelectedMedia(media.id);
                        setIsPreviewOpen(true);
                      }}
                    >
                      {media.thumbnail_url ? (
                        <img
                          src={media.thumbnail_url}
                          alt={media.title || 'Video thumbnail'}
                          className="w-full h-full object-cover"
                        />
                      ) : (
                        <Play className="w-8 h-8 text-white" />
                      )}
                    </div>
                  )}
                  


                  {/* Primary badge for images */}
                  {activeTab === 'images' && media.is_primary && (
                    <div className="absolute top-2 left-2 bg-yellow-500 text-white text-xs px-2 py-1 rounded">
                      Primary
                    </div>
                  )}

                  {/* Action buttons overlay */}
                  <div className="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-50 transition-opacity flex items-center justify-center opacity-0 group-hover:opacity-100">
                    <div className="flex space-x-2">
                      <button
                        onClick={() => {
                          setSelectedMedia(media.id);
                          setIsPreviewOpen(true);
                        }}
                        className="p-2 bg-white rounded-full text-gray-700 hover:bg-gray-100 transition-colors"
                        title="Preview"
                      >
                        <Eye className="w-4 h-4" />
                      </button>
                      
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          handleDownload(media.id, media.file_name);
                        }}
                        className="p-2 bg-white rounded-full text-gray-700 hover:bg-gray-100 transition-colors"
                        title="Download"
                      >
                        <Download className="w-4 h-4" />
                      </button>
                      
                      {editable && (
                        <>
                          <button
                            onClick={() => handleReplace(media.id, activeTab === 'images' ? 'image' : 'video')}
                            className="p-2 bg-white rounded-full text-gray-700 hover:bg-gray-100 transition-colors"
                            title="Replace"
                          >
                            <Replace className="w-4 h-4" />
                          </button>
                          
                          <button
                            onClick={() => handleDelete(media.id, activeTab === 'images' ? 'image' : 'video')}
                            className="p-2 bg-white rounded-full text-red-600 hover:bg-red-50 transition-colors"
                            title="Delete"
                          >
                            <Trash2 className="w-4 h-4" />
                          </button>
                        </>
                      )}
                    </div>
                  </div>
                </div>

                {/* Media info */}
                <div className="p-2">
                  <p className="text-xs text-gray-600 truncate">
                    {media.file_name || `${activeTab.slice(0, -1)} ${media.id}`}
                  </p>
                  {media.file_size && (
                    <p className="text-xs text-gray-400">
                      {(media.file_size / 1024 / 1024).toFixed(1)} MB
                    </p>
                  )}
                </div>
              </motion.div>
            ))}
          </div>
        )}
      </div>

      {/* Preview Modal */}
      <AnimatePresence>
        {isPreviewOpen && selectedMedia && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            className="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-50 p-4"
            onClick={() => setIsPreviewOpen(false)}
          >
            <motion.div
              initial={{ scale: 0.9 }}
              animate={{ scale: 1 }}
              exit={{ scale: 0.9 }}
              className="relative max-w-4xl max-h-full"
              onClick={(e) => e.stopPropagation()}
            >
              <button
                onClick={() => setIsPreviewOpen(false)}
                className="absolute top-4 right-4 z-10 p-2 bg-black bg-opacity-50 text-white rounded-full hover:bg-opacity-75 transition-colors"
              >
                <X className="w-6 h-6" />
              </button>

              {(() => {
                const media = currentMedia.find(m => m.id === selectedMedia);
                if (!media) return null;

                if (activeTab === 'images') {
                  return (
                    <img
                      src={media.url}
                      alt={media.file_name || 'Property image'}
                      className="max-w-full max-h-full object-contain rounded-lg"
                    />
                  );
                }

                if (activeTab === 'videos') {
                  return (
                    <video
                      src={media.url}
                      controls
                      className="max-w-full max-h-full rounded-lg"
                      autoPlay
                    />
                  );
                }

                return (
                  <div className="bg-white p-8 rounded-lg">
                    <div className="text-center">
                      <FileText className="w-16 h-16 text-blue-600 mx-auto mb-4" />
                      <h3 className="text-lg font-semibold mb-2">{media.file_name}</h3>
                      <button
                        onClick={() => handleDownload(media.id, media.file_name)}
                        className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                      >
                        Download Document
                      </button>
                    </div>
                  </div>
                );
              })()}
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  );
};

export default AzureBlobMediaGallery;