import React, { useEffect, useRef } from 'react';
import { useBehaviorTracker, EventType } from '../utils/behaviorTracker';

interface BehaviorTrackerProps {
  children: React.ReactNode;
  trackClicks?: boolean;
  trackScrolls?: boolean;
  propertyId?: number;
}

export const BehaviorTracker: React.FC<BehaviorTrackerProps> = ({
  children,
  trackClicks = true,
  trackScrolls = true,
  propertyId
}) => {
  const { trackEvent } = useBehaviorTracker();
  const containerRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const container = containerRef.current;
    if (!container) return;

    const handleClick = (event: MouseEvent) => {
      if (!trackClicks) return;

      const target = event.target as HTMLElement;
      const clickData = {
        element_type: target.tagName.toLowerCase(),
        element_class: target.className,
        element_id: target.id,
        element_text: target.textContent?.slice(0, 100),
        x: event.clientX,
        y: event.clientY
      };

      let eventType = EventType.PROPERTY_CLICK;
      
      if (target.tagName === 'IMG' || target.closest('img')) {
        eventType = EventType.IMAGE_CLICK;
      } else if (target.closest('[data-contact]') || target.textContent?.toLowerCase().includes('contact')) {
        eventType = EventType.CONTACT_CLICK;
      } else if (target.closest('[data-save]') || target.textContent?.toLowerCase().includes('save')) {
        eventType = EventType.SAVE_PROPERTY;
      } else if (target.closest('[data-visit]') || target.textContent?.toLowerCase().includes('visit')) {
        eventType = EventType.SCHEDULE_VISIT;
      }

      trackEvent(eventType, clickData, propertyId);
    };

    const handleScroll = () => {
      if (!trackScrolls) return;

      const scrollData = {
        scroll_top: container.scrollTop,
        scroll_height: container.scrollHeight,
        client_height: container.clientHeight,
        scroll_percent: Math.round((container.scrollTop / (container.scrollHeight - container.clientHeight)) * 100)
      };

      trackEvent(EventType.SCROLL, scrollData, propertyId);
    };

    if (trackClicks) {
      container.addEventListener('click', handleClick);
    }

    if (trackScrolls) {
      container.addEventListener('scroll', handleScroll);
    }

    return () => {
      container.removeEventListener('click', handleClick);
      container.removeEventListener('scroll', handleScroll);
    };
  }, [trackClicks, trackScrolls, propertyId, trackEvent]);

  return (
    <div ref={containerRef} style={{ height: '100%', width: '100%' }}>
      {children}
    </div>
  );
};

export const withBehaviorTracking = <P extends object>(
  WrappedComponent: React.ComponentType<P>,
  options: {
    trackClicks?: boolean;
    trackScrolls?: boolean;
    getPropertyId?: (props: P) => number | undefined;
  } = {}
) => {
  return React.forwardRef<any, P>((props, ref) => {
    const propertyId = options.getPropertyId?.(props);

    return (
      <BehaviorTracker
        trackClicks={options.trackClicks}
        trackScrolls={options.trackScrolls}
        propertyId={propertyId}
      >
        <WrappedComponent {...props} ref={ref} />
      </BehaviorTracker>
    );
  });
};
