import React, { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import { api } from "../utils/api";
import { PropertyCard } from "./PropertyCard";
import { ScrollFadeIn, ScrollStagger, ScrollStaggerItem, ScrollScale } from './ui/ScrollAnimation';

interface UrgentSaleProperty {
  id: number;
  title: string;
  locality: string;
  city: string;
  sale_price: number;
  urgent_pricing?: number;
  urgent_sale_end_date?: string;
  urgent_sale_end_time?: string;
  auction_end_date?: string;
  auction_end_time?: string;
  starting_bid_price?: number;
  starting_bidding_price?: number;
  photo?: string;
  images?: string[];
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
}

export function BiddingProperties() {
  const navigate = useNavigate();
  const [urgentSaleProperties, setUrgentSaleProperties] = useState<UrgentSaleProperty[]>([]);
  const [biddingProperties, setBiddingProperties] = useState<UrgentSaleProperty[]>([]);
  const [clickedHearts, setClickedHearts] = useState<Set<number>>(new Set());

  useEffect(() => {
    const loadProperties = async () => {
      try {
        const [urgentSales, bidding] = await Promise.all([
          api.fetchUrgentSaleProperties(),
          api.fetchBiddingProperties()
        ]);
        
        const sortedUrgentSales = urgentSales
          .sort((a: any, b: any) => new Date(b.created_at || b.id).getTime() - new Date(a.created_at || a.id).getTime())
          .slice(0, 3)
          .map((property: any) => ({
            ...property,
            photo: property.primary_image || property.photo
          }));
        
        const sortedBidding = bidding
          .sort((a: any, b: any) => new Date(b.created_at || b.id).getTime() - new Date(a.created_at || a.id).getTime())
          .slice(0, 3)
          .map((property: any) => ({
            ...property,
            photo: property.primary_image || property.photo
          }));
        
        setUrgentSaleProperties(sortedUrgentSales);
        setBiddingProperties(sortedBidding);
      } catch (error) {
        console.error('Error loading properties:', error);
        setUrgentSaleProperties([]);
        setBiddingProperties([]);
      }
    };
    loadProperties();
  }, []);

  const formatPrice = (price: number | null | undefined) => {
    if (!price || price === null || price === undefined) return '₹0';
    const numPrice = Number(price);
    if (isNaN(numPrice)) return '₹0';
    if (numPrice >= 10000000) return `₹${(numPrice / 10000000).toFixed(1)} Cr`;
    if (numPrice >= 100000) return `₹${(numPrice / 100000).toFixed(0)} L`;
    return `₹${numPrice.toLocaleString()}`;
  };

  const handleFavorite = async (propertyId: number) => {
    setClickedHearts(prev => new Set(prev).add(propertyId));
    
    const token = localStorage.getItem('access_token');
    if (token) {
      try {
        await fetch('http://localhost:8000/api/add-to-wishlist/', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`
          },
          body: JSON.stringify({ property_id: propertyId })
        });
      } catch (error) {
        console.error('Error adding to wishlist:', error);
      }
    } else {
      localStorage.setItem('pending_wishlist', propertyId.toString());
      setTimeout(() => navigate('/login'), 300);
    }
  };

  return (
    <>
    {/* Urgent Sale Section with Light Red Background */}
    <div style={{ backgroundColor: '#fef2f2', width: '100%', paddingTop: '2.5rem', paddingBottom: '2.5rem' }}>
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <ScrollFadeIn>
          <div className="text-center mb-12">
            <div className="flex items-center justify-center gap-3 mb-3">
              <svg className="w-10 h-10 text-red-600" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clipRule="evenodd" />
              </svg>
              <h2 className="text-4xl font-bold text-gray-900">Urgent Sale Properties</h2>
            </div>
            <div className="w-24 h-1 bg-gradient-to-r from-red-400 via-red-600 to-red-400 mx-auto mb-4 rounded-full"></div>
            <p className="text-lg text-gray-600 max-w-3xl mx-auto">Don't miss out on properties available for immediate sale with attractive pricing and fast possession.</p>
          </div>
        </ScrollFadeIn>
        
        {urgentSaleProperties.length > 0 ? (
          <ScrollStagger className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 justify-items-center" staggerDelay={0.15}>
          {urgentSaleProperties.map((property) => (
            <ScrollStaggerItem key={property.id}>
              <PropertyCard
                key={property.id}
                id={property.id}
                title={property.title}
                location={`${property.locality}, ${property.city}`}
                beds={property.bedrooms || 'N/A'}
                baths={property.bathrooms || 'N/A'}
                area={property.built_up_area ? `${property.built_up_area} sq ft` : 'N/A'}
                price={formatPrice(property.urgent_pricing || property.sale_price)}
                imageUrl={property.primary_image || property.photo}
                onFavorite={handleFavorite}
                isFavorited={clickedHearts.has(property.id)}
                isUrgent={true}
                urgentSaleEndDate={property.urgent_sale_end_date}
                urgentSaleEndTime={property.urgent_sale_end_time}
              />
            </ScrollStaggerItem>
          ))}
          </ScrollStagger>
        ) : (
          <div className="w-full flex flex-col justify-center items-center py-16">
            <div className="text-center">
              <div className="text-6xl mb-4">🔍</div>
              <p className="text-gray-500 text-lg mb-2">No urgent sale properties available at the moment.</p>
              <p className="text-gray-400 text-sm mb-4">Properties with urgent sale listings will appear here.</p>
            </div>
          </div>
        )}
      </div>
    </div>

    <div style={{ backgroundColor: '#fefce8', width: '100%', paddingTop: '2.5rem', paddingBottom: '2.5rem' }}>
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
      {/* Live Auctions Section */}
      <div>
        <ScrollFadeIn delay={0.2}>
          <div className="text-center mb-12">
            <div className="flex items-center justify-center gap-3 mb-3">
              <svg className="w-10 h-10 text-green-600" fill="currentColor" viewBox="0 0 20 20">
                <path d="M2 11a1 1 0 011-1h2a1 1 0 011 1v5a1 1 0 01-1 1H3a1 1 0 01-1-1v-5zM8 7a1 1 0 011-1h2a1 1 0 011 1v9a1 1 0 01-1 1H9a1 1 0 01-1-1V7zM14 4a1 1 0 011-1h2a1 1 0 011 1v12a1 1 0 01-1 1h-2a1 1 0 01-1-1V4z" />
              </svg>
              <h2 className="text-4xl font-bold text-gray-900">Bid Live on Premium Properties</h2>
            </div>
            <div className="w-24 h-1 bg-gradient-to-r from-green-400 via-green-600 to-green-400 mx-auto mb-4 rounded-full"></div>
            <p className="text-lg text-gray-600 max-w-3xl mx-auto">Participate in transparent, real-time property auctions and secure your next investment with confidence.</p>
          </div>
        </ScrollFadeIn>
        
        {biddingProperties.length > 0 ? (
          <ScrollStagger className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 justify-items-center" staggerDelay={0.15}>
            {biddingProperties.map((property) => (
              <ScrollStaggerItem key={property.id}>
                <PropertyCard
                  key={property.id}
                  id={property.id}
                  title={property.title}
                  location={`${property.locality}, ${property.city}`}
                  beds={property.bedrooms || 'N/A'}
                  baths={property.bathrooms || 'N/A'}
                  area={property.built_up_area ? `${property.built_up_area} sq ft` : 'N/A'}
                  price={formatPrice(property.starting_bid_price || property.sale_price)}
                  imageUrl={property.primary_image || property.photo}
                  onFavorite={handleFavorite}
                  isFavorited={clickedHearts.has(property.id)}
                  isBidding={true}
                  auctionEndDate={property.auction_end_date}
                  auctionEndTime={property.auction_end_time}
                  isCompleted={property.auction_status === 'completed'}
                  winnerName={property.winner_name}
                  finalBidAmount={property.final_bid_amount}
                />
              </ScrollStaggerItem>
            ))}
          </ScrollStagger>
        ) : (
          <div className="w-full flex flex-col justify-center items-center py-16">
            <div className="text-center">
              <div className="text-6xl mb-4">🏆</div>
              <p className="text-gray-500 text-lg mb-2">No live auctions available at the moment.</p>
              <p className="text-gray-400 text-sm mb-4">Properties with bidding/auction listings will appear here.</p>
            </div>
          </div>
        )}
      </div>
      
      {/* View All Properties Button */}
      <ScrollScale delay={0.3}>
        <div className="w-full bg-gradient-to-r from-gray-50 to-blue-50 mt-20 py-12 mt-8">
          <div className="max-w-4xl mx-auto text-center px-4">
            <h3 className="text-2xl font-bold text-gray-900 mb-4">Explore More Properties</h3>
            <p className="text-gray-600 mb-8">Discover thousands of verified properties across India</p>
            <button
              onClick={() => navigate('/properties')}
              className="px-6 py-3 bg-blue-500 hover:bg-blue-700 text-white rounded-xl transition-all duration-300 font-bold text-lg shadow-lg hover:shadow-xl transform hover:scale-105 flex items-center gap-4 mx-auto border-2 border-blue-800"
            >
              View All Properties
            </button>
          </div>
        </div>
      </ScrollScale>
      </div>
    </div>
    </>
  );
}
