import { useState, useEffect } from "react";
import { useNavigate, useLocation } from "react-router-dom";
import { toast } from "sonner";
import {
  sendOTP,
  loginUser,
  sendEmailOTP,
  loginWithEmail,
  sendRegistrationOTP,
  verifyRegistrationOTP,
  loginWithPassword,
} from "../api/auth";
import { initializeActivityTracking } from "../utils/auth";
import { clearAuthData } from "../utils/sessionValidator";
import { OTPInput } from "./ui/OTPInput";
import { GoogleLoginButton } from "./GoogleLoginButton";
import { Captcha } from "./SimpleCaptcha";
import { Header } from "./Header";
import { LockoutMessage, LoginAttemptsWarning } from "./ui/LockoutMessage";

import "../styles/animated-auth.css";

export function BuyerAuth() {
  const location = useLocation();
  const [isActive, setIsActive] = useState(location.pathname === "/signup");
  const navigate = useNavigate();

  // Clear any stale authentication data on component mount
  useEffect(() => {
    clearAuthData();
  }, []);

  // Function to store user data consistently
  const storeUserData = (data: any, email?: string, phoneNumber?: string) => {
    // Store tokens
    localStorage.setItem("access_token", data.access_token);
    localStorage.setItem("refresh_token", data.refresh_token);
    localStorage.removeItem("profile_modal_shown_this_login");

    // Store user data with consistent structure
    const userData = {
      user_id: data.user_id,
      id: data.user_id, // For compatibility
      email: email || data.email || "",
      phone_number: phoneNumber || data.phone_number || "",
      is_new_user: data.is_new_user,
      profile_exists: data.profile_exists,
      first_name: data.first_name || "User",
    };

    localStorage.setItem("user", JSON.stringify(userData));
    localStorage.setItem("user_id", data.user_id);

    // Store email in consistent location
    if (email) {
      localStorage.setItem("login_email", email);
      localStorage.setItem("user_email", email);
    }

    console.log("✅ User data stored:", userData);
  };

  // Get return URL from query parameters
  const searchParams = new URLSearchParams(location.search);
  const returnUrl = searchParams.get("returnUrl");

  useEffect(() => {
    setIsActive(location.pathname === "/signup");
  }, [location.pathname]);

  // Common states
  const [authMode, setAuthMode] = useState<"phone" | "email" | "password">(
    "phone",
  );
  const [countryCode, setCountryCode] = useState("+91");
  const [phoneNumber, setPhoneNumber] = useState("");
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [otpCode, setOtpCode] = useState("");
  const [loading, setLoading] = useState(false);
  const [success, setSuccess] = useState(false);
  const [error, setError] = useState(false);
  const [otpSent, setOtpSent] = useState(false);
  const [retryTimer, setRetryTimer] = useState(0);
  const [step, setStep] = useState<"input" | "otp">("input");
  const [emailExists, setEmailExists] = useState(false);
  const [passwordValidation, setPasswordValidation] = useState({
    isValid: false,
    errors: [],
    strength: 0,
  });
  const [showPasswordHelp, setShowPasswordHelp] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const [captchaValid, setCaptchaValid] = useState(false);
  
  // Lockout states
  const [lockoutInfo, setLockoutInfo] = useState({
    isLocked: false,
    remainingMinutes: 0,
    unlockAt: null,
    remainingAttempts: 5
  });

  useEffect(() => {
    let interval: NodeJS.Timeout;
    if (retryTimer > 0) {
      interval = setInterval(() => {
        setRetryTimer((prev) => {
          if (prev <= 1) {
            setError(false);
            return 0;
          }
          return prev - 1;
        });
      }, 1000);
    }
    return () => clearInterval(interval);
  }, [retryTimer]);

  const validatePhone = (phone: string): boolean => /^\d{4,15}$/.test(phone);
  const validateEmail = (email: string): boolean => {
    const emailRegex = /^[a-zA-Z0-9._%+-]+@[a-zA-Z.-]+\.[a-zA-Z]{2,}$/;
    if (!emailRegex.test(email)) return false;

    const allowedDomains = [
      "gmail.com",
      "yahoo.com",
      "outlook.com",
      "hotmail.com",
      "live.com",
      "icloud.com",
      "proton.me",
      "protonmail.com",
      "zoho.com",
      "gmx.com",
      "mail.com",
      "yandex.com",
      "rediffmail.com",
      "aol.com",
      "msn.com",
      "mit.edu",
      "stanford.edu",
      "harvard.edu",
      "iitd.ac.in",
      "iith.ac.in",
      "vit.ac.in",
      "ox.ac.uk",
      "utoronto.ca",
      "alstonair.com",
      "sitams.in",
    ];
    const domain = email.split("@")[1]?.toLowerCase();
    return allowedDomains.includes(domain);
  };
  const validatePassword = (
    password: string,
    email?: string,
  ): { isValid: boolean; errors: string[]; strength: number } => {
    const errors: string[] = [];
    let strength = 0;

    // Length checks
    if (password.length < 8) errors.push("At least 8 characters required");
    if (password.length > 128) errors.push("Maximum 128 characters allowed");
    else if (password.length >= 8) strength += 1;

    // 🚫 NO SPACES ALLOWED
    if (/\s/.test(password)) {
      errors.push("Spaces are not allowed in the password");
    } else {
      strength += 1;
    }

    // Complexity checks
    if (!/[a-z]/.test(password))
      errors.push("At least one lowercase letter required");
    else strength += 1;

    if (!/[A-Z]/.test(password))
      errors.push("At least one uppercase letter required");
    else strength += 1;

    if (!/\d/.test(password)) errors.push("At least one number required");
    else strength += 1;

    if (!/[!@#$%^&*(),.?":{}|<>]/.test(password))
      errors.push("At least one special character required");
    else strength += 1;

    // Common passwords check
    const commonPasswords = [
      "password",
      "123456",
      "123456789",
      "qwerty",
      "abc123",
      "password123",
      "12345678",
      "admin",
      "letmein",
      "welcome",
    ];
    if (commonPasswords.includes(password.toLowerCase())) {
      errors.push("Password is too common");
      strength = 0;
    }

    // Email similarity check
    if (
      email &&
      password.toLowerCase().includes(email.split("@")[0].toLowerCase())
    ) {
      errors.push("Password cannot contain your email");
      strength = Math.max(0, strength - 1);
    }

    return { isValid: errors.length === 0, errors, strength };
  };

  const getPasswordStrengthText = (
    strength: number,
  ): { text: string; color: string } => {
    if (strength <= 1) return { text: "Very Weak", color: "#ef4444" };
    if (strength === 2) return { text: "Weak", color: "#f97316" };
    if (strength === 3) return { text: "Fair", color: "#eab308" };
    if (strength === 4) return { text: "Good", color: "#22c55e" };
    return { text: "Strong", color: "#16a34a" };
  };

  const handleSendOTP = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setError(false);
    setSuccess(false);

    try {
      if (authMode === "email") {
        if (!validateEmail(email)) {
          setError(true);
          setTimeout(() => {
            toast.error("Please enter a valid email address");
            setError(false);
          }, 500);
          setLoading(false);
          return;
        }

        if (isActive && authMode === "email") {
          // Registration flow with password
          if (!captchaValid) {
            setError(true);
            setTimeout(() => {
              toast.error("Please complete the CAPTCHA verification");
              setError(false);
            }, 500);
            setLoading(false);
            return;
          }
          const response = await sendRegistrationOTP(email);
          setStep("otp");
          toast.success("OTP sent to your email!");
          setRetryTimer(response?.retry_after || 60);
        } else {
          // Login flow - send OTP directly, backend will handle validation
          try {
            const response = await sendEmailOTP(email);
            setOtpSent(true);
            toast.success("OTP sent to your email!");
            setRetryTimer(response?.retry_after || 60);
          } catch (err: any) {
            if (
              err.message?.includes("not found") ||
              err.message?.includes("not registered") ||
              err.message?.includes("EMAIL_OTP_SEND_FAILED")
            ) {
              toast.error(
                "This email is not registered. Please create an account first.",
              );
            } else {
              throw err;
            }
          }
        }
      } else if (authMode === "phone") {
        if (!validatePhone(phoneNumber)) {
          setError(true);
          setTimeout(() => {
            toast.error("Please enter a valid phone number");
            setError(false);
          }, 500);
          return;
        }
        if (!captchaValid) {
          setError(true);
          setTimeout(() => {
            toast.error("Please complete the CAPTCHA verification");
            setError(false);
          }, 500);
          setLoading(false);
          return;
        }

        const fullPhoneNumber = countryCode + phoneNumber;
        await sendOTP(fullPhoneNumber);
        setOtpSent(true);
        toast.success("OTP sent successfully!");
      }
    } catch (err: any) {
      setError(true);
      setTimeout(() => {
        let errorMessage = "Failed to send OTP. Please try again.";
        if (err.message?.includes("EMAIL_OTP_SEND_FAILED")) {
          errorMessage =
            "Unable to send OTP to this email. Please check your email address and try again.";
        } else if (
          err.message?.includes("not found") ||
          err.message?.includes("not registered")
        ) {
          errorMessage =
            "This email is not registered. Please create an account first.";
        } else if (
          err.message?.includes("already exists") ||
          err.message?.includes("already registered") ||
          err.message?.includes("User with this email already exists")
        ) {
          errorMessage =
            "This email is already registered. Please login instead.";
        } else if (err.message && !err.message.includes("detail")) {
          errorMessage = err.message;
        }
        toast.error(errorMessage);
        setError(false);
      }, 500);
      if (err?.retry_after) {
        setRetryTimer(err.retry_after);
      }
    } finally {
      setLoading(false);
    }
  };

  const handlePasswordLogin = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setError(false);
    setSuccess(false);

    try {
      if (!validateEmail(email)) {
        setError(true);
        setTimeout(() => {
          toast.error("Please enter a valid email address");
          setError(false);
        }, 500);
        return;
      }

      if (!captchaValid) {
        setError(true);
        setTimeout(() => {
          toast.error("Please complete the CAPTCHA verification");
          setError(false);
        }, 500);
        setLoading(false);
        return;
      }

      const data = await loginWithPassword(email, password);

      // Store user data consistently
      storeUserData(data, email);
      localStorage.setItem("login_type", "password");

      initializeActivityTracking();
      toast.success("Login successful!");

      // Check if user is admin and redirect accordingly
      try {
        const payload = JSON.parse(atob(data.access_token.split(".")[1]));
        if (payload.is_admin === true) {
          navigate("/admin/dashboard", { replace: true });
          return;
        }
      } catch (e) {
        console.warn("Could not parse token for admin check:", e);
      }

      // Redirect to return URL if provided, otherwise to home
      if (returnUrl) {
        navigate(decodeURIComponent(returnUrl), { replace: true });
      } else {
        const shouldShowProfileModal =
          data.is_new_user ||
          (data.profile_completion_percentage || 0) < 100 ||
          !data.profile_exists;
        navigate("/", {
          replace: true,
          state: { showProfileModal: shouldShowProfileModal },
        });
      }
    } catch (err: any) {
      setError(true);
      setTimeout(() => {
        toast.error("Invalid email or password");
        setError(false);
      }, 500);
    } finally {
      setLoading(false);
    }
  };

  const handleVerifyOTP = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!otpCode || otpCode.length !== 6) {
      setError(true);
      setTimeout(() => {
        toast.error("Please enter a valid 6-digit OTP");
        setError(false);
      }, 500);
      return;
    }

    setLoading(true);
    setError(false);
    setSuccess(false);

    try {
      let data;

      if (isActive && authMode === "email" && step === "otp") {
        // Registration with password
        const passwordCheck = validatePassword(password, email);
        if (!passwordCheck.isValid) {
          setError(true);
          setTimeout(() => {
            toast.error(passwordCheck.errors[0] || "Invalid password");
            setError(false);
          }, 500);
          return;
        }

        if (password !== confirmPassword) {
          setError(true);
          setTimeout(() => {
            toast.error("Passwords do not match");
            setError(false);
          }, 500);
          return;
        }

        await verifyRegistrationOTP(email, otpCode, password);

        toast.success("Account created successfully! Please login.");
        setIsActive(false); // Switch to login form
        setStep('input');
        setOtpSent(false); // Reset OTP sent state
        setOtpCode('');
        setPassword('');
        setConfirmPassword('');
        setLoading(false);
        return;
      } else if (authMode === "email") {
        data = await loginWithEmail(email, otpCode);
        localStorage.setItem("login_type", "email");
        localStorage.setItem("user_email", email);
      } else if (isActive) {
        // Phone registration - redirect to login
        const fullPhoneNumber = countryCode + phoneNumber;
        await loginUser(fullPhoneNumber, otpCode); // This creates the account

        toast.success("Account created successfully! Please login.");
        setIsActive(false); // Switch to login form
        setOtpSent(false); // Reset OTP sent state
        setOtpCode('');
        setLoading(false);
        return;
      } else {
        const fullPhoneNumber = countryCode + phoneNumber;
        data = await loginUser(fullPhoneNumber, otpCode);
        localStorage.setItem("login_type", "phone");
      }

      // Only for login flows (not registration)
      localStorage.setItem("access_token", data.access_token);
      localStorage.setItem("refresh_token", data.refresh_token);
      localStorage.removeItem("profile_modal_shown_this_login");

      const userData = {
        user_id: data.user_id,
        email: authMode === "email" ? email : data.email,
        phone_number:
          authMode === "phone"
            ? countryCode + phoneNumber
            : data.phone_number || "",
        is_new_user: data.is_new_user,
        profile_exists: data.profile_exists,
      };
      localStorage.setItem("user", JSON.stringify(userData));
      localStorage.setItem("user_id", data.user_id);

      initializeActivityTracking();
      setSuccess(true);

      setTimeout(() => {
        toast.success("Login successful!");

        // Check if user is admin and redirect accordingly
        try {
          const payload = JSON.parse(atob(data.access_token.split(".")[1]));
          if (payload.is_admin === true) {
            navigate("/admin/dashboard", { replace: true });
            return;
          }
        } catch (e) {
          console.warn("Could not parse token for admin check:", e);
        }

        // Redirect to return URL if provided, otherwise to home
        if (returnUrl) {
          navigate(decodeURIComponent(returnUrl), { replace: true });
        } else {
          const shouldShowProfileModal =
            data.is_new_user ||
            (data.profile_completion_percentage || 0) < 100 ||
            !data.profile_exists;
          navigate("/", {
            replace: true,
            state: { showProfileModal: shouldShowProfileModal },
          });
        }
      }, 600);
    } catch (err: any) {
      setError(true);
      setOtpCode(''); // Clear OTP input on error
      setTimeout(() => {
        toast.error(err.message || "Invalid OTP");
        setError(false);
      }, 500);
    } finally {
      setLoading(false);
    }
  };

  return (
    <>
      <Header />
      <div
        className={`seller-auth-container ${isActive ? "active" : ""}`}
        data-type="buyer"
        style={{ marginTop: '64px' }}
      >
        <div className="seller-auth-wrapper">
        {/* Left side illustration */}
        <div className="illustration-side">
          <div className="character-illustration"></div>
        </div>

        {/* Right side form */}
        <div className="form-side">
          <button
            onClick={() => navigate('/')}
            style={{
              position: 'absolute',
              top: '20px',
              right: '20px',
              background: 'none',
              border: 'none',
              cursor: 'pointer',
              color: '#6b7280',
              fontSize: '24px',
              lineHeight: '1',
              padding: '8px',
              zIndex: 10,
              transition: 'color 0.2s'
            }}
            onMouseEnter={(e) => e.currentTarget.style.color = '#1f2937'}
            onMouseLeave={(e) => e.currentTarget.style.color = '#6b7280'}
            title="Close"
          >
            ×
          </button>
          <div className={`container ${isActive ? 'active' : ''}`}>
            {/* Login Form */}
            <div className="form-box Login">
              <h2>Welcome Back</h2>
              
              {/* Lockout Message */}
              <LockoutMessage
                isLocked={lockoutInfo.isLocked}
                remainingMinutes={lockoutInfo.remainingMinutes}
                unlockAt={lockoutInfo.unlockAt}
                onClose={() => setLockoutInfo(prev => ({ ...prev, isLocked: false }))}
              />
              
              {/* Login Attempts Warning */}
              {!lockoutInfo.isLocked && lockoutInfo.remainingAttempts < 5 && (
                <LoginAttemptsWarning
                  remainingAttempts={lockoutInfo.remainingAttempts}
                  maxAttempts={5}
                />
              )}

              {!otpSent && (
                <div className="input-box">
                  <div className="auth-mode-toggle">
                    <button
                      type="button"
                      onClick={() => setAuthMode("phone")}
                      className={authMode === "phone" ? "active" : ""}
                    >
                      Phone + OTP
                    </button>
                    <button
                      type="button"
                      onClick={() => setAuthMode("email")}
                      className={authMode === "email" ? "active" : ""}
                    >
                      Email + OTP
                    </button>
                    <button
                      type="button"
                      onClick={() => setAuthMode("password")}
                      className={authMode === "password" ? "active" : ""}
                    >
                      Email + Password
                    </button>
                  </div>
                </div>
              )}

              <form
                onSubmit={
                  authMode === "password"
                    ? handlePasswordLogin
                    : otpSent
                      ? handleVerifyOTP
                      : handleSendOTP
                }
              >
                {authMode === "password" ? (
                  <>
                    <div className="input-box">
                      <label>Email Address</label>
                      <input
                        type="email"
                        value={email}
                        onChange={(e) => setEmail(e.target.value)}
                        className={error ? "error" : success ? "success" : ""}
                        placeholder="Enter your email"
                        required
                      />
                    </div>
                    <div className="input-box">
                      <label>Password</label>
                      <div style={{ position: "relative" }}>
                        <input
                          type={showPassword ? "text" : "password"}
                          value={password}
                          onChange={(e) => setPassword(e.target.value)}
                          className={error ? "error" : success ? "success" : ""}
                          placeholder="Enter your password"
                          required
                        />
                        <button
                          type="button"
                          onClick={() => setShowPassword(!showPassword)}
                          style={{
                            position: "absolute",
                            right: "12px",
                            top: "50%",
                            transform: "translateY(-50%)",
                            background: "none",
                            border: "none",
                            cursor: "pointer",
                            color: "#6b7280",
                          }}
                        >
                          {showPassword ? (
                            <svg
                              width="20"
                              height="20"
                              viewBox="0 0 24 24"
                              fill="none"
                              stroke="currentColor"
                              strokeWidth="2"
                            >
                              <path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24" />
                              <line x1="1" y1="1" x2="23" y2="23" />
                            </svg>
                          ) : (
                            <svg
                              width="20"
                              height="20"
                              viewBox="0 0 24 24"
                              fill="none"
                              stroke="currentColor"
                              strokeWidth="2"
                            >
                              <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z" />
                              <circle cx="12" cy="12" r="3" />
                            </svg>
                          )}
                        </button>
                      </div>
                    </div>
                    <div className="input-box">
                      <Captcha key={authMode} onVerify={setCaptchaValid} />
                    </div>
                  </>
                ) : !otpSent ? (
                  <>
                    <div className="input-box">
                      <label>
                        {authMode === "email"
                          ? "Email Address"
                          : "Phone Number"}
                      </label>
                      {authMode === "email" ? (
                        <input
                          type="email"
                          value={email}
                          onChange={(e) => setEmail(e.target.value)}
                          className={error ? "error" : success ? "success" : ""}
                          placeholder="Enter your email"
                          required
                        />
                      ) : (
                        <input
                          type="tel"
                          value={`${countryCode}${phoneNumber}`}
                          onChange={(e) => {
                            const value = e.target.value;
                            if (value.startsWith("+91")) {
                              setPhoneNumber(
                                value.slice(3).replace(/[^0-9]/g, ""),
                              );
                            } else {
                              setPhoneNumber(value.replace(/[^0-9]/g, ""));
                            }
                          }}
                          className={error ? "error" : success ? "success" : ""}
                          placeholder="+91 98765 43210"
                          required
                        />
                      )}
                    </div>
                    <div className="input-box">
                      <Captcha key={authMode} onVerify={setCaptchaValid} />
                    </div>
                  </>
                ) : (
                  <>
                    <div className="input-box">
                      <label>Enter 6-Digit OTP</label>
                      <div className="otp-container">
                        <OTPInput
                          length={6}
                          value={otpCode}
                          onChange={setOtpCode}
                          disabled={loading}
                        />
                      </div>
                    </div>
                    <div style={{ textAlign: 'center', marginTop: '12px', marginBottom: '12px' }}>
                      {retryTimer > 0 ? (
                        <p style={{ fontSize: '14px', color: '#6b7280' }}>
                          Resend OTP in {Math.floor(retryTimer / 60)}:{(retryTimer % 60).toString().padStart(2, '0')}
                        </p>
                      ) : (
                        <button
                          type="button"
                          onClick={handleSendOTP}
                          disabled={loading}
                          style={{
                            background: 'none',
                            border: 'none',
                            color: '#3b82f6',
                            fontSize: '14px',
                            fontWeight: '600',
                            cursor: 'pointer',
                            textDecoration: 'underline'
                          }}
                        >
                          Resend OTP
                        </button>
                      )}
                      <span style={{ margin: '0 8px', color: '#6b7280' }}>|</span>
                      <button
                        type="button"
                        onClick={() => {
                          setOtpSent(false);
                          setOtpCode('');
                          setRetryTimer(0);
                        }}
                        disabled={loading}
                        style={{
                          background: 'none',
                          border: 'none',
                          color: '#6b7280',
                          fontSize: '14px',
                          fontWeight: '600',
                          cursor: 'pointer',
                          textDecoration: 'underline'
                        }}
                      >
                        Cancel
                      </button>
                    </div>
                  </>
                )}

                <div style={{ display: 'flex', gap: '12px', marginTop: '16px', alignItems: 'stretch' }}>
                  <button
                    type="button"
                    onClick={() => navigate('/')}
                    className="btn"
                    style={{
                      flex: 1,
                      padding: '12px',
                      border: '2px solid transparent',
                      boxSizing: 'border-box'
                    }}
                  >
                    Cancel
                  </button>
                  <button
                    className={`btn`}
                    type="submit"
                    disabled={loading}
                    style={{ 
                      flex: 1, 
                      boxSizing: 'border-box',
                      padding: '12px',
                      border: '2px solid transparent'
                    }}
                  >
                    {authMode === "password"
                      ? loading
                        ? "Signing in..."
                        : success
                          ? "Success!"
                          : "Sign In"
                      : loading
                        ? otpSent
                          ? "Verifying..."
                          : "Sending OTP..."
                        : success
                          ? "Success!"
                          : otpSent
                            ? "Login"
                            : "Send OTP"}
                  </button>
                </div>

                {otpSent && (
                  <button
                    className={`btn ${loading ? "loading" : ""} ${success ? "success" : ""}`}
                    type="submit"
                    disabled={loading}
                    style={{ marginTop: '16px' }}
                  >
                    {loading ? "Verifying..." : success ? "Success!" : "Login"}
                  </button>
                )}

                <div style={{ margin: "16px 0" }}>
                  <div
                    style={{
                      display: "flex",
                      alignItems: "center",
                      margin: "16px 0",
                    }}
                  >
                    <div
                      style={{
                        flex: 1,
                        height: "1px",
                        backgroundColor: "#e5e7eb",
                      }}
                    ></div>
                    <span
                      style={{
                        padding: "0 16px",
                        color: "#6b7280",
                        fontSize: "14px",
                      }}
                    >
                      or
                    </span>
                    <div
                      style={{
                        flex: 1,
                        height: "1px",
                        backgroundColor: "#e5e7eb",
                      }}
                    ></div>
                  </div>
                  <GoogleLoginButton />
                </div>

                <div className="regi-link">
                  <p>
                    Don't have an account?
                    <a
                      href="#"
                      onClick={(e) => {
                        e.preventDefault();
                        setIsActive(true);
                      }}
                    >
                      Create Account
                    </a>
                  </p>
                  <p style={{ marginTop: "8px" }}>
                    <a
                      href="#"
                      onClick={(e) => {
                        e.preventDefault();
                        navigate("/forgot-password");
                      }}
                    >
                      Forgot Password?
                    </a>
                  </p>
                </div>
              </form>
            </div>

            {/* Register Form */}
            <div className="form-box Register">
              <h2>Create Account</h2>

              {step === "input" && (
                <div className="input-box">
                  <div className="auth-mode-toggle">
                    <button
                      type="button"
                      onClick={() => setAuthMode("phone")}
                      className={authMode === "phone" ? "active" : ""}
                    >
                      Phone
                    </button>
                    <button
                      type="button"
                      onClick={() => setAuthMode("email")}
                      className={authMode === "email" ? "active" : ""}
                    >
                      Email + Password
                    </button>
                  </div>
                </div>
              )}

              <form onSubmit={step === "otp" ? handleVerifyOTP : handleSendOTP}>
                {step === "input" ? (
                  <>
                    <>
                      <div className="input-box">
                        <label>
                          {authMode === "email"
                            ? "Email Address"
                            : "Phone Number"}
                        </label>
                        {authMode === "email" ? (
                          <input
                            type="email"
                            value={email}
                            onChange={(e) => setEmail(e.target.value)}
                            className={
                              error ? "error" : success ? "success" : ""
                            }
                            placeholder="Enter your email"
                            required
                          />
                        ) : (
                          <input
                            type="tel"
                            value={`${countryCode}${phoneNumber}`}
                            onChange={(e) => {
                              const value = e.target.value;
                              if (value.startsWith("+91")) {
                                setPhoneNumber(
                                  value.slice(3).replace(/[^0-9]/g, ""),
                                );
                              } else {
                                setPhoneNumber(value.replace(/[^0-9]/g, ""));
                              }
                            }}
                            className={
                              error ? "error" : success ? "success" : ""
                            }
                            placeholder="+91 98765 43210"
                            required
                          />
                        )}
                      </div>
                    </>
                    <div className="input-box">
                      <Captcha key={authMode} onVerify={setCaptchaValid} />
                    </div>
                  </>
                ) : (
                  <>
                    <div className="input-box">
                      <label>Enter 6-Digit OTP</label>
                      <div className="otp-container">
                        <OTPInput
                          length={6}
                          value={otpCode}
                          onChange={setOtpCode}
                          disabled={loading}
                        />
                      </div>
                    </div>
                    <div style={{ textAlign: 'center', marginTop: '12px', marginBottom: '12px' }}>
                      {retryTimer > 0 ? (
                        <p style={{ fontSize: '14px', color: '#6b7280' }}>
                          Resend OTP in {Math.floor(retryTimer / 60)}:{(retryTimer % 60).toString().padStart(2, '0')}
                        </p>
                      ) : (
                        <button
                          type="button"
                          onClick={handleSendOTP}
                          disabled={loading}
                          style={{
                            background: 'none',
                            border: 'none',
                            color: '#3b82f6',
                            fontSize: '14px',
                            fontWeight: '600',
                            cursor: 'pointer',
                            textDecoration: 'underline'
                          }}
                        >
                          Resend OTP
                        </button>
                      )}
                      <span style={{ margin: '0 8px', color: '#6b7280' }}>|</span>
                      <button
                        type="button"
                        onClick={() => {
                          setStep('input');
                          setOtpCode('');
                          setPassword('');
                          setConfirmPassword('');
                          setRetryTimer(0);
                        }}
                        disabled={loading}
                        style={{
                          background: 'none',
                          border: 'none',
                          color: '#6b7280',
                          fontSize: '14px',
                          fontWeight: '600',
                          cursor: 'pointer',
                          textDecoration: 'underline'
                        }}
                      >
                        Cancel
                      </button>
                    </div>
                    {isActive && authMode === 'email' && (
                      <>
                        <div className="input-box">
                          <label>Password</label>
                          <div style={{ position: "relative" }}>
                            <input
                              type={showPassword ? "text" : "password"}
                              value={password}
                              onChange={(e) => {
                                setPassword(e.target.value);
                                const validation = validatePassword(
                                  e.target.value,
                                  email,
                                );
                                setPasswordValidation(validation);
                              }}
                              onFocus={() => setShowPasswordHelp(true)}
                              onBlur={() => setShowPasswordHelp(false)}
                              className={
                                error ? "error" : success ? "success" : ""
                              }
                              placeholder="Create password (min 8 chars)"
                              required
                            />
                            <button
                              type="button"
                              onClick={() => setShowPassword(!showPassword)}
                              style={{
                                position: "absolute",
                                right: "12px",
                                top: "50%",
                                transform: "translateY(-50%)",
                                background: "none",
                                border: "none",
                                cursor: "pointer",
                                color: "#6b7280",
                              }}
                            >
                              {showPassword ? (
                                <svg
                                  width="20"
                                  height="20"
                                  viewBox="0 0 24 24"
                                  fill="none"
                                  stroke="currentColor"
                                  strokeWidth="2"
                                >
                                  <path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24" />
                                  <line x1="1" y1="1" x2="23" y2="23" />
                                </svg>
                              ) : (
                                <svg
                                  width="20"
                                  height="20"
                                  viewBox="0 0 24 24"
                                  fill="none"
                                  stroke="currentColor"
                                  strokeWidth="2"
                                >
                                  <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z" />
                                  <circle cx="12" cy="12" r="3" />
                                </svg>
                              )}
                            </button>
                          </div>
                          {password && (
                            <div
                              className="password-strength"
                              style={{ marginTop: "8px" }}
                            >
                              <div
                                style={{
                                  display: "flex",
                                  alignItems: "center",
                                  gap: "8px",
                                }}
                              >
                                <div
                                  style={{
                                    flex: 1,
                                    height: "4px",
                                    backgroundColor: "#e5e7eb",
                                    borderRadius: "2px",
                                  }}
                                >
                                  <div
                                    style={{
                                      width: `${(passwordValidation.strength / 5) * 100}%`,
                                      height: "100%",
                                      backgroundColor: getPasswordStrengthText(
                                        passwordValidation.strength,
                                      ).color,
                                      borderRadius: "2px",
                                      transition: "all 0.3s ease",
                                    }}
                                  />
                                </div>
                                <span
                                  style={{
                                    fontSize: "12px",
                                    color: getPasswordStrengthText(
                                      passwordValidation.strength,
                                    ).color,
                                    fontWeight: "500",
                                  }}
                                >
                                  {
                                    getPasswordStrengthText(
                                      passwordValidation.strength,
                                    ).text
                                  }
                                </span>
                              </div>
                              {(showPasswordHelp ||
                                passwordValidation.errors.length > 0) && (
                                <div style={{ marginTop: "4px" }}>
                                  {passwordValidation.errors.map(
                                    (error, index) => (
                                      <div
                                        key={index}
                                        style={{
                                          fontSize: "11px",
                                          color: "#ef4444",
                                          marginTop: "2px",
                                        }}
                                      >
                                        • {error}
                                      </div>
                                    ),
                                  )}
                                </div>
                              )}
                            </div>
                          )}
                        </div>
                        <div className="input-box">
                          <label>Confirm Password</label>
                          <div style={{ position: "relative" }}>
                            <input
                              type={showConfirmPassword ? "text" : "password"}
                              value={confirmPassword}
                              onChange={(e) =>
                                setConfirmPassword(e.target.value)
                              }
                              className={
                                error ? "error" : success ? "success" : ""
                              }
                              placeholder="Confirm your password"
                              required
                            />
                            <button
                              type="button"
                              onClick={() =>
                                setShowConfirmPassword(!showConfirmPassword)
                              }
                              style={{
                                position: "absolute",
                                right: "12px",
                                top: "50%",
                                transform: "translateY(-50%)",
                                background: "none",
                                border: "none",
                                cursor: "pointer",
                                color: "#6b7280",
                              }}
                            >
                              {showConfirmPassword ? (
                                <svg
                                  width="20"
                                  height="20"
                                  viewBox="0 0 24 24"
                                  fill="none"
                                  stroke="currentColor"
                                  strokeWidth="2"
                                >
                                  <path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24" />
                                  <line x1="1" y1="1" x2="23" y2="23" />
                                </svg>
                              ) : (
                                <svg
                                  width="20"
                                  height="20"
                                  viewBox="0 0 24 24"
                                  fill="none"
                                  stroke="currentColor"
                                  strokeWidth="2"
                                >
                                  <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z" />
                                  <circle cx="12" cy="12" r="3" />
                                </svg>
                              )}
                            </button>
                          </div>
                        </div>
                      </>
                    )}
                  </>
                )}

                <div style={{ display: 'flex', gap: '12px', marginTop: '16px', alignItems: 'stretch' }}>
                  <button
                    type="button"
                    onClick={() => navigate('/')}
                    className="btn"
                    style={{
                      flex: 1,
                      padding: '12px',
                      border: '2px solid transparent',
                      boxSizing: 'border-box'
                    }}
                  >
                    Cancel
                  </button>
                  <button
                    className={`btn`}
                    type="submit"
                    disabled={loading}
                    style={{ 
                      flex: 1, 
                      boxSizing: 'border-box',
                      padding: '12px',
                      border: '2px solid transparent'
                    }}
                  >
                    {loading
                      ? step === "otp"
                        ? "Creating Account..."
                        : "Sending OTP..."
                      : success
                        ? "Success!"
                        : step === "otp"
                          ? "Create Account"
                          : "Send OTP"}
                  </button>
                </div>

                {step === "otp" && (
                  <button
                    className={`btn ${loading ? "loading" : ""} ${success ? "success" : ""}`}
                    type="submit"
                    disabled={loading}
                    style={{ marginTop: '16px' }}
                  >
                    {loading ? "Creating Account..." : success ? "Success!" : "Create Account"}
                  </button>
                )}

                <div style={{ margin: "16px 0" }}>
                  <div
                    style={{
                      display: "flex",
                      alignItems: "center",
                      margin: "16px 0",
                    }}
                  >
                    <div
                      style={{
                        flex: 1,
                        height: "1px",
                        backgroundColor: "#e5e7eb",
                      }}
                    ></div>
                    <span
                      style={{
                        padding: "0 16px",
                        color: "#6b7280",
                        fontSize: "14px",
                      }}
                    >
                      or
                    </span>
                    <div
                      style={{
                        flex: 1,
                        height: "1px",
                        backgroundColor: "#e5e7eb",
                      }}
                    ></div>
                  </div>
                  <GoogleLoginButton />
                </div>

                <div className="regi-link">
                  <p>
                    Already have an account?
                    <a
                      href="#"
                      onClick={(e) => {
                        e.preventDefault();
                        setIsActive(false);
                      }}
                    >
                      Sign In
                    </a>
                  </p>
                </div>
              </form>
            </div>
          </div>
        </div>
        </div>
      </div>
    </>
  );
}
