import React, { useState } from 'react';
import { ArrowLeft, Upload, Calendar, Target, Users, CreditCard, Shield, CheckCircle } from 'lucide-react';
import { useNavigate } from 'react-router-dom';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Textarea } from './ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { Checkbox } from './ui/checkbox';
import { RadioGroup, RadioGroupItem } from './ui/radio-group';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Separator } from './ui/separator';
import { createCampaign, CampaignData } from '../api/campaigns';
import { toast } from 'sonner';

interface CampaignFormData {
  // Campaign Basics
  campaignName: string;
  propertyId: string;
  objective: string;
  startDate: string;
  endDate: string;
  budgetType: 'daily' | 'total';
  budget: string;
  
  // Property Information
  propertyType: string;
  location: string;
  price: string;
  verifiedDocs: boolean;
  keyFeatures: string[];
  amenities: string[];
  
  // Media & Content
  images: File[];
  video?: File;
  virtualTour: string;
  description: string;
  usps: string[];
  callToAction: string;
  
  // Audience Targeting
  city: string;
  radius: string;
  ageRange: string;
  buyerType: string[];
  budgetRange: string;
  
  // Lead Handling
  leadChannels: string[];
  autoResponse: boolean;
  
  // Verification
  agentName: string;
  agentContact: string;
  kycVerified: boolean;
  reraId: string;
  companyName: string;
  
  // Scheduling
  adSchedule: 'fullDay' | 'custom';
  customHours: string;
  pauseConditions: string[];
  smartRecommendations: boolean;
  
  // Platforms
  platforms: string[];
  
  // Payment
  billingType: 'prepaid' | 'postpaid';
  gstDetails: string;
  paymentMethod: string;
  
  // Legal
  mediaPermission: boolean;
  accuracyConfirmation: boolean;
  retargetingConsent: boolean;
}

export function CampaignForm() {
  const navigate = useNavigate();
  const [currentStep, setCurrentStep] = useState(1);
  const [formData, setFormData] = useState<CampaignFormData>({
    campaignName: '',
    propertyId: '',
    objective: '',
    startDate: '',
    endDate: '',
    budgetType: 'daily',
    budget: '',
    propertyType: '',
    location: '',
    price: '',
    verifiedDocs: false,
    keyFeatures: [],
    amenities: [],
    images: [],
    virtualTour: '',
    description: '',
    usps: [],
    callToAction: '',
    city: '',
    radius: '',
    ageRange: '',
    buyerType: [],
    budgetRange: '',
    leadChannels: [],
    autoResponse: false,
    agentName: '',
    agentContact: '',
    kycVerified: false,
    reraId: '',
    companyName: '',
    adSchedule: 'fullDay',
    customHours: '',
    pauseConditions: [],
    smartRecommendations: false,
    platforms: [],
    billingType: 'prepaid',
    gstDetails: '',
    paymentMethod: '',
    mediaPermission: false,
    accuracyConfirmation: false,
    retargetingConsent: false
  });

  const totalSteps = 5;

  const updateFormData = (field: keyof CampaignFormData, value: any) => {
    setFormData(prev => ({ ...prev, [field]: value }));
  };

  const handleArrayUpdate = (field: keyof CampaignFormData, value: string, checked: boolean) => {
    setFormData(prev => ({
      ...prev,
      [field]: checked 
        ? [...(prev[field] as string[]), value]
        : (prev[field] as string[]).filter(item => item !== value)
    }));
  };

  const nextStep = () => setCurrentStep(prev => Math.min(prev + 1, totalSteps));
  const prevStep = () => setCurrentStep(prev => Math.max(prev - 1, 1));

  const openPostCampaignLinks = (campaign: CampaignData) => {
    const baseUrl = window.location.origin;
    const propertyUrl = `${baseUrl}/property/${encodeURIComponent(campaign.propertyId)}`;
    const shareText = encodeURIComponent(
      `${campaign.campaignName} - ${campaign.propertyDescription || 'Property campaign'}`
    );
    const shareUrl = encodeURIComponent(propertyUrl);

    const links = [
      'https://ads.google.com/',
      'https://www.facebook.com/business/ads',
      'https://advertising.amazon.com/',
      `https://www.facebook.com/sharer/sharer.php?u=${shareUrl}&quote=${shareText}`,
      `https://www.linkedin.com/sharing/share-offsite/?url=${shareUrl}`,
      `https://twitter.com/intent/tweet?text=${shareText}&url=${shareUrl}`,
      'https://www.instagram.com/',
      'https://www.youtube.com/',
    ];

    links.forEach((link) => window.open(link, '_blank', 'noopener,noreferrer'));
  };

  const renderProgressBar = () => (
    <div className="mb-4">
      <div className="flex items-center justify-between mb-3">
        <div>
          <h1 className="text-2xl font-bold bg-gradient-to-r from-gray-900 to-blue-900 bg-clip-text text-transparent">
            Create Campaign
          </h1>
          <p className="text-gray-600 mt-1">Launch your property marketing campaign in just {totalSteps} simple steps</p>
        </div>
        <div className="text-right">
          <div className="text-xl font-bold text-blue-600">{currentStep}</div>
          <div className="text-sm text-gray-500">of {totalSteps} steps</div>
        </div>
      </div>
      <div className="relative mb-3">
        <div className="flex justify-between items-center mb-2">
          <span className="text-sm text-gray-600">Step {currentStep} of {totalSteps}</span>
          <span className="text-sm font-semibold text-blue-600">{Math.round((currentStep / totalSteps) * 100)}% Complete</span>
        </div>
        <div className="w-full bg-gray-200 rounded-full h-2">
          <div 
            className="bg-gradient-to-r from-blue-500 to-blue-600 h-2 rounded-full transition-all duration-500" 
            style={{ width: `${(currentStep / totalSteps) * 100}%` }}
          />
        </div>
      </div>
    </div>
  );

  const renderStep = () => {
    switch (currentStep) {
      case 1:
        return (
          <Card className="border-0 shadow-none">
            <CardHeader className="pb-4">
              <CardTitle className="flex items-center gap-2 text-xl font-bold text-gray-900">
                <div className="p-1 bg-blue-100 rounded-lg">
                  <Target className="w-5 h-5 text-blue-600" />
                </div>
                Campaign Setup & Property Details
              </CardTitle>
              <p className="text-gray-600 mt-1">Configure your campaign basics and property information</p>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                <div className="space-y-2">
                  <Label htmlFor="campaignName" className="text-sm font-semibold text-gray-700">Campaign Name *</Label>
                  <Input
                    id="campaignName"
                    value={formData.campaignName}
                    onChange={(e) => updateFormData('campaignName', e.target.value)}
                    placeholder="e.g., Luxury Apartments - Bangalore"
                    className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="propertyId" className="text-sm font-semibold text-gray-700">Property ID / Link *</Label>
                  <Select value={formData.propertyId} onValueChange={(value) => updateFormData('propertyId', value)}>
                    <SelectTrigger className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg">
                      <SelectValue placeholder="Select from your listings" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="prop1">NAL001 - 3BHK Apartment, Whitefield</SelectItem>
                      <SelectItem value="prop2">NAL002 - Villa, Electronic City</SelectItem>
                      <SelectItem value="prop3">NAL003 - Plot, Sarjapur Road</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="space-y-4">
                <Label className="text-sm font-semibold text-gray-700">Campaign Objective *</Label>
                <RadioGroup 
                  value={formData.objective} 
                  onValueChange={(value) => updateFormData('objective', value)}
                  className="grid grid-cols-1 md:grid-cols-2 gap-4"
                >
                  <div className="flex items-center space-x-3 p-4 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                    <RadioGroupItem value="lead_generation" id="lead_generation" className="text-blue-600" />
                    <Label htmlFor="lead_generation" className="font-medium cursor-pointer">Lead Generation</Label>
                  </div>
                  <div className="flex items-center space-x-3 p-4 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                    <RadioGroupItem value="property_awareness" id="property_awareness" className="text-blue-600" />
                    <Label htmlFor="property_awareness" className="font-medium cursor-pointer">Property Awareness</Label>
                  </div>
                  <div className="flex items-center space-x-3 p-4 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                    <RadioGroupItem value="urgent_sale" id="urgent_sale" className="text-blue-600" />
                    <Label htmlFor="urgent_sale" className="font-medium cursor-pointer">Urgent Sale Boost</Label>
                  </div>
                  <div className="flex items-center space-x-3 p-4 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                    <RadioGroupItem value="featured_boost" id="featured_boost" className="text-blue-600" />
                    <Label htmlFor="featured_boost" className="font-medium cursor-pointer">Featured Property Boost</Label>
                  </div>
                  <div className="flex items-center space-x-3 p-4 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                    <RadioGroupItem value="open_house" id="open_house" className="text-blue-600" />
                    <Label htmlFor="open_house" className="font-medium cursor-pointer">Open House / Site Visit Promotion</Label>
                  </div>
                </RadioGroup>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                <div className="space-y-2">
                  <Label htmlFor="startDate" className="text-sm font-semibold text-gray-700">Start Date *</Label>
                  <Input
                    id="startDate"
                    type="date"
                    value={formData.startDate}
                    onChange={(e) => updateFormData('startDate', e.target.value)}
                    className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="endDate" className="text-sm font-semibold text-gray-700">End Date *</Label>
                  <Input
                    id="endDate"
                    type="date"
                    value={formData.endDate}
                    onChange={(e) => updateFormData('endDate', e.target.value)}
                    className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                  />
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                <div className="space-y-4">
                  <Label className="text-sm font-semibold text-gray-700">Budget Type *</Label>
                  <RadioGroup 
                    value={formData.budgetType} 
                    onValueChange={(value: 'daily' | 'total') => updateFormData('budgetType', value)}
                    className="space-y-3"
                  >
                    <div className="flex items-center space-x-3 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                      <RadioGroupItem value="daily" id="daily" className="text-blue-600" />
                      <Label htmlFor="daily" className="font-medium cursor-pointer">Daily Budget</Label>
                    </div>
                    <div className="flex items-center space-x-3 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                      <RadioGroupItem value="total" id="total" className="text-blue-600" />
                      <Label htmlFor="total" className="font-medium cursor-pointer">Total Budget</Label>
                    </div>
                  </RadioGroup>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="budget" className="text-sm font-semibold text-gray-700">Budget Amount (₹) *</Label>
                  <Input
                    id="budget"
                    type="number"
                    value={formData.budget}
                    onChange={(e) => updateFormData('budget', e.target.value)}
                    placeholder="Enter amount"
                    className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                  />
                </div>
              </div>

              <Separator className="my-8" />
              
              <div className="space-y-6">
                <h3 className="text-lg font-semibold text-gray-900 flex items-center gap-2">
                  <CheckCircle className="w-5 h-5 text-green-600" />
                  Property Information
                </h3>
                
                <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                  <div className="space-y-2">
                    <Label className="text-sm font-semibold text-gray-700">Property Type *</Label>
                    <Select value={formData.propertyType} onValueChange={(value) => updateFormData('propertyType', value)}>
                      <SelectTrigger className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg">
                        <SelectValue placeholder="Select property type" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="apartment">Apartment</SelectItem>
                        <SelectItem value="villa">Villa</SelectItem>
                        <SelectItem value="plot">Plot</SelectItem>
                        <SelectItem value="commercial">Commercial</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="location" className="text-sm font-semibold text-gray-700">Location *</Label>
                    <Input
                      id="location"
                      value={formData.location}
                      onChange={(e) => updateFormData('location', e.target.value)}
                      placeholder="e.g., Whitefield, Bangalore"
                      className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                    />
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                  <div className="space-y-2">
                    <Label htmlFor="price" className="text-sm font-semibold text-gray-700">Price / FMV (₹) *</Label>
                    <Input
                      id="price"
                      type="number"
                      value={formData.price}
                      onChange={(e) => updateFormData('price', e.target.value)}
                      placeholder="e.g., 8500000"
                      className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                    />
                  </div>
                  <div className="flex items-center space-x-2 mt-8">
                    <Checkbox
                      id="verifiedDocs"
                      checked={formData.verifiedDocs}
                      onCheckedChange={(checked) => updateFormData('verifiedDocs', checked)}
                    />
                    <Label htmlFor="verifiedDocs" className="font-medium">Verified Documents Status</Label>
                  </div>
                </div>

                <div className="space-y-4">
                  <Label className="text-sm font-semibold text-gray-700">Key Features & Highlights</Label>
                  <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
                    {['East-facing', 'Near Metro', 'Fully Furnished', 'Gated Community', 'Swimming Pool', 'Gym'].map((feature) => (
                      <div key={feature} className="flex items-center space-x-2 p-2 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                        <Checkbox
                          id={feature}
                          checked={formData.keyFeatures.includes(feature)}
                          onCheckedChange={(checked) => handleArrayUpdate('keyFeatures', feature, checked as boolean)}
                        />
                        <Label htmlFor={feature} className="text-sm cursor-pointer">{feature}</Label>
                      </div>
                    ))}
                  </div>
                </div>

                <div className="space-y-4">
                  <Label className="text-sm font-semibold text-gray-700">Amenities</Label>
                  <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
                    {['Parking', '24/7 Security', 'Power Backup', 'Lift', 'Garden', 'Club House'].map((amenity) => (
                      <div key={amenity} className="flex items-center space-x-2 p-2 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                        <Checkbox
                          id={amenity}
                          checked={formData.amenities.includes(amenity)}
                          onCheckedChange={(checked) => handleArrayUpdate('amenities', amenity, checked as boolean)}
                        />
                        <Label htmlFor={amenity} className="text-sm cursor-pointer">{amenity}</Label>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        );

      case 2:
        return (
          <Card className="border-0 shadow-none">
            <CardHeader className="pb-4">
              <CardTitle className="flex items-center gap-2 text-xl font-bold text-gray-900">
                <div className="p-1 bg-purple-100 rounded-lg">
                  <Upload className="w-5 h-5 text-purple-600" />
                </div>
                Ad Media & Content
              </CardTitle>
              <p className="text-gray-600 mt-1">Upload high-quality media and create compelling content</p>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-4">
                <Label className="text-sm font-semibold text-gray-700">Property Images (5-10 high-quality photos) *</Label>
                <div className="border-2 border-dashed border-gray-300 rounded-lg p-8 text-center hover:border-blue-400 transition-colors">
                  <Upload className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                  <p className="text-gray-600 mb-2">Drag & drop images or click to browse</p>
                  <Input type="file" multiple accept="image/*" className="max-w-xs mx-auto" />
                  <p className="text-xs text-gray-500 mt-2">JPG, PNG, WebP up to 5MB each</p>
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                <div className="space-y-2">
                  <Label className="text-sm font-semibold text-gray-700">Video (Optional)</Label>
                  <Input type="file" accept="video/*" className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg" />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="virtualTour" className="text-sm font-semibold text-gray-700">360° Walkthrough / AR Link</Label>
                  <Input
                    id="virtualTour"
                    value={formData.virtualTour}
                    onChange={(e) => updateFormData('virtualTour', e.target.value)}
                    placeholder="Enter virtual tour URL"
                    className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                  />
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="description" className="text-sm font-semibold text-gray-700">Property Description (50-150 words) *</Label>
                <Textarea
                  id="description"
                  value={formData.description}
                  onChange={(e) => updateFormData('description', e.target.value)}
                  placeholder="Describe your property's unique features..."
                  rows={4}
                  className="border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                />
              </div>

              <div className="space-y-4">
                <Label className="text-sm font-semibold text-gray-700">Unique Selling Points (USPs)</Label>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                  {['Prime Location', 'Ready to Move', 'Investment Opportunity', 'High ROI', 'Luxury Amenities', 'Vastu Compliant'].map((usp) => (
                    <div key={usp} className="flex items-center space-x-2 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                      <Checkbox
                        id={usp}
                        checked={formData.usps.includes(usp)}
                        onCheckedChange={(checked) => handleArrayUpdate('usps', usp, checked as boolean)}
                      />
                      <Label htmlFor={usp} className="text-sm cursor-pointer">{usp}</Label>
                    </div>
                  ))}
                </div>
              </div>

              <div className="space-y-4">
                <Label className="text-sm font-semibold text-gray-700">Call to Action Preference *</Label>
                <RadioGroup 
                  value={formData.callToAction} 
                  onValueChange={(value) => updateFormData('callToAction', value)}
                  className="grid grid-cols-1 md:grid-cols-2 gap-3"
                >
                  <div className="flex items-center space-x-3 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                    <RadioGroupItem value="call" id="call" className="text-blue-600" />
                    <Label htmlFor="call" className="font-medium cursor-pointer">Call Agent</Label>
                  </div>
                  <div className="flex items-center space-x-3 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                    <RadioGroupItem value="enquire" id="enquire" className="text-blue-600" />
                    <Label htmlFor="enquire" className="font-medium cursor-pointer">Enquire</Label>
                  </div>
                  <div className="flex items-center space-x-3 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                    <RadioGroupItem value="site_visit" id="site_visit" className="text-blue-600" />
                    <Label htmlFor="site_visit" className="font-medium cursor-pointer">Book Site Visit</Label>
                  </div>
                  <div className="flex items-center space-x-3 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                    <RadioGroupItem value="whatsapp" id="whatsapp" className="text-blue-600" />
                    <Label htmlFor="whatsapp" className="font-medium cursor-pointer">WhatsApp Chat</Label>
                  </div>
                </RadioGroup>
              </div>
            </CardContent>
          </Card>
        );

      case 3:
        return (
          <Card className="border-0 shadow-none">
            <CardHeader className="pb-4">
              <CardTitle className="flex items-center gap-2 text-xl font-bold text-gray-900">
                <div className="p-1 bg-orange-100 rounded-lg">
                  <Users className="w-5 h-5 text-orange-600" />
                </div>
                Audience Targeting
              </CardTitle>
              <p className="text-gray-600 mt-1">Define your ideal buyer demographics and preferences</p>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                <div className="space-y-2">
                  <Label htmlFor="city" className="text-sm font-semibold text-gray-700">City *</Label>
                  <Select value={formData.city} onValueChange={(value) => updateFormData('city', value)}>
                    <SelectTrigger className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg">
                      <SelectValue placeholder="Select city" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="bangalore">Bangalore</SelectItem>
                      <SelectItem value="mumbai">Mumbai</SelectItem>
                      <SelectItem value="delhi">Delhi NCR</SelectItem>
                      <SelectItem value="pune">Pune</SelectItem>
                      <SelectItem value="hyderabad">Hyderabad</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="radius" className="text-sm font-semibold text-gray-700">Locality Radius *</Label>
                  <Select value={formData.radius} onValueChange={(value) => updateFormData('radius', value)}>
                    <SelectTrigger className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg">
                      <SelectValue placeholder="Select radius" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="5km">5 km</SelectItem>
                      <SelectItem value="10km">10 km</SelectItem>
                      <SelectItem value="15km">15 km</SelectItem>
                      <SelectItem value="25km">25 km</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                <div className="space-y-2">
                  <Label htmlFor="ageRange" className="text-sm font-semibold text-gray-700">Age Range</Label>
                  <Select value={formData.ageRange} onValueChange={(value) => updateFormData('ageRange', value)}>
                    <SelectTrigger className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg">
                      <SelectValue placeholder="Select age range" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="25-35">25-35 years</SelectItem>
                      <SelectItem value="35-45">35-45 years</SelectItem>
                      <SelectItem value="45-55">45-55 years</SelectItem>
                      <SelectItem value="55+">55+ years</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="budgetRange" className="text-sm font-semibold text-gray-700">Budget Range</Label>
                  <Select value={formData.budgetRange} onValueChange={(value) => updateFormData('budgetRange', value)}>
                    <SelectTrigger className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg">
                      <SelectValue placeholder="Select budget range" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="50L-1Cr">₹50L - ₹1Cr</SelectItem>
                      <SelectItem value="1Cr-2Cr">₹1Cr - ₹2Cr</SelectItem>
                      <SelectItem value="2Cr-5Cr">₹2Cr - ₹5Cr</SelectItem>
                      <SelectItem value="5Cr+">₹5Cr+</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="space-y-4">
                <Label className="text-sm font-semibold text-gray-700">Buyer Intent</Label>
                <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
                  {['investor', 'end_user', 'nri'].map((type) => (
                    <div key={type} className="flex items-center space-x-2 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                      <Checkbox
                        id={type}
                        checked={formData.buyerType.includes(type)}
                        onCheckedChange={(checked) => handleArrayUpdate('buyerType', type, checked as boolean)}
                      />
                      <Label htmlFor={type} className="text-sm cursor-pointer">{type === 'end_user' ? 'End Users' : type === 'nri' ? 'NRI Buyers' : 'Investors'}</Label>
                    </div>
                  ))}
                </div>
              </div>
            </CardContent>
          </Card>
        );

      case 4:
        return (
          <Card className="border-0 shadow-none">
            <CardHeader className="pb-4">
              <CardTitle className="flex items-center gap-2 text-xl font-bold text-gray-900">
                <div className="p-1 bg-teal-100 rounded-lg">
                  <Calendar className="w-5 h-5 text-teal-600" />
                </div>
                Campaign Settings & Platforms
              </CardTitle>
              <p className="text-gray-600 mt-1">Configure targeting, scheduling, and platform preferences</p>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-6">
                <h3 className="text-lg font-semibold text-gray-900 flex items-center gap-2">
                  <Users className="w-5 h-5 text-orange-600" />
                  Lead Handling & Verification
                </h3>
                
                <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                  <div className="space-y-4">
                    <Label className="text-sm font-semibold text-gray-700">How do you want to receive leads? *</Label>
                    <div className="grid grid-cols-2 gap-3">
                      {['call', 'whatsapp', 'email'].map((channel) => (
                        <div key={channel} className="flex items-center space-x-2 p-2 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                          <Checkbox
                            id={channel}
                            checked={formData.leadChannels.includes(channel)}
                            onCheckedChange={(checked) => handleArrayUpdate('leadChannels', channel, checked as boolean)}
                          />
                          <Label htmlFor={channel} className="text-sm cursor-pointer">{channel === 'call' ? 'Call' : channel === 'whatsapp' ? 'WhatsApp' : 'Email'}</Label>
                        </div>
                      ))}
                    </div>
                  </div>
                  
                  <div className="space-y-4">
                    <Label className="text-sm font-semibold text-gray-700">Agent Details</Label>
                    <div className="space-y-3">
                      <Input
                        placeholder="Agent / Owner Name *"
                        value={formData.agentName}
                        onChange={(e) => updateFormData('agentName', e.target.value)}
                        className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                      />
                      <Input
                        placeholder="Agent Contact Number *"
                        value={formData.agentContact}
                        onChange={(e) => updateFormData('agentContact', e.target.value)}
                        className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                      />
                      <Input
                        placeholder="Company / Agency Name"
                        value={formData.companyName}
                        onChange={(e) => updateFormData('companyName', e.target.value)}
                        className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                      />
                      <Input
                        placeholder="RERA ID (if applicable)"
                        value={formData.reraId}
                        onChange={(e) => updateFormData('reraId', e.target.value)}
                        className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                      />
                    </div>
                  </div>
                </div>
              </div>

              <Separator />
              
              <div className="space-y-6">
                <h3 className="text-lg font-semibold text-gray-900 flex items-center gap-2">
                  <Target className="w-5 h-5 text-pink-600" />
                  Platforms & Scheduling
                </h3>
                
                <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                  <div className="space-y-4">
                    <Label className="text-sm font-semibold text-gray-700">Select Platforms *</Label>
                    <div className="space-y-3">
                      {[
                        'google_ads',
                        'meta_ads', 
                        'nal_featured'
                      ].map((platform) => (
                        <div key={platform} className="flex items-center space-x-2 p-2 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                          <Checkbox
                            id={platform}
                            checked={formData.platforms.includes(platform)}
                            onCheckedChange={(checked) => handleArrayUpdate('platforms', platform, checked as boolean)}
                          />
                          <Label htmlFor={platform} className="text-sm cursor-pointer">
                            {platform === 'google_ads' ? 'Google Ads' : 
                             platform === 'meta_ads' ? 'Facebook & Instagram' : 
                             'NAL Featured Property'}
                          </Label>
                        </div>
                      ))}
                    </div>
                  </div>
                  
                  <div className="space-y-4">
                    <Label className="text-sm font-semibold text-gray-700">Schedule & Automation</Label>
                    <RadioGroup 
                      value={formData.adSchedule} 
                      onValueChange={(value: 'fullDay' | 'custom') => updateFormData('adSchedule', value)}
                      className="space-y-3"
                    >
                      <div className="flex items-center space-x-3 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                        <RadioGroupItem value="fullDay" id="fullDay" className="text-blue-600" />
                        <Label htmlFor="fullDay" className="font-medium cursor-pointer">Full Day</Label>
                      </div>
                      <div className="flex items-center space-x-3 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                        <RadioGroupItem value="custom" id="custom" className="text-blue-600" />
                        <Label htmlFor="custom" className="font-medium cursor-pointer">Custom Time Slots</Label>
                      </div>
                    </RadioGroup>
                    
                    {formData.adSchedule === 'custom' && (
                      <Input
                        placeholder="e.g., 9:00 AM - 6:00 PM"
                        value={formData.customHours}
                        onChange={(e) => updateFormData('customHours', e.target.value)}
                        className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                      />
                    )}
                    
                    <div className="flex items-center space-x-2 p-3 border border-gray-200 rounded-lg">
                      <Checkbox
                        id="smartRecommendations"
                        checked={formData.smartRecommendations}
                        onCheckedChange={(checked) => updateFormData('smartRecommendations', checked)}
                      />
                      <Label htmlFor="smartRecommendations" className="text-sm cursor-pointer">Enable Smart Recommendations</Label>
                    </div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        );

      case 5:
        return (
          <Card className="border-0 shadow-none">
            <CardHeader className="pb-4">
              <CardTitle className="flex items-center gap-2 text-xl font-bold text-gray-900">
                <div className="p-1 bg-green-100 rounded-lg">
                  <CreditCard className="w-5 h-5 text-green-600" />
                </div>
                Payment & Final Review
              </CardTitle>
              <p className="text-gray-600 mt-1">Configure billing and review your campaign before launch</p>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-6">
                <h3 className="text-lg font-semibold text-gray-900 flex items-center gap-2">
                  <CreditCard className="w-5 h-5 text-yellow-600" />
                  Payment & Billing
                </h3>
                
                <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                  <div className="space-y-4">
                    <Label className="text-sm font-semibold text-gray-700">Billing Type *</Label>
                    <RadioGroup 
                      value={formData.billingType} 
                      onValueChange={(value: 'prepaid' | 'postpaid') => updateFormData('billingType', value)}
                      className="space-y-3"
                    >
                      <div className="flex items-center space-x-3 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                        <RadioGroupItem value="prepaid" id="prepaid" className="text-blue-600" />
                        <Label htmlFor="prepaid" className="font-medium cursor-pointer">Prepaid</Label>
                      </div>
                      <div className="flex items-center space-x-3 p-3 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors">
                        <RadioGroupItem value="postpaid" id="postpaid" className="text-blue-600" />
                        <Label htmlFor="postpaid" className="font-medium cursor-pointer">Postpaid</Label>
                      </div>
                    </RadioGroup>
                  </div>
                  
                  <div className="space-y-4">
                    <Label className="text-sm font-semibold text-gray-700">Payment Details</Label>
                    <div className="space-y-3">
                      <Input
                        placeholder="GST Details (Optional)"
                        value={formData.gstDetails}
                        onChange={(e) => updateFormData('gstDetails', e.target.value)}
                        className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg"
                      />
                      <Select value={formData.paymentMethod} onValueChange={(value) => updateFormData('paymentMethod', value)}>
                        <SelectTrigger className="h-12 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-lg">
                          <SelectValue placeholder="Select payment method *" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="upi">UPI</SelectItem>
                          <SelectItem value="card">Card</SelectItem>
                          <SelectItem value="net_banking">Net Banking</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                  </div>
                </div>
              </div>

              <Separator />
              
              <div className="space-y-6">
                <h3 className="text-lg font-semibold text-gray-900 flex items-center gap-2">
                  <CheckCircle className="w-5 h-5 text-red-600" />
                  Legal Consent & Campaign Summary
                </h3>
                
                <div className="space-y-4">
                  <div className="flex items-start space-x-3 p-4 border border-gray-200 rounded-lg">
                    <Checkbox
                      id="mediaPermission"
                      checked={formData.mediaPermission}
                      onCheckedChange={(checked) => updateFormData('mediaPermission', checked)}
                    />
                    <Label htmlFor="mediaPermission" className="text-sm cursor-pointer">
                      I give permission to use property media in advertisements across selected platforms
                    </Label>
                  </div>

                  <div className="flex items-start space-x-3 p-4 border border-gray-200 rounded-lg">
                    <Checkbox
                      id="accuracyConfirmation"
                      checked={formData.accuracyConfirmation}
                      onCheckedChange={(checked) => updateFormData('accuracyConfirmation', checked)}
                    />
                    <Label htmlFor="accuracyConfirmation" className="text-sm cursor-pointer">
                      I confirm that all property details provided are accurate and up-to-date
                    </Label>
                  </div>

                  <div className="flex items-start space-x-3 p-4 border border-gray-200 rounded-lg">
                    <Checkbox
                      id="retargetingConsent"
                      checked={formData.retargetingConsent}
                      onCheckedChange={(checked) => updateFormData('retargetingConsent', checked)}
                    />
                    <Label htmlFor="retargetingConsent" className="text-sm cursor-pointer">
                      I allow retargeting campaigns for interested buyers who viewed this property
                    </Label>
                  </div>
                </div>

                <div className="bg-gradient-to-r from-blue-50 to-indigo-50 p-6 rounded-xl border border-blue-200">
                  <h4 className="font-bold text-blue-900 mb-4 text-lg">Campaign Summary</h4>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm text-blue-800">
                    <div className="space-y-2">
                      <p><strong>Campaign:</strong> {formData.campaignName || 'Not specified'}</p>
                      <p><strong>Objective:</strong> {formData.objective || 'Not specified'}</p>
                      <p><strong>Property:</strong> {formData.propertyType} in {formData.location}</p>
                    </div>
                    <div className="space-y-2">
                      <p><strong>Budget:</strong> ₹{formData.budget} ({formData.budgetType})</p>
                      <p><strong>Duration:</strong> {formData.startDate} to {formData.endDate}</p>
                      <p><strong>Platforms:</strong> {formData.platforms.length} selected</p>
                    </div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        );

      default:
        return null;
    }
  };

  const handleSubmit = async () => {
    const requiredFields = ['campaignName', 'propertyId', 'objective', 'startDate', 'endDate', 'budget', 'propertyType', 'location', 'price', 'city', 'radius', 'callToAction', 'agentName', 'agentContact', 'description', 'billingType', 'paymentMethod'];
    const missingFields = requiredFields.filter(field => {
      const value = formData[field as keyof CampaignFormData];
      return !value || (typeof value === 'string' && value.trim().length === 0);
    });
    
    if (missingFields.length > 0) {
      toast.error(`Please fill in all required fields: ${missingFields.join(', ')}`);
      return;
    }

    if (!formData.mediaPermission || !formData.accuracyConfirmation) {
      toast.error('Please accept all legal terms and conditions');
      return;
    }

    // Additional validation for backend requirements
    if (!formData.agentName || formData.agentName.length < 3) {
      toast.error('Agent name must be at least 3 characters');
      return;
    }

    if (!formData.agentContact || formData.agentContact.length < 10) {
      toast.error('Agent contact must be at least 10 digits');
      return;
    }

    if (!formData.description || formData.description.split(' ').length < 10) {
      toast.error('Property description must be at least 10 words');
      return;
    }

    if (!formData.location || formData.location.trim().length === 0) {
      toast.error('Property location is required');
      return;
    }

    if (!formData.price || parseFloat(formData.price) <= 0) {
      toast.error('Property price must be a valid positive number');
      return;
    }

    if (!formData.radius || parseFloat(formData.radius.replace('km', '')) <= 0) {
      toast.error('Locality radius must be selected');
      return;
    }

    try {
      // Get user ID from different auth systems
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const sellerToken = localStorage.getItem('sellerToken');
      const agentToken = localStorage.getItem('agentToken');
      
      let userId = user.user_id || localStorage.getItem('user_id');
      
      // Check for seller authentication - get UUID from sellerId
      if (sellerToken && !userId) {
        userId = localStorage.getItem('sellerId');
      }
      
      // Check for agent authentication - get UUID from agentId
      if (agentToken && !userId) {
        userId = localStorage.getItem('agentId') || localStorage.getItem('user_id');
      }
      
      if (!userId) {
        toast.error('Please login to continue.');
        return;
      }

      // Map form data to backend schema with validation
      const campaignData: CampaignData = {
        user_id: userId,
        campaignName: formData.campaignName,
        campaignObjective: formData.objective,
        propertyId: formData.propertyId,
        startDate: formData.startDate,
        endDate: formData.endDate,
        budgetType: formData.budgetType,
        budgetAmount: parseFloat(formData.budget) || 1000,
        propertyType: formData.propertyType || 'apartment',
        propertyLocation: formData.location || 'Bangalore',
        propertyPrice: parseFloat(formData.price) || 5000000,
        propertyImages: formData.images.length > 0 ? formData.images.map(f => f.name) : ['default1.jpg', 'default2.jpg', 'default3.jpg', 'default4.jpg', 'default5.jpg'],
        propertyDescription: formData.description,
        primaryCTA: formData.callToAction || 'call',
        targetCity: formData.city || 'bangalore',
        localityRadius: parseFloat(formData.radius?.replace('km', '')) || 10,
        budgetRange: formData.budgetRange || 'platform_defined',
        buyerType: formData.buyerType.length > 0 ? formData.buyerType[0] : 'investor',
        leadChannels: formData.leadChannels.length > 0 ? formData.leadChannels : ['call'],
        agentName: formData.agentName,
        agentContact: formData.agentContact,
        platforms: formData.platforms.length > 0 ? formData.platforms : ['google_ads'],
        scheduleType: formData.adSchedule === 'fullDay' ? 'full_day' : 'custom',
        billingType: formData.billingType,
        paymentMethod: formData.paymentMethod || 'upi',
        mediaUsageConsent: formData.mediaPermission,
        dataAccuracyConsent: formData.accuracyConfirmation
      };

      const response = await createCampaign(campaignData);
      toast.success('Campaign created successfully!');
      console.log('Campaign created:', response);
      openPostCampaignLinks(campaignData);
      navigate('/campaign-management');
    } catch (error: any) {
      console.error('Error creating campaign:', error);
      console.error('Error response:', error.response?.data);
      let errorMessage = 'Failed to create campaign';
      
      if (error.response?.data?.detail) {
        if (Array.isArray(error.response.data.detail)) {
          // Handle validation errors array
          errorMessage = error.response.data.detail.map((err: any) => {
            if (typeof err === 'object' && err.msg) {
              return `${err.loc ? err.loc.join('.') + ': ' : ''}${err.msg}`;
            }
            return err;
          }).join(', ');
        } else if (typeof error.response.data.detail === 'string') {
          errorMessage = error.response.data.detail;
        } else {
          errorMessage = 'Validation error occurred';
        }
      } else if (error.message) {
        errorMessage = error.message;
      }
      
      toast.error(errorMessage);
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-gray-50 to-blue-50 py-12">
      <div className="max-w-5xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="mb-4">
          <Button
            variant="outline"
            onClick={() => navigate('/campaign-management')}
            className="flex items-center gap-2 bg-white shadow-sm hover:shadow-md transition-all duration-200"
          >
            <ArrowLeft className="w-4 h-4" />
            Back to Campaign Management
          </Button>
        </div>

        <div className="mb-4">
          {renderProgressBar()}
        </div>
        
        <div className="bg-white rounded-2xl shadow-xl border border-gray-100 overflow-hidden">
          <div className="p-12">
            {renderStep()}
          </div>

          <div className="px-8 py-6 bg-gradient-to-r from-gray-50 to-blue-50 border-t border-gray-100">
            <div className="flex justify-between items-center">
              <Button
                variant="outline"
                onClick={prevStep}
                disabled={currentStep === 1}
                className="px-6 py-3 font-medium bg-white hover:bg-gray-50 border-gray-300 disabled:opacity-50 disabled:cursor-not-allowed transition-all duration-200"
              >
                Previous
              </Button>
              
              {currentStep === totalSteps ? (
                <Button
                  onClick={handleSubmit}
                  disabled={!formData.mediaPermission || !formData.accuracyConfirmation}
                  className="px-8 py-3 bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white font-semibold rounded-lg shadow-lg hover:shadow-xl transform hover:scale-105 transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none"
                >
                  Create Campaign
                </Button>
              ) : (
                <Button
                  onClick={nextStep}
                  className="px-8 py-3 bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white font-semibold rounded-lg shadow-lg hover:shadow-xl transform hover:scale-105 transition-all duration-200"
                >
                  Next Step →
                </Button>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
