import React from 'react';
import { useNavigate } from 'react-router-dom';
import { Check, Eye } from 'lucide-react';
import { ScrollFadeIn, ScrollScale } from './ui/ScrollAnimation';
import { isAdmin } from '../utils/auth';

export function CampaignManagement() {
  const navigate = useNavigate();

  const getCurrentRole = (): 'buyer' | 'seller' | 'agent' | 'admin' | 'user' => {
    try {
      const accessToken = localStorage.getItem('access_token');
      if (accessToken) {
        const payload = JSON.parse(atob(accessToken.split('.')[1]));
        const roles: string[] = payload.roles || [];
        if (roles.includes('Admin')) return 'admin';
        if (roles.includes('Agent')) return 'agent';
        if (roles.includes('Seller')) return 'seller';
        if (roles.includes('Buyer')) return 'buyer';
      }
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const role = (user.user_role || user.role || 'user').toString().toLowerCase();
      if (role === 'admin') return 'admin';
      if (role === 'agent') return 'agent';
      if (role === 'seller') return 'seller';
      if (role === 'buyer') return 'buyer';
      return 'user';
    } catch {
      return 'user';
    }
  };

  const handleStartCampaign = () => {
    const accessToken = localStorage.getItem('access_token');
    const sellerToken = localStorage.getItem('sellerToken');
    const agentToken = localStorage.getItem('agentToken');
    const user = localStorage.getItem('user');
    
    const isLoggedIn = accessToken || sellerToken || agentToken || user;
    
    if (!isLoggedIn) {
      alert('Please login to continue');
      return;
    }
    
    // Check if user is seller or agent
    if (!agentToken && !sellerToken) {
      alert('Only sellers and agents can create campaigns');
      return;
    }
    
    navigate('/campaign-form');
  };

  const role = getCurrentRole();
  const hasSellerToken = Boolean(localStorage.getItem('sellerToken'));
  const hasAgentToken = Boolean(localStorage.getItem('agentToken'));
  const isSellerAgentAdmin =
    isAdmin() ||
    role === 'seller' ||
    role === 'agent' ||
    role === 'admin' ||
    hasSellerToken ||
    hasAgentToken;
  const primaryButtonLabel = isSellerAgentAdmin
    ? 'Start Your Campaign Today'
    : 'View Campaigns';
  const handlePrimaryAction = () => {
    if (!isSellerAgentAdmin) {
      navigate('/campaign-management');
      return;
    }
    handleStartCampaign();
  };

  return (
    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-20 bg-white">
      {/* Header Section */}
      <ScrollFadeIn>
        <div className="text-center mb-16">
          <h1 className="text-4xl md:text-5xl font-bold text-black mb-4">Campaign Management</h1>
          <p className="text-xl text-gray-800 max-w-3xl mx-auto">Create and manage powerful marketing campaigns to boost your property visibility and reach qualified buyers faster.</p>
        </div>
      </ScrollFadeIn>
      
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-start">
        {/* Left Content */}
        <ScrollFadeIn delay={0.2}>
          <div>
          <h2 className="text-7xl font-bold text-gray-900 mb-6">Boost Your Property Reach</h2>
          <h3 className="text-3xl font-semibold text-gray-700 mb-8">Launch Smart Campaigns That Sell Faster</h3>
          
          <p className="text-lg text-gray-600 mb-8 leading-relaxed">
            Transform your property marketing with our intelligent campaign system. Reach qualified buyers, maximize visibility, and close deals faster than ever before.
          </p>

          <div className="space-y-6 mb-8">
            <div className="flex items-start space-x-4">
              <div className="bg-green-100 p-2 rounded-full">
                <Check className="w-5 h-5 text-green-600" />
              </div>
              <div>
                <h4 className="font-semibold text-gray-900 mb-1">Targeted Reach</h4>
                <p className="text-gray-600">Connect with serious buyers</p>
              </div>
            </div>
            
            <div className="flex items-start space-x-4">
              <div className="bg-green-100 p-2 rounded-full">
                <Check className="w-5 h-5 text-green-600" />
              </div>
              <div>
                <h4 className="font-semibold text-gray-900 mb-1">Smart Analytics</h4>
                <p className="text-gray-600">Track performance in real-time</p>
              </div>
            </div>
            
            <div className="flex items-start space-x-4">
              <div className="bg-purple-100 p-2 rounded-full">
                <Check className="w-5 h-5 text-purple-600" />
              </div>
              <div>
                <h4 className="font-semibold text-gray-900 mb-1">Faster Sales</h4>
                <p className="text-gray-600">Reduce time on market</p>
              </div>
            </div>
            
            <div className="flex items-start space-x-4">
              <div className="bg-orange-100 p-2 rounded-full">
                <Check className="w-5 h-5 text-orange-600" />
              </div>
              <div>
                <h4 className="font-semibold text-gray-900 mb-1">Premium Exposure</h4>
                <p className="text-gray-600">Featured listings & highlights</p>
              </div>
            </div>
          </div>

          <div className="mt-10 mb-8 flex flex-col items-start gap-4">
            <button
              onClick={handlePrimaryAction}
              style={{ backgroundColor: '#2563eb' }}
              className="text-white px-8 py-3 rounded-xl font-semibold text-base hover:bg-blue-700 transition-all duration-300 shadow-lg hover:shadow-xl transform hover:scale-105 border-2 border-blue-600 min-h-[52px] inline-flex items-center justify-center"
            >
              {primaryButtonLabel}
            </button>
            <button
              onClick={() => navigate('/properties')}
              className="flex items-center justify-center space-x-2 text-gray-700 hover:text-blue-600 transition-colors px-6 py-3 border-2 border-gray-300 rounded-xl hover:bg-gray-50 hover:border-blue-300 font-semibold"
            >
              <Eye className="w-5 h-5" />
              <span>View Properties</span>
            </button>
          </div>
        </div>
        </ScrollFadeIn>

        {/* Right Image/Illustration */}
        <ScrollScale delay={0.3}>
          <div className="relative mt-8 lg:mt-0">
            <img 
              src="/Campaign poster (2).png" 
              alt="Campaign Management Dashboard" 
              className="w-full h-auto rounded-2xl shadow-2xl"
            />
          </div>
        </ScrollScale>
      </div>
    </div>
  );
}
