import React, { Suspense } from 'react';
import { Canvas } from '@react-three/fiber';
import { useGLTF, OrbitControls } from '@react-three/drei';
import * as THREE from 'three';

interface Character3DProps {
  isListening?: boolean;
  isSpeaking?: boolean;
  language?: string;
}

function Model({ isListening, isSpeaking, language }: Character3DProps) {
  const modelPath = language === 'kn' ? '/models/women.glb' : '/models/Default.glb';
  const { scene } = useGLTF(modelPath);
  const [mouthMeshes, setMouthMeshes] = React.useState<any[]>([]);
  
  // Use the model's embedded materials with enhanced lighting
  React.useEffect(() => {
    // Calculate bounding box and center the model
    const box = new (THREE as any).Box3().setFromObject(scene);
    const center = box.getCenter(new (THREE as any).Vector3());
    
    // Center the model at origin
    scene.position.x = -center.x;
    scene.position.y = -center.y;
    scene.position.z = -center.z;
    
    const meshesWithMouth: any[] = [];
    
    scene.traverse((child: any) => {
      if (child.isMesh) {
        if (child.material) {
          child.material.needsUpdate = true;
        }
        // Find mesh with morph targets for mouth animation
        if (child.morphTargetInfluences && child.morphTargetDictionary) {
          console.log('Found morph targets:', child.morphTargetDictionary);
          
          const dict = child.morphTargetDictionary;
          if (dict['mouthOpen'] !== undefined) {
            meshesWithMouth.push({
              mesh: child,
              mouthIndex: dict['mouthOpen']
            });
            console.log('Added mesh with mouthOpen at index', dict['mouthOpen']);
          }
        }
      }
    });
    
    setMouthMeshes(meshesWithMouth);
  }, [scene]);

  // Animate lip movement when speaking
  React.useEffect(() => {
    console.log('Animation effect:', { isSpeaking, meshCount: mouthMeshes.length });
    if (!isSpeaking || mouthMeshes.length === 0) return;
    
    console.log('Starting mouth animation on', mouthMeshes.length, 'meshes');
    let animationId: number;
    const animate = () => {
      const time = Date.now() * 0.015;
      const mouthOpen = Math.abs(Math.sin(time)) * 0.8;
      
      mouthMeshes.forEach(({ mesh, mouthIndex }) => {
        mesh.morphTargetInfluences[mouthIndex] = mouthOpen;
      });
      
      animationId = requestAnimationFrame(animate);
    };
    
    animate();
    return () => {
      console.log('Stopping mouth animation');
      cancelAnimationFrame(animationId);
      mouthMeshes.forEach(({ mesh, mouthIndex }) => {
        mesh.morphTargetInfluences[mouthIndex] = 0;
      });
    };
  }, [isSpeaking, mouthMeshes]);

  return (
    <primitive 
      object={scene} 
      scale={1}
      position={[0, 0, 0]}
      rotation={[0, 0, 0]}
    />
  );
}

function Fallback() {
  return (
    <mesh>
      <boxGeometry args={[1, 2, 1]} />
      <meshStandardMaterial color="orange" />
    </mesh>
  );
}

export const Character3D: React.FC<Character3DProps> = ({ isListening, isSpeaking, language }) => {
  return (
    <div style={{ width: '100%', height: '450px', backgroundColor: '#f0f0f0' }}>
      <Canvas key={language} camera={{ position: [0, 0, 2], fov: 50 }}>
        <Suspense fallback={<Fallback />}>
          <ambientLight intensity={1} />
          <directionalLight position={[5, 5, 5]} intensity={1} />
          <directionalLight position={[-5, 5, 5]} intensity={0.5} />
          <Model isListening={isListening} isSpeaking={isSpeaking} language={language} />
          <OrbitControls 
            enableZoom={true}
            enablePan={false}
            target={[0, 0, 0]}
            enableRotate={true}
          />
        </Suspense>
      </Canvas>
    </div>
  );
};
