import React, { useState, useEffect, useRef } from 'react';
import { RefreshCw } from 'lucide-react';

interface CombinedCaptchaProps {
  onVerify: (isValid: boolean) => void;
}

export function CombinedCaptcha({ onVerify }: CombinedCaptchaProps) {
  const [captchaText, setCaptchaText] = useState('');
  const [userInput, setUserInput] = useState('');
  const [recaptchaToken, setRecaptchaToken] = useState('');
  const [simpleCaptchaValid, setSimpleCaptchaValid] = useState(false);
  const [recaptchaLoaded, setRecaptchaLoaded] = useState(false);
  const containerRef = useRef<HTMLDivElement>(null);
  const recaptchaId = useRef(`recaptcha-${Math.random().toString(36).substr(2, 9)}`);

  const generateCaptcha = () => {
    const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*';
    let result = '';
    for (let i = 0; i < 6; i++) {
      result += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    setCaptchaText(result);
    setUserInput('');
    setSimpleCaptchaValid(false);
  };

  useEffect(() => {
    generateCaptcha();
  }, []);

  useEffect(() => {
    const isSimpleValid = userInput === captchaText;
    setSimpleCaptchaValid(isSimpleValid);
    onVerify(isSimpleValid && !!recaptchaToken);
  }, [userInput, captchaText, recaptchaToken, onVerify]);

  const loadRecaptcha = () => {
    console.log('Loading reCAPTCHA...');
    if (window.grecaptcha && window.grecaptcha.render && containerRef.current) {
      if (containerRef.current.innerHTML === '') {
        try {
          console.log('Rendering reCAPTCHA with sitekey:', import.meta.env.VITE_RECAPTCHA_SITE_KEY || '6LeIxAcTAAAAAJcZVRqyHh71UMIEGNQ_MXjiZKhI');
          window.grecaptcha.render(containerRef.current, {
            sitekey: import.meta.env.VITE_RECAPTCHA_SITE_KEY || '6LeIxAcTAAAAAJcZVRqyHh71UMIEGNQ_MXjiZKhI',
            callback: (token: string) => {
              console.log('reCAPTCHA token received');
              setRecaptchaToken(token);
            },
            'expired-callback': () => {
              console.log('reCAPTCHA expired');
              setRecaptchaToken('');
            },
            'error-callback': () => {
              console.log('reCAPTCHA error');
              setRecaptchaToken('');
            }
          });
        } catch (error) {
          console.warn('reCAPTCHA render error:', error);
        }
      } else {
        console.log('reCAPTCHA container already has content');
      }
    } else {
      console.log('reCAPTCHA not ready:', { grecaptcha: !!window.grecaptcha, render: !!(window.grecaptcha && window.grecaptcha.render), container: !!containerRef.current });
    }
  };

  useEffect(() => {
    const loadScript = () => {
      if (window.grecaptcha) {
        setRecaptchaLoaded(true);
        setTimeout(loadRecaptcha, 100);
        return;
      }
      
      if (!document.querySelector('script[src*="recaptcha"]')) {
        const script = document.createElement('script');
        script.src = 'https://www.google.com/recaptcha/api.js?render=explicit';
        script.async = true;
        script.onload = () => {
          setRecaptchaLoaded(true);
          setTimeout(loadRecaptcha, 100);
        };
        document.head.appendChild(script);
      }
    };
    
    loadScript();
  }, []);

  useEffect(() => {
    if (recaptchaLoaded) {
      setTimeout(loadRecaptcha, 100);
    }
  }, [recaptchaLoaded]);

  return (
    <div>
      <label style={{ display: 'block', fontSize: '14px', color: '#4a5568', fontWeight: '600', marginBottom: '8px' }}>
        Security Verification
      </label>
      
      <div style={{ display: 'flex', gap: '8px', marginBottom: '8px' }}>
        <div style={{
          flex: 1,
          backgroundColor: '#f8fafc',
          border: '2px dashed #e2e8f0',
          borderRadius: '12px',
          padding: '12px',
          textAlign: 'center',
          fontFamily: 'monospace',
          fontSize: '18px',
          fontWeight: 'bold',
          color: '#1a202c',
          userSelect: 'none',
          letterSpacing: '4px'
        }}>
          {captchaText}
        </div>
        <button
          type="button"
          onClick={generateCaptcha}
          style={{
            padding: '12px',
            border: '2px solid #e2e8f0',
            borderRadius: '12px',
            backgroundColor: '#ffffff',
            cursor: 'pointer',
            color: '#4a5568',
            transition: 'all 0.2s',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center'
          }}
          title="Refresh captcha"
        >
          <RefreshCw size={16} />
        </button>
      </div>
      
      <input
        type="text"
        value={userInput}
        onChange={(e) => setUserInput(e.target.value)}
        style={{
          width: '100%',
          height: '56px',
          backgroundColor: '#f8fafc',
          border: `2px solid ${simpleCaptchaValid ? '#22c55e' : '#e2e8f0'}`,
          borderRadius: '12px',
          outline: 'none',
          fontSize: '16px',
          color: '#1a202c',
          fontWeight: '500',
          padding: '0 16px',
          letterSpacing: '2px',
          textAlign: 'center',
          marginBottom: '12px'
        }}
        placeholder="Enter the code above"
        maxLength={6}
        required
      />
      
      <div
        ref={containerRef}
        id={recaptchaId.current}
        style={{
          display: 'flex',
          justifyContent: 'center',
          alignItems: 'center',
          padding: '12px',
          border: '2px solid #e2e8f0',
          borderRadius: '12px',
          backgroundColor: '#f8fafc',
          minHeight: '78px',
          visibility: 'visible',
          opacity: 1,
          zIndex: 1000,
          overflow: 'visible',
          position: 'relative'
        }}
      />
    </div>
  );
}