import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { createInquiry, InquiryRequest } from '../api/inquiries';
import { toast } from 'sonner';

interface Property {
  id: number;
  title: string;
  locality: string;
  city: string;
  state: string;
  sale_price?: string;
  monthly_rent?: string;
  listing_type?: 'sell' | 'rent' | 'urgent-sale' | 'bidding';
  photo: string;
  email?: string;
  owner_name?: string;
  mobile?: string;
  created_at?: string;
  built_up_area?: number;
  carpet_area?: number;
  listed_by_id?: string;
  user_id?: string;
}

export function ContactAgentForm() {
  const { id } = useParams();
  const navigate = useNavigate();
  const [property, setProperty] = useState<Property | null>(null);
  const [ownerDetails, setOwnerDetails] = useState<{name: string, email: string, phone: string} | null>(null);
  const [loading, setLoading] = useState(true);
  const [submitting, setSubmitting] = useState(false);
  const [hasInquired, setHasInquired] = useState(false);
  const [formData, setFormData] = useState<InquiryRequest>({
    property_id: parseInt(id || '0'),
    reason_to_buy: '',
    buying_timeline: 'Within 3 months',
    interested_in_loan: false,
    interested_in_visit: false,
    buyer_name: '',
    buyer_email: '',
    buyer_phone: '',
    // Rental specific fields
    preferred_move_in: '',
    rental_duration: '',
    enquiry_for: 'SALE' // Will be updated based on property type
  });

  useEffect(() => {
    const fetchProperty = async () => {
      try {
        const { api } = await import('../utils/api');
        const data = await api.fetchPropertyById(id!);
        console.log('🏠 Property data:', data);
        if (data) {
          setProperty(data);
          
          // Update enquiry type based on property listing_type
          const isRental = data.listing_type === 'rent';
          setFormData(prev => ({
            ...prev,
            enquiry_for: isRental ? 'RENTAL' : 'SALE'
          }));
        }
        
        // Check if user has already inquired and fetch owner details
        const user = JSON.parse(localStorage.getItem('user') || '{}');
        console.log('🔍 User data:', user);
        console.log('🔍 Property ID:', id);
        
        if (user.user_id) {
          try {
            // Use the correct inquiry API to check existing inquiries
            const { checkExistingInquiry } = await import('../api/inquiries');
            const hasExistingInquiry = await checkExistingInquiry(user.user_id, parseInt(id || '0'));
            
            console.log('🔍 Existing inquiry check:', hasExistingInquiry);
            
            if (hasExistingInquiry) {
              console.log('🔍 Found existing inquiry - fetching owner profile');
              setHasInquired(true);
              
              // Fetch owner profile using property's listed_by_id
              if (data && (data.listed_by_id || data.user_id)) {
                try {
                  const ownerId = data.listed_by_id || data.user_id;
                  const token = localStorage.getItem('access_token');
                  
                  const profileResponse = await fetch(`/api/profile/${ownerId}`, {
                    headers: {
                      'Authorization': `Bearer ${token}`,
                      'Content-Type': 'application/json'
                    }
                  });
                  
                  if (profileResponse.ok) {
                    const ownerProfile = await profileResponse.json();
                    console.log('🔍 Owner profile data:', ownerProfile);
                    
                    setOwnerDetails({
                      name: `${ownerProfile.first_name || ''} ${ownerProfile.last_name || ''}`.trim() || 'Property Owner',
                      email: ownerProfile.email || '',
                      phone: ownerProfile.phone_number || ''
                    });
                  } else {
                    // Fallback to property data
                    setOwnerDetails({
                      name: data.owner_name || 'Property Owner',
                      email: data.email || '',
                      phone: data.mobile || ''
                    });
                  }
                } catch (error) {
                  console.error('🔍 Error fetching owner profile:', error);
                  // Fallback to property data
                  setOwnerDetails({
                    name: data.owner_name || 'Property Owner',
                    email: data.email || '',
                    phone: data.mobile || ''
                  });
                }
              }
            } else {
              console.log('🔍 No existing inquiry found for property ID:', id);
            }
          } catch (error) {
            console.error('🔍 Error checking existing inquiries:', error);
          }
        } else {
          console.log('🔍 No user_id found in localStorage');
        }
        
        // Only set hasInquired to true if we actually found an existing inquiry from API
        // Don't rely on localStorage for this check
      } catch (error) {
        console.error('Error fetching property:', error);
      } finally {
        setLoading(false);
      }
    };
    fetchProperty();
  }, [id]);

  useEffect(() => {
    const fetchUserProfile = async () => {
      try {
        const token = localStorage.getItem('access_token');
        const response = await fetch(`http://localhost:8090/api/profile/me`, {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        });
        
        if (response.ok) {
          const profileData = await response.json();
          console.log('📋 Profile data for form:', profileData);
          
          // Try multiple name field combinations
          let fullName = '';
          if (profileData.first_name || profileData.last_name) {
            fullName = `${profileData.first_name || ''} ${profileData.last_name || ''}`.trim();
          } else if (profileData.name) {
            fullName = profileData.name;
          } else if (profileData.full_name) {
            fullName = profileData.full_name;
          }
          
          setFormData(prev => ({
            ...prev,
            buyer_name: fullName || 'User',
            buyer_email: profileData.email || '',
            buyer_phone: profileData.phone_number || profileData.phone || ''
          }));
        }
      } catch (error) {
        console.error('Error fetching user profile:', error);
      }
    };
    
    fetchUserProfile();
  }, []);

  const formatPrice = (price: string | number) => {
    const numPrice = typeof price === 'string' ? parseFloat(price) : price;
    if (isNaN(numPrice) || numPrice === 0) return 'Price on request';
    if (numPrice >= 10000000) return `₹${(numPrice / 10000000).toFixed(1)} Cr`;
    if (numPrice >= 100000) return `₹${(numPrice / 100000).toFixed(0)} L`;
    return `₹${numPrice.toLocaleString()}`;
  };

  const maskPhone = (phone: string) => {
    if (!phone) return '+91 98***543**';
    return phone.replace(/(\d{2})(\d{3})(\d{3})(\d{2})/, '$1***$3**');
  };

  const maskEmail = (email: string) => {
    if (!email) return '*****@*****.com';
    const [username, domain] = email.split('@');
    const maskedUsername = username.length > 2 ? username.substring(0, 2) + '***' : '***';
    const maskedDomain = domain ? '***' + domain.substring(domain.lastIndexOf('.')) : '*****.com';
    return maskedUsername + '@' + maskedDomain;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    if (!user.user_id) {
      toast.error('Please login to contact the agent');
      navigate('/login');
      return;
    }

    try {
      setSubmitting(true);
      console.log('📝 Starting inquiry submission...');
      
      // Determine if this is a rental enquiry
      const isRental = property?.listing_type === 'rent';
      
      // Use new Property Management API for rental enquiries
      if (isRental) {
        const enquiryData = {
          property_id: parseInt(id || '0'),
          enquiry_for: 'RENTAL',
          tenant_name: formData.buyer_name,
          tenant_email: formData.buyer_email,
          tenant_phone: formData.buyer_phone,
          preferred_move_in: formData.preferred_move_in,
          rental_duration: formData.rental_duration,
          interested_in_visit: formData.interested_in_visit,
          message: formData.reason_to_buy
        };
        
        console.log('📝 Submitting rental enquiry:', enquiryData);
        
        const response = await fetch('/api/property-management/enquiries', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json'
          },
          body: JSON.stringify(enquiryData)
        });
        
        if (!response.ok) {
          const errorText = await response.text();
          console.error('❌ Backend error response:', response.status, errorText);
          try {
            const error = JSON.parse(errorText);
            throw new Error(error.detail || 'Failed to submit rental enquiry');
          } catch (parseError) {
            throw new Error(`Server Error ${response.status}: ${errorText}`);
          }
        }
        
        const result = await response.json();
        console.log('✅ Rental enquiry created successfully:', result);
      } else {
        // Use existing API for sale enquiries
        const result = await createInquiry(user.user_id, formData);
        console.log('✅ Sale inquiry created successfully:', result);
        
        if (!result || !result.inquiry_id) {
          throw new Error('Inquiry creation failed - no inquiry ID returned');
        }
      }
      
      console.log('🔍 Fetching owner profile after successful inquiry...');
        
        // Fetch owner profile after successful inquiry
        if (property && (property.listed_by_id || property.user_id)) {
          try {
            const ownerId = property.listed_by_id || property.user_id;
            const token = localStorage.getItem('access_token');
            
            const profileResponse = await fetch(`/api/profile/${ownerId}`, {
              headers: {
                'Authorization': `Bearer ${token}`,
                'Content-Type': 'application/json'
              }
            });
            
            if (profileResponse.ok) {
              const ownerProfile = await profileResponse.json();
              console.log('🔍 Owner profile fetched:', ownerProfile);
              
              setOwnerDetails({
                name: `${ownerProfile.first_name || ''} ${ownerProfile.last_name || ''}`.trim() || 'Property Owner',
                email: ownerProfile.email || '',
                phone: ownerProfile.phone_number || ''
              });
            } else {
              console.warn('⚠️ Failed to fetch owner profile, using property data');
              // Fallback to property data
              setOwnerDetails({
                name: property.owner_name || 'Property Owner',
                email: property.email || '',
                phone: property.mobile || ''
              });
            }
          } catch (profileError) {
            console.error('❌ Error fetching owner profile:', profileError);
            // Fallback to property data
            setOwnerDetails({
              name: property.owner_name || 'Property Owner',
              email: property.email || '',
              phone: property.mobile || ''
            });
          }
        }
        
        // Mark inquiry as successful and unlock contact details
        setHasInquired(true);
        
        // Update localStorage for tracking (optional)
        const inquiredProperties = JSON.parse(localStorage.getItem('inquired_properties') || '[]');
        if (!inquiredProperties.includes(parseInt(id || '0'))) {
          inquiredProperties.push(parseInt(id || '0'));
          localStorage.setItem('inquired_properties', JSON.stringify(inquiredProperties));
        }
        
        toast.success(`${property?.listing_type === 'rent' ? 'Rental enquiry' : 'Contact details'} ${property?.listing_type === 'rent' ? 'submitted' : 'unlocked'}!`, {
          description: property?.listing_type === 'rent' 
            ? 'Your rental enquiry has been sent to the property owner.' 
            : 'You can now see the owner\'s contact information below.',
          duration: 4000
        });
    } catch (error) {
      console.error('❌ Error creating inquiry:', error);
      const errorMessage = error instanceof Error ? error.message : 'Failed to send inquiry';
      toast.error(errorMessage);
      
      // Do NOT unlock contact details on failure
      // Do NOT set hasInquired to true
    } finally {
      setSubmitting(false);
    }
  };

  if (loading) {
    return (
      <div className="max-w-4xl mx-auto p-8 text-center">
        <p className="text-gray-500">Loading...</p>
      </div>
    );
  }

  if (!property) {
    return (
      <div className="max-w-4xl mx-auto p-8 text-center">
        <p className="text-gray-500">Property not found</p>
        <button onClick={() => navigate(-1)} className="mt-4 bg-blue-600 text-white px-4 py-2 rounded">
          Go Back
        </button>
      </div>
    );
  }

  // Check if current user owns this property
  const user = JSON.parse(localStorage.getItem('user') || '{}');
  const isOwner = user.email && property.email && user.email.toLowerCase() === property.email.toLowerCase();

  if (isOwner) {
    return (
      <div className="bg-gray-50 min-h-screen pt-4 pb-8">
        <div className="max-w-4xl mx-auto px-4 mb-4">
          <div className="bg-white border border-gray-100 rounded-2xl shadow-lg p-8 text-center">
            <div className="w-16 h-16 bg-amber-100 rounded-2xl flex items-center justify-center mx-auto mb-6">
              <svg className="w-8 h-8 text-amber-600" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
              </svg>
            </div>
            <h3 className="text-2xl font-bold text-slate-900 mb-3 tracking-tight">This is Your Property</h3>
            <p className="text-slate-600 text-lg mb-8 max-w-md mx-auto leading-relaxed">You cannot contact yourself for your own property listing. Manage inquiries from your dashboard.</p>
            <div className="flex gap-4 justify-center">
              <button
                onClick={() => navigate(`/property/${id}`)}
                className="inline-flex items-center px-8 py-3 bg-slate-900 text-white rounded-xl hover:bg-slate-800 transition-all duration-200 font-semibold tracking-tight shadow-lg hover:shadow-xl"
              >
                <svg className="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 19l-7-7m0 0l7-7m-7 7h18" />
                </svg>
                View Property
              </button>
              <button
                onClick={() => navigate('/user/dashboard')}
                className="inline-flex items-center px-8 py-3 border border-slate-300 text-slate-700 rounded-xl hover:bg-slate-50 transition-all duration-200 font-semibold tracking-tight"
              >
                <svg className="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                </svg>
                Go to Dashboard
              </button>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-gray-50 min-h-screen pt-4 pb-8">
      <div className="max-w-4xl mx-auto px-4 mb-4">


        {/* Property Owner Details - Premium Design */}
        {hasInquired ? (
          <div className="bg-white rounded-2xl shadow-lg border border-gray-100 mb-8" style={{ boxShadow: '0px 4px 20px rgba(0,0,0,0.05)' }}>
            <div className="p-8">
              {/* Section Header */}
              <div className="flex items-center gap-4 mb-8">
                {/* <div className="w-12 h-12 bg-gradient-to-br from-teal-500 to-teal-600 rounded-xl flex items-center justify-center shadow-lg">
                  <svg className="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                  </svg>
                </div> */}
                <div>
                  <h2 className="text-2xl font-bold text-gray-900 tracking-tight">Property Owner Details</h2>
                  <div className="flex items-center gap-2 mt-1">
                    {/* <svg className="w-4 h-4 text-teal-600" fill="currentColor" viewBox="0 0 20 20">
                      <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                    </svg> */}
                    <span className="text-teal-700 text-sm font-medium">Contact details unlocked after inquiry</span>
                  </div>
                </div>
              </div>
              
              {/* Owner Info Cards Grid */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
                {/* Owner Name */}
                <div className="bg-gray-50 rounded-xl p-6 border border-gray-100">
                  <div className="flex items-center gap-3 mb-3">
<div className="w-8 h-8 bg-orange-100 shadow-sm rounded-lg flex items-center justify-center">
                      <svg className="w-4 h-4 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                      </svg>
                    </div>
                    <span className="text-xs font-semibold text-gray-500 uppercase tracking-wider">Owner Name</span>
                  </div>
                  <p className="text-lg font-bold text-gray-900">{ownerDetails?.name || property.owner_name || 'Property Owner'}</p>
                </div>

                {/* Phone Number */}
                <div className="bg-gray-50 rounded-xl p-6 border border-gray-100">
                  <div className="flex items-center gap-3 mb-3">
<div className="w-8 h-8 bg-orange-100 shadow-sm rounded-lg flex items-center justify-center">
                      <svg className="w-4 h-4 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
                      </svg>
                    </div>
                    <span className="text-xs font-semibold text-gray-500 uppercase tracking-wider">Phone Number</span>
                  </div>
                  <p className="text-lg font-bold text-gray-900 font-mono">{ownerDetails?.phone || property.mobile || 'Phone not available'}</p>
                </div>

                {/* Email Address */}
                <div className="bg-gray-50 rounded-xl p-6 border border-gray-100">
                  <div className="flex items-center gap-3 mb-3">
<div className="w-8 h-8 bg-orange-100 shadow-sm rounded-lg flex items-center justify-center">
                      <svg className="w-4 h-4 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M3 8l7.89 4.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
                      </svg>
                    </div>
                    <span className="text-xs font-semibold text-gray-500 uppercase tracking-wider">Email Address</span>
                  </div>
                  <p className="text-lg font-bold text-gray-900 font-mono break-all">{ownerDetails?.email || property.email || 'Email not available'}</p>
                </div>

                {/* Posted Date */}
                <div className="bg-gray-50 rounded-xl p-6 border border-gray-100">
                  <div className="flex items-center gap-3 mb-3">
<div className="w-8 h-8 bg-orange-100 shadow-sm rounded-lg flex items-center justify-center">
                      <svg className="w-4 h-4 text-orange-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
                      </svg>
                    </div>
                    <span className="text-xs font-semibold text-gray-500 uppercase tracking-wider">Posted Date</span>
                  </div>
                  <p className="text-lg font-bold text-gray-900">{new Date(property.created_at || Date.now()).toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' })}</p>
                </div>




              </div>


              {/* Back to Property Button */}
              <div className="mt-8 flex justify-center">
                <button
                  onClick={() => navigate(`/property/${id}`)}
                  className="inline-flex items-center gap-3 px-8 py-4 bg-gray-100 hover:bg-gray-200 text-gray-700 rounded-xl font-semibold transition-all duration-200 border border-gray-200 hover:border-gray-300"
                >
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M10 19l-7-7m0 0l7-7m-7 7h18" />
                  </svg>
                  Back to Property
                </button>
              </div>
            </div>
          </div>
        ) : (
          <div className="bg-white rounded-lg shadow-sm mb-6">
            <div className="p-6">
              <div className="flex items-start justify-between">
                <div className="flex-1">
                  <div className="flex items-center gap-3 mb-4">
                    <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                      <svg className="w-6 h-6 text-blue-600" fill="currentColor" viewBox="0 0 20 20">
                        <path fillRule="evenodd" d="M10 9a3 3 0 100-6 3 3 0 000 6zm-7 9a7 7 0 1114 0H3z" clipRule="evenodd" />
                      </svg>
                    </div>
                    <div>
                      <h3 className="text-lg font-semibold text-gray-900">POSTED BY Owner:</h3>
                      <div className="flex items-center gap-4 text-sm text-gray-600 mt-1">
                        <span>{maskPhone(ownerDetails?.phone || property.mobile || '')} | {maskEmail(ownerDetails?.email || property.email || '')}</span>
                      </div>
                    </div>
                  </div>
                  
                  <div className="bg-gray-50 rounded-lg p-4 mt-4">
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
                      <div className="flex items-center">
                        <span className="font-medium text-gray-700 min-w-[80px]">Owner:</span>
                        <span className="text-gray-900">Property Owner</span>
                      </div>
                      <div className="flex items-center">
                        <span className="font-medium text-gray-700 min-w-[80px]">Posted On:</span>
                        <span className="text-gray-900">{new Date(property.created_at || Date.now()).toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' })}</span>
                      </div>
                      <div className="flex items-center">
                        <span className="font-medium text-gray-700 min-w-[80px]">Price:</span>
                        <span className="text-blue-600 font-semibold">{property.listing_type === 'rent' ? (property.monthly_rent || 'Rent on request') : formatPrice(property.sale_price || 0)}</span>
                      </div>
                      <div className="flex items-center">
                        <span className="font-medium text-gray-700 min-w-[80px]">Area:</span>
                        <span className="text-gray-900">{property.built_up_area || property.carpet_area || 'N/A'} Sq.Ft.</span>
                      </div>
                    </div>
                    <div className="mt-3 pt-3 border-t border-gray-200">
                      <div className="flex items-center">
                        <span className="font-medium text-gray-700 min-w-[80px]">Property:</span>
                        <span className="text-gray-900">{property.title}</span>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Contact Form */}
        {!hasInquired && (
        <div className="bg-white rounded-lg shadow-sm">
          <div className="border-b border-gray-200 px-6 py-4">
            <h2 className="text-2xl font-bold text-gray-900">{property?.listing_type === 'rent' ? 'Enquire for Rental' : 'Contact Property Owner'}</h2>
            <p className="text-gray-600 mt-1">{property?.listing_type === 'rent' ? 'Fill out the form below to send your rental enquiry' : 'Fill out the form below to send your inquiry'}</p>
          </div>
          
          <form onSubmit={handleSubmit} className="p-6 space-y-6">
            {/* Personal Information */}
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
              <h3 className="text-lg font-semibold text-blue-900 mb-4">Your Information</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Full Name *</label>
                  <input
                    type="text"
                    required
                    value={formData.buyer_name}
                    onChange={(e) => setFormData(prev => ({ ...prev, buyer_name: e.target.value }))}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Phone Number *</label>
                  <input
                    type="tel"
                    required
                    value={formData.buyer_phone}
                    onChange={(e) => setFormData(prev => ({ ...prev, buyer_phone: e.target.value }))}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                </div>
                <div className="md:col-span-2">
                  <label className="block text-sm font-medium text-gray-700 mb-1">Email Address *</label>
                  <input
                    type="email"
                    required
                    value={formData.buyer_email}
                    onChange={(e) => setFormData(prev => ({ ...prev, buyer_email: e.target.value }))}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                </div>
              </div>
            </div>

            {/* Inquiry Details */}
            <div className="bg-green-50 border border-green-200 rounded-lg p-4">
              <h3 className="text-lg font-semibold text-green-900 mb-4">{property?.listing_type === 'rent' ? 'Rental Requirements' : 'Inquiry Details'}</h3>
              <div className="space-y-4">
                {property?.listing_type === 'rent' ? (
                  // Rental-specific fields
                  <>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">Preferred Move-in Date</label>
                      <input
                        type="date"
                        value={formData.preferred_move_in || ''}
                        onChange={(e) => setFormData(prev => ({ ...prev, preferred_move_in: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                      />
                    </div>
                    
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">Rental Duration</label>
                      <select
                        value={formData.rental_duration || ''}
                        onChange={(e) => setFormData(prev => ({ ...prev, rental_duration: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                      >
                        <option value="">Select duration</option>
                        <option value="6 months">6 months</option>
                        <option value="11 months">11 months</option>
                        <option value="1 year">1 year</option>
                        <option value="2 years">2 years</option>
                        <option value="3+ years">3+ years</option>
                      </select>
                    </div>
                    
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">Message</label>
                      <textarea
                        rows={4}
                        value={formData.reason_to_buy}
                        onChange={(e) => setFormData(prev => ({ ...prev, reason_to_buy: e.target.value }))}
                        placeholder="Tell us about your rental requirements..."
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                      />
                    </div>
                  </>
                ) : (
                  // Sale-specific fields
                  <>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">Reason to Buy *</label>
                      <textarea
                        required
                        rows={4}
                        value={formData.reason_to_buy}
                        onChange={(e) => setFormData(prev => ({ ...prev, reason_to_buy: e.target.value }))}
                        placeholder="Tell us why you're interested in this property..."
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                      />
                    </div>
                    
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">Buying Timeline</label>
                      <select
                        value={formData.buying_timeline}
                        onChange={(e) => setFormData(prev => ({ ...prev, buying_timeline: e.target.value as any }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                      >
                        <option value="Immediate">Immediate</option>
                        <option value="Within 1 month">Within 1 month</option>
                        <option value="Within 3 months">Within 3 months</option>
                        <option value="Within 6 months">Within 6 months</option>
                        <option value="More than 6 months">More than 6 months</option>
                      </select>
                    </div>
                  </>
                )}
                
                <div className="space-y-3">
                  {property?.listing_type !== 'rent' && (
                    <div className="flex items-center">
                      <input
                        type="checkbox"
                        id="loan"
                        checked={formData.interested_in_loan}
                        onChange={(e) => setFormData(prev => ({ ...prev, interested_in_loan: e.target.checked }))}
                        className="mr-2"
                      />
                      <label htmlFor="loan" className="text-sm text-gray-700">
                        I'm interested in loan assistance
                      </label>
                    </div>
                  )}
                  
                  <div className="flex items-center">
                    <input
                      type="checkbox"
                      id="visit"
                      checked={formData.interested_in_visit}
                      onChange={(e) => setFormData(prev => ({ ...prev, interested_in_visit: e.target.checked }))}
                      className="mr-2"
                    />
                    <label htmlFor="visit" className="text-sm text-gray-700">
                      {property?.listing_type === 'rent' ? 'I would like to schedule a property visit' : 'I would like to schedule a property visit'}
                    </label>
                  </div>
                </div>
              </div>
            </div>

            {/* Submit Buttons */}
            <div className="flex space-x-4 pt-6">
              <button
                type="button"
                onClick={() => navigate(`/property/${id}`)}
                className="flex-1 px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
              >
                Cancel
              </button>
              <button
                type="submit"
                disabled={submitting}
                onClick={() => {
                  // Track contact click
                  import("../utils/behaviorTracker").then(({ behaviorTracker }) => {
                    behaviorTracker.trackContactClick(parseInt(id || '0'), {
                      property_title: property?.title,
                      contact_type: 'inquiry_form'
                    });
                  });
                }}
                className="flex-1 px-8 py-4 rounded-lg font-bold text-lg transition-colors flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
                style={{backgroundColor: submitting ? '#9ca3af' : '#16a34a', color: 'white'}}
              >
                {submitting ? (
                  <>
                    <svg className="w-5 h-5 animate-spin" fill="none" viewBox="0 0 24 24">
                      <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                      <path className="opacity-75" fill="currentColor" d="m4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                    </svg>
                    Sending...
                  </>
                ) : (
                  <>
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
                    </svg>
                    {property?.listing_type === 'rent' ? 'Send Rental Enquiry' : 'Send Inquiry'}
                  </>
                )}
              </button>
            </div>
          </form>
        </div>
        )}
        

      </div>
    </div>
  );
}