import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Textarea } from "./ui/textarea";
import {
  Phone,
  Mail,
  MapPin,
  Clock,
  MessageCircle,
  Headphones,
  CheckCircle,
  LogIn,
} from "lucide-react";
import { useState, useEffect } from "react";
import { toast } from "sonner";
import { isAuthenticated } from "../utils/auth";
import { useNavigate } from "react-router-dom";

export function ContactUs() {
  const navigate = useNavigate();
  const [isLoggedIn, setIsLoggedIn] = useState(false);

  useEffect(() => {
    try {
      setIsLoggedIn(isAuthenticated());
    } catch (error) {
      console.error("Auth check failed:", error);
      setIsLoggedIn(false);
    }
  }, []);
  const [formData, setFormData] = useState({
    firstName: "",
    lastName: "",
    email: "",
    phone: "",
    subject: "",
    message: "",
  });

  const handleInputChange = (
    e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>
  ) => {
    setFormData({ ...formData, [e.target.name]: e.target.value });
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!isLoggedIn) {
      toast.error("Login required", {
        description:
          "Please login to send a message. You will be redirected to login page.",
        icon: <LogIn className="w-4 h-4" />,
        duration: 3000,
      });
      setTimeout(() => navigate("/login"), 1500);
      return;
    }

    try {
      const response = await fetch(
        "http://localhost:8090/api/contact/contact-messages/",
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${localStorage.getItem("access_token")}`,
          },
          body: JSON.stringify({
            first_name: formData.firstName,
            last_name: formData.lastName,
            email: formData.email,
            phone: formData.phone,
            subject: formData.subject,
            message: formData.message,
          }),
        }
      );

      if (response.ok) {
        toast.success("Message sent successfully!", {
          description:
            "Thank you for contacting us. We will get back to you within 24 hours.",
          icon: <CheckCircle className="w-4 h-4" />,
          duration: 4000,
        });
        setFormData({
          firstName: "",
          lastName: "",
          email: "",
          phone: "",
          subject: "",
          message: "",
        });
      } else {
        toast.error("Failed to send message", {
          description: "Please try again later.",
          duration: 3000,
        });
      }
    } catch (error) {
      toast.error("Network error", {
        description: "Please check your connection and try again.",
        duration: 3000,
      });
    }
  };

  const contactInfo = [
    {
      icon: Phone,
      title: "Phone",
      info: "+91- 8068447416",
      subInfo: "Mon-Sat 9AM-7PM",
    },
    {
      icon: Mail,
      title: "Email",
      info: "info@nglindia.com",
      subInfo: "24/7 Support",
    },
    {
      icon: MapPin,
      title: "Address",
      info: "NAL India Pvt Ltd. #28, Third Floor, MCHS Layout KV Jayaram Road, Jakkur",
      subInfo: "Bengaluru, India",
    },
    {
      icon: Clock,
      title: "Working Hours",
      info: "Mon-Sat: 9AM-7PM",
      subInfo: "Sunday: 10AM-5PM",
    },
  ];

  const services = [
    {
      icon: MessageCircle,
      title: "Sales Inquiry",
      description: "Get help with buying or selling properties",
    },
    {
      icon: Headphones,
      title: "Customer Support",
      description: "Technical support and general assistance",
    },
    {
      icon: Phone,
      title: "Premium Support",
      description: "Priority support for NAL Premium members",
    },
  ];

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <section className="bg-[#0056D2] py-20">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <h1 className="text-4xl md:text-6xl font-bold text-white mb-6">
            Contact Us
          </h1>
          <p className="text-xl text-white/90 max-w-3xl mx-auto">
            We're here to help you with all your real estate needs. Get in touch
            with our expert team today.
          </p>
        </div>
      </section>

      {/* Contact Form & Info Section */}
      <section className="py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid lg:grid-cols-2 gap-12 items-start">
            {/* Contact Form */}
            <div className="bg-white rounded-lg shadow-lg p-8 h-fit">
              <h2 className="text-2xl font-bold text-gray-900 mb-6">
                Send us a Message
              </h2>
              <form className="space-y-6" onSubmit={handleSubmit}>
                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      First Name *
                    </label>
                    <Input
                      name="firstName"
                      value={formData.firstName}
                      onChange={handleInputChange}
                      placeholder="Enter your first name"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Last Name *
                    </label>
                    <Input
                      name="lastName"
                      value={formData.lastName}
                      onChange={handleInputChange}
                      placeholder="Enter your last name"
                      required
                    />
                  </div>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Email *
                  </label>
                  <Input
                    type="email"
                    name="email"
                    value={formData.email}
                    onChange={handleInputChange}
                    placeholder="Enter your email"
                    required
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Phone Number *
                  </label>
                  <Input
                    type="tel"
                    name="phone"
                    value={formData.phone}
                    onChange={handleInputChange}
                    placeholder="Enter your phone number"
                    required
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Subject *
                  </label>
                  <Input
                    name="subject"
                    value={formData.subject}
                    onChange={handleInputChange}
                    placeholder="What is this regarding?"
                    required
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Message *
                  </label>
                  <Textarea
                    name="message"
                    value={formData.message}
                    onChange={handleInputChange}
                    placeholder="Tell us how we can help you..."
                    className="min-h-[120px]"
                    required
                  />
                </div>
                <Button
                  type="submit"
                  className="w-full bg-[#0056D2] hover:bg-[#0056D2]/90 text-white"
                >
                  Send Message
                </Button>
              </form>
            </div>

            {/* Contact Information */}
            <div>
              <h2 className="text-2xl font-bold text-gray-900 mb-6">
                Get in Touch
              </h2>
              <p className="text-gray-600 mb-8">
                Have questions about our services? Need help finding the perfect
                property? Our team is ready to assist you every step of the way.
              </p>

              <div className="space-y-6 mb-8">
                {contactInfo.map((item, index) => (
                  <div
                    key={index}
                    className="flex items-start gap-4 p-4 bg-white rounded-lg shadow-sm"
                  >
                    <item.icon className="w-6 h-6 text-[#0056D2] mt-1" />
                    <div>
                      <h3 className="font-semibold text-gray-900">
                        {item.title}
                      </h3>
                      <p className="text-gray-700">{item.info}</p>
                      <p className="text-sm text-gray-500">{item.subInfo}</p>
                    </div>
                  </div>
                ))}
              </div>

              {/* Services */}
              <h3 className="text-xl font-bold text-gray-900 mb-4">
                How Can We Help?
              </h3>
              <div className="space-y-4">
                {services.map((service, index) => (
                  <div
                    key={index}
                    className="flex items-start gap-3 p-4 bg-white rounded-lg shadow-sm"
                  >
                    <service.icon className="w-5 h-5 text-[#00BFA6] mt-1" />
                    <div>
                      <h4 className="font-semibold text-gray-900">
                        {service.title}
                      </h4>
                      <p className="text-sm text-gray-600">
                        {service.description}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Map Section */}
      <section className="py-16 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-8">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Visit Our Office
            </h2>
            <p className="text-lg text-gray-600">
              Come meet our team in person
            </p>
          </div>
          <div className="rounded-lg overflow-hidden shadow-lg">
            <iframe
              src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3886.8267!2d77.5946!3d13.0827!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x0%3A0x0!2zMTPCsDA0JzU3LjciTiA3N8KwMzUnNDAuNiJF!5e0!3m2!1sen!2sin!4v1234567890123!5m2!1sen!2sin"
              width="100%"
              height="400"
              style={{ border: 0 }}
              allowFullScreen
              loading="lazy"
              referrerPolicy="no-referrer-when-downgrade"
              title="NAL India Office Location"
            ></iframe>
          </div>
        </div>
      </section>

      {/* FAQ Section */}
      <section className="py-16">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Frequently Asked Questions
            </h2>
            <p className="text-lg text-gray-600">
              Quick answers to common questions
            </p>
          </div>
          <div className="space-y-6">
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="font-semibold text-gray-900 mb-2">
                How do I list my property on NAL India?
              </h3>
              <p className="text-gray-600">
                Simply click on "Post Property" and fill out the required
                details. Our team will verify and list your property within 24
                hours.
              </p>
            </div>
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="font-semibold text-gray-900 mb-2">
                Is there any commission for buyers?
              </h3>
              <p className="text-gray-600">
                No, buyers can contact property owners directly without any
                commission charges through our platform.
              </p>
            </div>
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="font-semibold text-gray-900 mb-2">
                What is NAL Premium?
              </h3>
              <p className="text-gray-600">
                NAL Premium offers exclusive features like contacting up to 30
                owners directly, access to premium properties, and priority
                customer support.
              </p>
            </div>
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="font-semibold text-gray-900 mb-2">
                How long does property verification take?
              </h3>
              <p className="text-gray-600">
                Property verification typically takes 24-48 hours. Our team
                ensures all documents are authentic and property details are
                accurate before listing.
              </p>
            </div>
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="font-semibold text-gray-900 mb-2">
                Can I edit my property listing after posting?
              </h3>
              <p className="text-gray-600">
                Yes, you can edit your property details, update photos, and
                modify pricing anytime through your dashboard after logging in.
              </p>
            </div>
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="font-semibold text-gray-900 mb-2">
                What documents are required for property listing?
              </h3>
              <p className="text-gray-600">
                You need property ownership documents, identity proof, address
                proof, and recent property photos. Additional documents may be
                required based on property type.
              </p>
            </div>
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="font-semibold text-gray-900 mb-2">
                How do I schedule property visits?
              </h3>
              <p className="text-gray-600">
                Interested buyers can schedule visits directly through the
                property listing page. You'll receive notifications and can
                confirm or reschedule as needed.
              </p>
            </div>
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="font-semibold text-gray-900 mb-2">
                Is my personal information secure?
              </h3>
              <p className="text-gray-600">
                Yes, we use advanced encryption and security measures to protect
                your personal and financial information. Your data is never
                shared with third parties without consent.
              </p>
            </div>
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="font-semibold text-gray-900 mb-2">
                What are the payment options available?
              </h3>
              <p className="text-gray-600">
                We accept all major payment methods including credit/debit
                cards, net banking, UPI, and digital wallets for premium
                services and subscriptions.
              </p>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
}