import React, { useState } from 'react';
import { MapPin, Navigation } from 'lucide-react';
import MapLibreMap from './MapLibreMap';

interface CoordinateInputProps {
  onLocationSelect?: (lat: number, lng: number) => void;
  initialLat?: number;
  initialLng?: number;
}

const CoordinateInput: React.FC<CoordinateInputProps> = ({
  onLocationSelect,
  initialLat = 12.9716,
  initialLng = 77.5946
}) => {
  const [latitude, setLatitude] = useState<string>(initialLat.toString());
  const [longitude, setLongitude] = useState<string>(initialLng.toString());
  const [mapCenter, setMapCenter] = useState({ lat: initialLat, lng: initialLng });
  const [error, setError] = useState<string>('');

  const validateAndUpdateMap = () => {
    const lat = parseFloat(latitude);
    const lng = parseFloat(longitude);

    if (isNaN(lat) || isNaN(lng)) {
      setError('Please enter valid coordinates');
      return;
    }

    if (lat < -90 || lat > 90) {
      setError('Latitude must be between -90 and 90');
      return;
    }

    if (lng < -180 || lng > 180) {
      setError('Longitude must be between -180 and 180');
      return;
    }

    setError('');
    setMapCenter({ lat, lng });
    onLocationSelect?.(lat, lng);
  };

  const handleMapCoordinateChange = (lat: number, lng: number) => {
    setLatitude(lat.toFixed(6));
    setLongitude(lng.toFixed(6));
    onLocationSelect?.(lat, lng);
  };

  const getCurrentLocation = () => {
    if (!navigator.geolocation) {
      setError('Geolocation is not supported by this browser');
      return;
    }

    navigator.geolocation.getCurrentPosition(
      (position) => {
        const lat = position.coords.latitude;
        const lng = position.coords.longitude;
        setLatitude(lat.toFixed(6));
        setLongitude(lng.toFixed(6));
        setMapCenter({ lat, lng });
        onLocationSelect?.(lat, lng);
        setError('');
      },
      (error) => {
        setError('Unable to retrieve your location');
      }
    );
  };

  return (
    <div className="space-y-4">
      <div className="bg-white rounded-lg border p-4">
        <h3 className="text-lg font-semibold mb-4 flex items-center">
          <MapPin className="w-5 h-5 mr-2 text-blue-600" />
          Location Coordinates
        </h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Latitude
            </label>
            <input
              type="number"
              step="any"
              value={latitude}
              onChange={(e) => setLatitude(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
              placeholder="12.9716"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Longitude
            </label>
            <input
              type="number"
              step="any"
              value={longitude}
              onChange={(e) => setLongitude(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
              placeholder="77.5946"
            />
          </div>
        </div>

        <div className="flex flex-wrap gap-2 mb-4">
          <button
            onClick={validateAndUpdateMap}
            className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors"
          >
            Update Map
          </button>
          
          <button
            onClick={getCurrentLocation}
            className="px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 transition-colors flex items-center"
          >
            <Navigation className="w-4 h-4 mr-2" />
            Use Current Location
          </button>
        </div>

        {error && (
          <div className="text-red-600 text-sm mb-4">
            {error}
          </div>
        )}
      </div>

      <div className="bg-white rounded-lg border overflow-hidden">
        <MapLibreMap
          latitude={mapCenter.lat}
          longitude={mapCenter.lng}
          zoom={15}
          onCoordinateChange={handleMapCoordinateChange}
          showMarker={true}
          interactive={true}
          className="w-full h-96"
        />
      </div>

      <div className="text-sm text-gray-600">
        <p>Click on the map to select a location, or enter coordinates manually above.</p>
        <p>Current coordinates: {latitude}, {longitude}</p>
      </div>
    </div>
  );
};

export default CoordinateInput;