import React, { useEffect, useRef, useState } from 'react';
import { Shield, ShieldCheck, MapPin, Activity, AlertTriangle } from 'lucide-react';

interface CrimeZone {
  city: string;
  zone: string;
  lat: number;
  lon: number;
  crime_score: number;
  crime_intensity: number;
  distance_km?: number;
}

interface CrimeHeatmapProps {
  propertyLat: number;
  propertyLon: number;
  propertyName: string;
}

declare global {
  interface Window {
    google: any;
  }
}

export const CrimeHeatmap: React.FC<CrimeHeatmapProps> = ({
  propertyLat,
  propertyLon,
  propertyName
}) => {
  const mapRef = useRef<HTMLDivElement>(null);
  const [crimeZones, setCrimeZones] = useState<CrimeZone[]>([]);
  const [loading, setLoading] = useState(true);
  const [safetyScore, setSafetyScore] = useState<number>(0);
  const [radius, setRadius] = useState<number>(50); // km

  useEffect(() => {
    fetchCrimeData();
  }, [propertyLat, propertyLon, radius]);

  const fetchCrimeData = async () => {
    try {
      setLoading(true);
      const response = await fetch(
        `http://localhost:8090/api/crime/nearby?lat=${propertyLat}&lon=${propertyLon}&radius=${radius}`
      );
      
      if (response.ok) {
        const data = await response.json();
        setCrimeZones(data.zones || []);
        setSafetyScore(data.safety_score || 0);
      }
    } catch (error) {
      console.error('Error fetching crime data:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (!mapRef.current || !window.google || loading) return;

    const map = new window.google.maps.Map(mapRef.current, {
      center: { lat: propertyLat, lng: propertyLon },
      zoom: 11,
      mapTypeId: 'roadmap',
    });

    // Property marker
    new window.google.maps.Marker({
      position: { lat: propertyLat, lng: propertyLon },
      map: map,
      title: propertyName,
      icon: {
        url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
          <svg width="32" height="32" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg">
            <circle cx="16" cy="16" r="12" fill="#2563eb" stroke="white" stroke-width="3"/>
            <text x="16" y="20" text-anchor="middle" fill="white" font-size="16">🏠</text>
          </svg>
        `),
        scaledSize: new window.google.maps.Size(32, 32),
      }
    });

    // Create heatmap data from crime zones
    const heatmapData = crimeZones.map(zone => ({
      location: new window.google.maps.LatLng(zone.lat, zone.lon),
      weight: zone.crime_intensity / 15
    }));

    // Add heatmap layer
    if (heatmapData.length > 0) {
      const heatmap = new window.google.maps.visualization.HeatmapLayer({
        data: heatmapData,
        radius: 35,
        opacity: 0.6,
        maxIntensity: 1,
        gradient: [
          'rgba(0, 255, 0, 0)',
          'rgba(255, 255, 0, 1)',
          'rgba(255, 165, 0, 1)',
          'rgba(255, 0, 0, 1)'
        ]
      });
      heatmap.setMap(map);
    }

    // Add zone markers
    crimeZones.forEach((zone) => {
      const color = zone.crime_intensity >= 11 ? '#ef4444' : 
                    zone.crime_intensity >= 9 ? '#f59e0b' : '#10b981';
      
      const marker = new window.google.maps.Marker({
        position: { lat: zone.lat, lng: zone.lon },
        map: map,
        icon: {
          path: window.google.maps.SymbolPath.CIRCLE,
          scale: 8,
          fillColor: color,
          fillOpacity: 0.7,
          strokeColor: 'white',
          strokeWeight: 2,
        }
      });

      const infoWindow = new window.google.maps.InfoWindow({
        content: `
          <div style="padding: 8px;">
            <h4 style="margin: 0 0 4px 0; font-weight: bold;">${zone.zone}, ${zone.city}</h4>
            <p style="margin: 2px 0; font-size: 12px;">Crime Score: ${zone.crime_score}</p>
            <p style="margin: 2px 0; font-size: 12px;">Intensity: ${zone.crime_intensity}/15</p>
            ${zone.distance_km ? `<p style="margin: 2px 0; font-size: 12px;">Distance: ${zone.distance_km}km</p>` : ''}
          </div>
        `
      });

      marker.addListener('click', () => {
        infoWindow.open(map, marker);
      });
    });

  }, [propertyLat, propertyLon, crimeZones, loading]);

  const getSafetyLevel = (score: number) => {
    if (score >= 8) return { label: 'Very Safe', color: 'text-green-600', bg: 'bg-green-100' };
    if (score >= 6) return { label: 'Safe', color: 'text-blue-600', bg: 'bg-blue-100' };
    if (score >= 4) return { label: 'Moderate', color: 'text-yellow-600', bg: 'bg-yellow-100' };
    return { label: 'Caution', color: 'text-red-600', bg: 'bg-red-100' };
  };

  const safety = getSafetyLevel(safetyScore);
  const lowRisk = crimeZones.filter(z => z.crime_intensity <= 8).length;
  const mediumRisk = crimeZones.filter(z => z.crime_intensity > 8 && z.crime_intensity <= 10).length;
  const highRisk = crimeZones.filter(z => z.crime_intensity > 10).length;

  return (
    <div className="bg-white rounded-2xl shadow-xl p-6 space-y-8 transition-all duration-300">
      {/* Header with Gradient */}
      <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-2xl p-6 border border-blue-100">
        <div className="flex items-center justify-between">
          <div className="flex-1">
            <div className="flex items-center gap-3 mb-4">
              <ShieldCheck className="h-8 w-8 text-blue-600" />
              <h2 className="text-2xl font-bold tracking-tight text-gray-900">Neighborhood Safety Score</h2>
            </div>
            <div className="flex items-center gap-6">
              <div className="text-5xl font-bold text-gray-900">{safetyScore.toFixed(1)}</div>
              <div className="flex flex-col gap-2">
                <span className="text-lg text-gray-600">/10</span>
                <div className={`px-4 py-1.5 rounded-full text-sm font-semibold ${safety.bg} ${safety.color} shadow-md`}>
                  {safety.label}
                </div>
              </div>
            </div>
            <p className="text-sm text-gray-500 mt-3 italic">Based on NCRB crime intensity data</p>
          </div>
          <div className="hidden md:flex items-center justify-center">
            <div className="relative">
              <Shield className="h-20 w-20 text-blue-500 opacity-20" />
              <Activity className="h-12 w-12 text-blue-600 absolute top-4 left-4" />
            </div>
          </div>
        </div>
      </div>

      {/* Statistics Grid */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl shadow-md p-6 border border-green-100 hover:shadow-lg transition-all duration-300">
          <div className="flex items-center gap-3 mb-3">
            <div className="p-2 bg-green-500 rounded-full">
              <Shield className="w-5 h-5 text-white" />
            </div>
            <span className="text-sm font-semibold text-green-700">Low Risk Zones</span>
          </div>
          <div className="text-3xl font-bold text-green-800 mb-1">{lowRisk}</div>
          <p className="text-xs text-green-600 font-medium">Intensity ≤ 8</p>
        </div>
        
        <div className="bg-gradient-to-br from-yellow-50 to-orange-50 rounded-xl shadow-md p-6 border border-yellow-100 hover:shadow-lg transition-all duration-300">
          <div className="flex items-center gap-3 mb-3">
            <div className="p-2 bg-yellow-500 rounded-full">
              <AlertTriangle className="w-5 h-5 text-white" />
            </div>
            <span className="text-sm font-semibold text-yellow-700">Medium Risk Zones</span>
          </div>
          <div className="text-3xl font-bold text-yellow-800 mb-1">{mediumRisk}</div>
          <p className="text-xs text-yellow-600 font-medium">Intensity 9-10</p>
        </div>
        
        <div className="bg-gradient-to-br from-red-50 to-rose-50 rounded-xl shadow-md p-6 border border-red-100 hover:shadow-lg transition-all duration-300">
          <div className="flex items-center gap-3 mb-3">
            <div className="p-2 bg-red-500 rounded-full">
              <Activity className="w-5 h-5 text-white" />
            </div>
            <span className="text-sm font-semibold text-red-700">High Risk Zones</span>
          </div>
          <div className="text-3xl font-bold text-red-800 mb-1">{highRisk}</div>
          <p className="text-xs text-red-600 font-medium">Intensity ≥ 11</p>
        </div>
      </div>

      {/* Controls */}
      <div className="bg-gray-50 rounded-2xl p-6 border border-gray-100">
        <div className="flex flex-col sm:flex-row items-start sm:items-center gap-4">
          <span className="text-sm font-semibold text-gray-700">Show zones within:</span>
          <div className="flex flex-wrap gap-2">
            {[10, 25, 50, 100].map((dist) => (
              <button
                key={dist}
                onClick={() => setRadius(dist)}
                className={`px-4 py-2 rounded-full text-sm font-semibold transition-all duration-300 ${
                  radius === dist
                    ? 'bg-blue-200 text-black shadow-md scale-105 ring-2 ring-blue-200'
                    : 'bg-white text-gray-700 hover:bg-blue-50 hover:text-blue-600 shadow-sm border border-gray-200'
                }`}
              >
                {dist}km
              </button>
            ))}
          </div>
        </div>
      </div>

      {/* Map Container */}
      <div className="relative">
        <div className="bg-white rounded-2xl shadow-md border border-gray-100 overflow-hidden hover:shadow-lg transition-all duration-300">
          <div className="p-4 bg-gradient-to-r from-gray-50 to-gray-100 border-b border-gray-200">
            <h3 className="text-lg font-bold text-gray-900 flex items-center gap-2">
              <MapPin className="w-5 h-5 text-blue-600" />
              Crime Heatmap
            </h3>
          </div>
          <div className="relative">
            <div
              ref={mapRef}
              className="w-full h-96"
              style={{ minHeight: '400px' }}
            />
            
            {/* Floating Legend */}
            <div className="absolute bottom-4 left-4 backdrop-blur-md bg-white/70 rounded-xl p-3 shadow-lg border border-white/20">
              <div className="flex flex-wrap items-center gap-3 text-xs font-medium">
                <div className="flex items-center gap-1.5">
                  <div className="w-3 h-3 bg-blue-600 rounded-full shadow-sm"></div>
                  <span className="text-gray-700">Property</span>
                </div>
                <div className="flex items-center gap-1.5">
                  <div className="w-3 h-3 bg-green-500 rounded-full shadow-sm"></div>
                  <span className="text-gray-700">Low</span>
                </div>
                <div className="flex items-center gap-1.5">
                  <div className="w-3 h-3 bg-yellow-500 rounded-full shadow-sm"></div>
                  <span className="text-gray-700">Medium</span>
                </div>
                <div className="flex items-center gap-1.5">
                  <div className="w-3 h-3 bg-red-500 rounded-full shadow-sm"></div>
                  <span className="text-gray-700">High</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Disclaimer */}
      <div className="bg-gradient-to-r from-amber-50 to-yellow-50 border border-amber-200 rounded-2xl p-6 shadow-sm">
        <div className="flex items-start gap-3">
          <AlertTriangle className="w-5 h-5 text-amber-600 mt-0.5 flex-shrink-0" />
          <div>
            <h4 className="text-sm font-semibold text-amber-800 mb-1">Data Disclaimer</h4>
            <p className="text-sm text-amber-700 leading-relaxed">
              Safety scores are calculated using inverse distance weighting from nearby crime zones. 
              Data is based on NCRB crime intensity patterns and is for informational purposes only.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
};