import React, { useEffect, useRef, useState } from 'react';
import { Shield, AlertTriangle, MapPin, List, Eye, BarChart3 } from 'lucide-react';

interface CrimeZone {
  city: string;
  zone: string;
  lat: number;
  lon: number;
  crime_score: number;
  crime_intensity: number;
  distance_km?: number;
}

interface CrimeHeatmapProps {
  propertyLat: number;
  propertyLon: number;
  propertyName: string;
}

type ViewMode = 'heatmap' | 'pins' | 'list';

declare global {
  interface Window {
    google: any;
  }
}

export const CrimeHeatmapUpgraded: React.FC<CrimeHeatmapProps> = ({
  propertyLat,
  propertyLon,
  propertyName
}) => {
  const mapRef = useRef<HTMLDivElement>(null);
  const [crimeZones, setCrimeZones] = useState<CrimeZone[]>([]);
  const [loading, setLoading] = useState(true);
  const [safetyScore, setSafetyScore] = useState<number>(0);
  const [radius, setRadius] = useState<number>(50);
  const [viewMode, setViewMode] = useState<ViewMode>('heatmap');
  const [showCard, setShowCard] = useState(true);
  const [nearestZones, setNearestZones] = useState<CrimeZone[]>([]);

  useEffect(() => {
    fetchCrimeData();
  }, [propertyLat, propertyLon, radius]);

  const fetchCrimeData = async () => {
    try {
      setLoading(true);
      const response = await fetch(
        `http://localhost:8090/api/crime/nearby?lat=${propertyLat}&lon=${propertyLon}&radius=${radius}`
      );
      
      if (response.ok) {
        const data = await response.json();
        const zones = data.zones || [];
        setCrimeZones(zones);
        setNearestZones(zones.slice(0, 3));
        setSafetyScore(data.safety_score || 0);
      }
    } catch (error) {
      console.error('Error fetching crime data:', error);
    } finally {
      setLoading(false);
    }
  };

  const getSafetyLevel = (score: number) => {
    if (score >= 7) return { label: 'Safe', color: 'green', icon: '🟢', bgColor: 'bg-green-50', textColor: 'text-green-700' };
    if (score >= 4) return { label: 'Moderate', color: 'orange', icon: '🟡', bgColor: 'bg-orange-50', textColor: 'text-orange-700' };
    return { label: 'High Risk', color: 'red', icon: '🔴', bgColor: 'bg-red-50', textColor: 'text-red-700' };
  };

  const getCrimeIcon = (intensity: number) => {
    if (intensity >= 12) return '🔥';
    if (intensity >= 9) return '🚨';
    if (intensity >= 6) return '⚠️';
    return '🟩';
  };

  const getCrimeBadge = (intensity: number) => {
    if (intensity >= 12) return { text: 'High Crime Area', color: 'bg-red-100 text-red-700' };
    if (intensity >= 9) return { text: 'Moderate Risk', color: 'bg-orange-100 text-orange-700' };
    return { text: 'Safe Residential', color: 'bg-green-100 text-green-700' };
  };

  useEffect(() => {
    if (!mapRef.current || !window.google || loading) return;

    const map = new window.google.maps.Map(mapRef.current, {
      center: { lat: propertyLat, lng: propertyLon },
      zoom: 11,
      mapTypeId: 'roadmap',
    });

    // Property marker
    new window.google.maps.Marker({
      position: { lat: propertyLat, lng: propertyLon },
      map: map,
      title: propertyName,
      icon: {
        url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
          <svg width="32" height="32" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg">
            <circle cx="16" cy="16" r="12" fill="#2563eb" stroke="white" stroke-width="3"/>
            <circle cx="16" cy="16" r="4" fill="white"/>
          </svg>
        `),
        scaledSize: new window.google.maps.Size(32, 32),
      }
    });

    if (viewMode === 'heatmap' && crimeZones.length > 0) {
      // Heatmap layer
      const heatmapData = crimeZones.map(zone => ({
        location: new window.google.maps.LatLng(zone.lat, zone.lon),
        weight: zone.crime_intensity / 15
      }));

      const heatmap = new window.google.maps.visualization.HeatmapLayer({
        data: heatmapData,
        radius: 35,
        opacity: 0.6,
        maxIntensity: 1,
        gradient: [
          'rgba(0, 255, 0, 0)',
          'rgba(255, 255, 0, 1)',
          'rgba(255, 165, 0, 1)',
          'rgba(255, 0, 0, 1)'
        ]
      });
      heatmap.setMap(map);
    }

    if (viewMode === 'pins') {
      // Zone markers with click info
      crimeZones.forEach((zone) => {
        const color = zone.crime_intensity >= 12 ? '#dc2626' : 
                      zone.crime_intensity >= 9 ? '#ea580c' : '#16a34a';
        
        const marker = new window.google.maps.Marker({
          position: { lat: zone.lat, lng: zone.lon },
          map: map,
          icon: {
            path: window.google.maps.SymbolPath.CIRCLE,
            scale: 10,
            fillColor: color,
            fillOpacity: 0.8,
            strokeColor: 'white',
            strokeWeight: 2,
          }
        });

        const badge = getCrimeBadge(zone.crime_intensity);
        const infoWindow = new window.google.maps.InfoWindow({
          content: `
            <div style="padding: 12px; max-width: 250px;">
              <h4 style="margin: 0 0 8px 0; font-weight: bold; color: #1f2937;">${zone.zone}</h4>
              <p style="margin: 0 0 4px 0; font-size: 12px; color: #6b7280;">${zone.city}</p>
              <div style="margin: 8px 0;">
                <span style="display: inline-block; padding: 4px 8px; border-radius: 12px; font-size: 10px; font-weight: 600; ${badge.color.replace('bg-', 'background-color: ').replace('text-', 'color: ')}">${badge.text}</span>
              </div>
              <div style="margin-top: 8px; font-size: 12px; color: #374151;">
                <div>Crime Score: <strong>${zone.crime_score}</strong></div>
                <div>Intensity: <strong>${zone.crime_intensity}/15</strong></div>
                ${zone.distance_km ? `<div>Distance: <strong>${zone.distance_km.toFixed(1)} km</strong></div>` : ''}
              </div>
            </div>
          `
        });

        marker.addListener('click', () => {
          infoWindow.open(map, marker);
        });
      });
    }

  }, [propertyLat, propertyLon, crimeZones, loading, viewMode]);

  const safetyLevel = getSafetyLevel(safetyScore);

  if (loading) {
    return (
      <div className="flex items-center justify-center h-96">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-4 relative">
      {/* View Mode Toggle */}
      <div className="flex items-center justify-between bg-white p-4 rounded-lg shadow-sm border">
        <div className="flex items-center space-x-2">
          <button
            onClick={() => setViewMode('heatmap')}
            className={`px-3 py-2 rounded-md text-sm font-medium transition-colors ${
              viewMode === 'heatmap' ? 'bg-blue-600 text-white' : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
            }`}
          >
            <BarChart3 className="w-4 h-4 inline mr-1" />
            HEATMAP
          </button>
          <button
            onClick={() => setViewMode('pins')}
            className={`px-3 py-2 rounded-md text-sm font-medium transition-colors ${
              viewMode === 'pins' ? 'bg-blue-600 text-white' : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
            }`}
          >
            <MapPin className="w-4 h-4 inline mr-1" />
            ZONE PINS
          </button>
          <button
            onClick={() => setViewMode('list')}
            className={`px-3 py-2 rounded-md text-sm font-medium transition-colors ${
              viewMode === 'list' ? 'bg-blue-600 text-white' : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
            }`}
          >
            <List className="w-4 h-4 inline mr-1" />
            LIST
          </button>
        </div>
        
        <div className="flex items-center space-x-4">
          <select
            value={radius}
            onChange={(e) => setRadius(Number(e.target.value))}
            className="border border-gray-300 rounded-md px-3 py-1 text-sm focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value={10}>10 km</option>
            <option value={25}>25 km</option>
            <option value={50}>50 km</option>
            <option value={100}>100 km</option>
          </select>
          <button
            onClick={() => setShowCard(!showCard)}
            className="p-2 text-gray-600 hover:text-gray-800"
            title="Toggle Info Card"
          >
            <Eye className="w-4 h-4" />
          </button>
        </div>
      </div>

      {/* Map Container */}
      <div className="relative">
        {viewMode === 'list' ? (
          /* List View */
          <div className="bg-white rounded-lg border shadow-sm">
            <div className="p-4 border-b">
              <h3 className="text-lg font-semibold text-gray-900">Crime Zones Near Property</h3>
              <p className="text-sm text-gray-600">Sorted by distance from property</p>
            </div>
            <div className="max-h-96 overflow-y-auto">
              {crimeZones.map((zone, index) => {
                const badge = getCrimeBadge(zone.crime_intensity);
                return (
                  <div key={index} className="p-4 border-b hover:bg-gray-50 transition-colors">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-3">
                        <span className="text-2xl">{getCrimeIcon(zone.crime_intensity)}</span>
                        <div>
                          <h4 className="font-medium text-gray-900">{zone.zone}</h4>
                          <p className="text-sm text-gray-600">{zone.city}</p>
                          <span className={`inline-block mt-1 px-2 py-1 rounded-full text-xs font-medium ${badge.color}`}>
                            {badge.text}
                          </span>
                        </div>
                      </div>
                      <div className="text-right">
                        <div className="text-sm font-medium text-gray-900">{zone.distance_km?.toFixed(1)} km</div>
                        <div className={`text-xs px-2 py-1 rounded-full ${
                          zone.crime_intensity >= 12 ? 'bg-red-100 text-red-700' :
                          zone.crime_intensity >= 9 ? 'bg-orange-100 text-orange-700' :
                          'bg-green-100 text-green-700'
                        }`}>
                          Intensity: {zone.crime_intensity}
                        </div>
                      </div>
                    </div>
                  </div>
                );
              })}
            </div>
          </div>
        ) : (
          /* Map View */
          <div className="relative h-96 rounded-lg overflow-hidden border">
            <div
              ref={mapRef}
              className="w-full h-full"
            />

            {/* Floating Info Card */}
            {showCard && (
              <div className="absolute top-4 left-4 bg-white rounded-lg shadow-lg border p-4 max-w-sm z-10">
                {/* Safety Score Header */}
                <div className={`flex items-center justify-between p-3 rounded-lg mb-3 ${safetyLevel.bgColor}`}>
                  <div className="flex items-center space-x-2">
                    <span className="text-2xl">{safetyLevel.icon}</span>
                    <div>
                      <div className="text-lg font-bold text-gray-900">{safetyScore.toFixed(1)} / 10</div>
                      <div className={`text-sm font-medium ${safetyLevel.textColor}`}>{safetyLevel.label}</div>
                    </div>
                  </div>
                  <Shield className={`w-6 h-6 ${safetyLevel.textColor}`} />
                </div>

                {/* Nearest Crime Zones */}
                <div className="mb-3">
                  <h4 className="text-sm font-semibold text-gray-900 mb-2">Nearest Crime Zones</h4>
                  <div className="space-y-2">
                    {nearestZones.map((zone, index) => (
                      <div key={index} className="flex items-center justify-between text-xs">
                        <div className="flex items-center space-x-2">
                          <span>{getCrimeIcon(zone.crime_intensity)}</span>
                          <span className="font-medium">{zone.zone}</span>
                        </div>
                        <div className="text-right">
                          <div>{zone.distance_km?.toFixed(1)} km</div>
                          <div className="text-gray-500">Intensity {zone.crime_intensity}</div>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>

                {/* Heatmap Legend */}
                <div className="border-t pt-3">
                  <h4 className="text-sm font-semibold text-gray-900 mb-2 flex items-center">
                    <span className="mr-1">🔥</span> Heatmap Legend
                  </h4>
                  <div className="flex items-center justify-between text-xs">
                    <span className="text-green-600">Low</span>
                    <div className="flex space-x-1">
                      <div className="w-3 h-3 bg-green-400 rounded-sm"></div>
                      <div className="w-3 h-3 bg-yellow-400 rounded-sm"></div>
                      <div className="w-3 h-3 bg-orange-400 rounded-sm"></div>
                      <div className="w-3 h-3 bg-red-400 rounded-sm"></div>
                      <div className="w-3 h-3 bg-red-600 rounded-sm"></div>
                    </div>
                    <span className="text-red-600">High</span>
                  </div>
                </div>
              </div>
            )}
          </div>
        )}
      </div>

      {/* Statistics - Only show when not in list view */}
      {viewMode !== 'list' && (
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="bg-white p-4 rounded-lg shadow-sm border">
            <div className="flex items-center space-x-2">
              <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                <MapPin className="w-4 h-4 text-blue-600" />
              </div>
              <div>
                <div className="text-2xl font-bold text-gray-900">{crimeZones.length}</div>
                <div className="text-sm text-gray-600">Crime Zones</div>
              </div>
            </div>
          </div>

          <div className="bg-white p-4 rounded-lg shadow-sm border">
            <div className="flex items-center space-x-2">
              <div className="w-8 h-8 bg-red-100 rounded-full flex items-center justify-center">
                <AlertTriangle className="w-4 h-4 text-red-600" />
              </div>
              <div>
                <div className="text-2xl font-bold text-gray-900">
                  {crimeZones.filter(zone => zone.crime_intensity >= 10).length}
                </div>
                <div className="text-sm text-gray-600">High Risk Zones</div>
              </div>
            </div>
          </div>

          <div className="bg-white p-4 rounded-lg shadow-sm border">
            <div className="flex items-center space-x-2">
              <div className={`w-8 h-8 rounded-full flex items-center justify-center ${safetyLevel.bgColor}`}>
                <Shield className={`w-4 h-4 ${safetyLevel.textColor}`} />
              </div>
              <div>
                <div className={`text-2xl font-bold ${safetyLevel.textColor}`}>
                  {safetyLevel.label}
                </div>
                <div className="text-sm text-gray-600">Safety Rating</div>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* How We Calculate Section */}
      <div className="bg-gray-50 p-4 rounded-lg border">
        <h4 className="text-sm font-semibold text-gray-900 mb-2">How We Calculate Safety Score</h4>
        <div className="text-xs text-gray-600 space-y-1">
          <div>• <strong>Crime intensity:</strong> Based on historical crime data and incident reports</div>
          <div>• <strong>Distance weighting:</strong> Closer zones have higher impact on score</div>
          <div>• <strong>Zone risk level:</strong> Categorized by crime frequency and severity</div>
          <div>• <strong>Population density:</strong> Normalized for area demographics</div>
        </div>
      </div>
    </div>
  );
};