/**
 * Document Verification Final Page
 * Displays complete verification results with cross-document analysis and RIBIL scorecard
 * Fixes decimal/JSON/float errors with proper data handling
 */

import React, { useState, useEffect } from 'react';
import { AlertCircle, CheckCircle, XCircle, TrendingUp, FileText } from 'lucide-react';
import {
  parseNumericValue,
  formatNumericValue,
  parseJsonSafely,
  ensureArray,
  ensureObject,
  cleanApiResponse,
  validateFeatureScore,
  formatScoreForDisplay,
  getGradeColor,
  getConsistencyColor
} from '../utils/dataTypeHelpers';

interface DocumentResult {
  id: number;
  doc_type: string;
  filename: string;
  verification_status: string;
  ocr_confidence: number;
  extracted_json: any;
  features: {
    f1_data_validation: number;
    f2_forgery_detection: number;
    f3_duplicate_detection: string;
    f4_authenticity: number;
    f5_dispute_risk: number;
  };
}

interface CrossDocumentAnalysis {
  total_documents: number;
  overall_consistency: number;
  matching_fields: string[];
  mismatched_fields: string[];
  consistency_level: string;
}

interface RibilScorecard {
  overall_score: number;
  grade: string;
  document_verification: number;
  completeness: number;
  quality: number;
  cross_document_bonus: number;
  duplicate_penalty: number;
}

interface VerificationResult {
  session_id: string;
  property_id: string;
  owner_name: string;
  documents: DocumentResult[];
  cross_document_analysis: CrossDocumentAnalysis;
  ribil_scorecard: RibilScorecard;
  final_verdict: {
    status: string;
    risk_level: string;
    recommendation: string;
    issues: string[];
    warnings: string[];
  };
}

export function DocumentVerificationFinalPage() {
  const [result, setResult] = useState<VerificationResult | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchVerificationResults();
  }, []);

  const fetchVerificationResults = async () => {
    try {
      setLoading(true);
      const sessionId = new URLSearchParams(window.location.search).get('session_id');
      
      if (!sessionId) {
        throw new Error('No session ID provided');
      }

      const response = await fetch(`/api/verification/final-results/${sessionId}`);
      
      if (!response.ok) {
        throw new Error(`Failed to fetch results: ${response.statusText}`);
      }

      const data = await response.json();
      
      // Clean API response to fix decimal/JSON/float errors
      const cleanedData = cleanApiResponse(data);
      setResult(cleanedData as VerificationResult);
      setError(null);
    } catch (err) {
      console.error('Error fetching verification results:', err);
      setError(err instanceof Error ? err.message : 'Unknown error');
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (error || !result) {
    return (
      <div className="min-h-screen bg-red-50 p-6">
        <div className="max-w-4xl mx-auto">
          <div className="bg-white rounded-lg shadow-lg p-6 border-l-4 border-red-600">
            <div className="flex items-center gap-3 mb-4">
              <XCircle className="w-6 h-6 text-red-600" />
              <h2 className="text-2xl font-bold text-red-800">Error Loading Results</h2>
            </div>
            <p className="text-red-700">{error || 'Failed to load verification results'}</p>
          </div>
        </div>
      </div>
    );
  }

  const ribilScore = parseNumericValue(result.ribil_scorecard?.overall_score, 0);
  const ribilGrade = result.ribil_scorecard?.grade || 'E';
  const consistency = parseNumericValue(result.cross_document_analysis?.overall_consistency, 0);

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-50 p-6">
      <div className="max-w-6xl mx-auto">
        {/* Header */}
        <div className="bg-white rounded-lg shadow-lg p-6 mb-6">
          <h1 className="text-3xl font-bold text-gray-800 mb-2">Document Verification Results</h1>
          <p className="text-gray-600">Session: {result.session_id}</p>
          <p className="text-gray-600">Property: {result.property_id} | Owner: {result.owner_name}</p>
        </div>

        {/* Final Verdict */}
        <div className={`rounded-lg shadow-lg p-6 mb-6 border-l-4 ${
          result.final_verdict?.status === 'VERIFIED' 
            ? 'bg-green-50 border-green-600' 
            : 'bg-yellow-50 border-yellow-600'
        }`}>
          <div className="flex items-center justify-between mb-4">
            <div className="flex items-center gap-3">
              {result.final_verdict?.status === 'VERIFIED' ? (
                <CheckCircle className="w-8 h-8 text-green-600" />
              ) : (
                <AlertCircle className="w-8 h-8 text-yellow-600" />
              )}
              <div>
                <h2 className="text-2xl font-bold text-gray-800">
                  {result.final_verdict?.status || 'PENDING'}
                </h2>
                <p className="text-gray-600">Risk Level: {result.final_verdict?.risk_level || 'UNKNOWN'}</p>
              </div>
            </div>
            <div className="text-right">
              <p className="text-sm text-gray-600">Recommendation</p>
              <p className="text-lg font-semibold text-gray-800">
                {result.final_verdict?.recommendation || 'Review required'}
              </p>
            </div>
          </div>
        </div>

        {/* RIBIL Scorecard */}
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-6">
          {/* Overall Score */}
          <div className="bg-white rounded-lg shadow-lg p-6">
            <h3 className="text-lg font-semibold text-gray-800 mb-4">RIBIL Score</h3>
            <div className="text-center">
              <div className="text-5xl font-bold text-blue-600 mb-2">
                {formatNumericValue(ribilScore, 1)}
              </div>
              <div className={`inline-block px-4 py-2 rounded-lg font-bold ${getGradeColor(ribilGrade)}`}>
                Grade: {ribilGrade}
              </div>
            </div>
          </div>

          {/* Component Scores */}
          <div className="bg-white rounded-lg shadow-lg p-6">
            <h3 className="text-lg font-semibold text-gray-800 mb-4">Components</h3>
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="text-gray-600">Verification:</span>
                <span className="font-semibold">
                  {formatScoreForDisplay(result.ribil_scorecard?.document_verification, 1)}
                </span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Completeness:</span>
                <span className="font-semibold">
                  {formatScoreForDisplay(result.ribil_scorecard?.completeness, 1)}
                </span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Quality:</span>
                <span className="font-semibold">
                  {formatScoreForDisplay(result.ribil_scorecard?.quality, 1)}
                </span>
              </div>
            </div>
          </div>

          {/* Cross-Document Analysis */}
          <div className="bg-white rounded-lg shadow-lg p-6">
            <h3 className="text-lg font-semibold text-gray-800 mb-4">Cross-Document</h3>
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="text-gray-600">Documents:</span>
                <span className="font-semibold">
                  {result.cross_document_analysis?.total_documents || 0}
                </span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Consistency:</span>
                <span className="font-semibold">
                  {formatScoreForDisplay(consistency, 1)}
                </span>
              </div>
              <div className={`inline-block px-2 py-1 rounded text-sm font-semibold ${
                getConsistencyColor(result.cross_document_analysis?.consistency_level || 'POOR')
              }`}>
                {result.cross_document_analysis?.consistency_level || 'UNKNOWN'}
              </div>
            </div>
          </div>
        </div>

        {/* Documents Detail */}
        <div className="bg-white rounded-lg shadow-lg p-6 mb-6">
          <h3 className="text-xl font-semibold text-gray-800 mb-4 flex items-center gap-2">
            <FileText className="w-5 h-5" />
            Uploaded Documents
          </h3>
          <div className="space-y-4">
            {ensureArray(result.documents).map((doc: DocumentResult, idx: number) => (
              <div key={doc.id || idx} className="border rounded-lg p-4">
                <div className="flex justify-between items-start mb-3">
                  <div>
                    <h4 className="font-semibold text-gray-800">{doc.doc_type}</h4>
                    <p className="text-sm text-gray-600">{doc.filename}</p>
                  </div>
                  <div className="text-right">
                    <span className={`px-3 py-1 rounded-full text-sm font-semibold ${
                      doc.verification_status === 'completed' 
                        ? 'bg-green-100 text-green-800' 
                        : 'bg-yellow-100 text-yellow-800'
                    }`}>
                      {doc.verification_status}
                    </span>
                  </div>
                </div>

                {/* Feature Scores */}
                <div className="grid grid-cols-2 md:grid-cols-5 gap-3">
                  <div className="bg-blue-50 p-3 rounded">
                    <p className="text-xs text-gray-600">Data Validation</p>
                    <p className="text-lg font-bold text-blue-600">
                      {formatScoreForDisplay(doc.features?.f1_data_validation, 1)}
                    </p>
                  </div>
                  <div className="bg-purple-50 p-3 rounded">
                    <p className="text-xs text-gray-600">Forgery Detection</p>
                    <p className="text-lg font-bold text-purple-600">
                      {formatScoreForDisplay(doc.features?.f2_forgery_detection, 1)}
                    </p>
                  </div>
                  <div className="bg-indigo-50 p-3 rounded">
                    <p className="text-xs text-gray-600">Duplicates</p>
                    <p className="text-lg font-bold text-indigo-600">
                      {doc.features?.f3_duplicate_detection || 'CLEAN'}
                    </p>
                  </div>
                  <div className="bg-green-50 p-3 rounded">
                    <p className="text-xs text-gray-600">Authenticity</p>
                    <p className="text-lg font-bold text-green-600">
                      {formatScoreForDisplay(doc.features?.f4_authenticity, 1)}
                    </p>
                  </div>
                  <div className="bg-red-50 p-3 rounded">
                    <p className="text-xs text-gray-600">Dispute Risk</p>
                    <p className="text-lg font-bold text-red-600">
                      {formatScoreForDisplay(doc.features?.f5_dispute_risk, 1)}
                    </p>
                  </div>
                </div>

                {/* Extracted Data */}
                <details className="mt-3 cursor-pointer">
                  <summary className="text-sm text-blue-600 hover:text-blue-800">
                    View Extracted Data
                  </summary>
                  <div className="mt-2 bg-gray-50 p-3 rounded text-sm font-mono overflow-auto max-h-48">
                    <pre>{JSON.stringify(parseJsonSafely(doc.extracted_json), null, 2)}</pre>
                  </div>
                </details>
              </div>
            ))}
          </div>
        </div>

        {/* Issues and Warnings */}
        {(ensureArray(result.final_verdict?.issues).length > 0 || 
          ensureArray(result.final_verdict?.warnings).length > 0) && (
          <div className="bg-white rounded-lg shadow-lg p-6">
            {ensureArray(result.final_verdict?.issues).length > 0 && (
              <div className="mb-4">
                <h4 className="font-semibold text-red-800 mb-2">Issues</h4>
                <ul className="space-y-1">
                  {ensureArray(result.final_verdict?.issues).map((issue: string, idx: number) => (
                    <li key={idx} className="text-red-700 flex items-start gap-2">
                      <span className="text-red-600 mt-1">•</span>
                      <span>{issue}</span>
                    </li>
                  ))}
                </ul>
              </div>
            )}
            {ensureArray(result.final_verdict?.warnings).length > 0 && (
              <div>
                <h4 className="font-semibold text-yellow-800 mb-2">Warnings</h4>
                <ul className="space-y-1">
                  {ensureArray(result.final_verdict?.warnings).map((warning: string, idx: number) => (
                    <li key={idx} className="text-yellow-700 flex items-start gap-2">
                      <span className="text-yellow-600 mt-1">•</span>
                      <span>{warning}</span>
                    </li>
                  ))}
                </ul>
              </div>
            )}
          </div>
        )}
      </div>
    </div>
  );
}
