import React, { useState, useEffect } from 'react';
import { getPropertyDocuments, generateDocumentDownloadURL, PropertyDocument } from '../api/documents';
import { toast } from 'sonner';

interface DocumentsListProps {
  propertyId: number;
  refreshTrigger?: number;
  showOnlyScore?: boolean;
}

const DocumentsList: React.FC<DocumentsListProps> = ({ propertyId, refreshTrigger, showOnlyScore = false }) => {
  const [documents, setDocuments] = useState<PropertyDocument[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [animateChart, setAnimateChart] = useState(false);

  const loadDocuments = async () => {
    try {
      setLoading(true);
      console.log('=== LOADING DOCUMENTS DEBUG ===');
      console.log('Property ID:', propertyId);
      console.log('Property ID type:', typeof propertyId);
      console.log('Property ID as string:', propertyId.toString());
      console.log('Access token exists:', !!localStorage.getItem('access_token'));
      console.log('API URL will be:', `http://localhost:8090/api/documents/${propertyId}`);
      
      const docs = await getPropertyDocuments(propertyId.toString());
      console.log('Documents API response:', docs);
      console.log('Documents count:', docs.length);
      console.log('================================');
      
      // Filter out test documents
      const filteredDocs = docs.filter(doc => 
        doc.doc_type.toLowerCase() !== 'test document' && 
        doc.doc_type.toLowerCase() !== 'test_document' &&
        doc.doc_type.toLowerCase() !== 'test'
      );
      
      setDocuments(filteredDocs);
      setError(null);
    } catch (err) {
      console.error('Documents API error:', err);
      const errorMessage = err instanceof Error ? err.message : 'Failed to load documents';
      
      // If showOnlyScore is true, don't show errors - just show score of 0
      if (showOnlyScore) {
        setDocuments([]);
        setError(null);
      } else {
        // Check if it's an ownership error
        if (errorMessage.includes('only view documents for your own properties')) {
          setError('Documents are only visible to the property owner');
        } else {
          setError(errorMessage);
        }
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    loadDocuments();
  }, [propertyId, refreshTrigger]);

  useEffect(() => {
    // Trigger animation after documents load
    if (!loading && documents.length > 0) {
      setTimeout(() => setAnimateChart(true), 100);
    }
  }, [loading, documents]);

  const handleDownload = async (document: PropertyDocument) => {
    try {
      // Track document click
      import('../utils/behaviorTracker').then(({ behaviorTracker }) => {
        behaviorTracker.trackDocumentClick(propertyId, {
          document_type: document.doc_type,
          document_name: document.file_url.split('/').pop()
        });
      });
      
      console.log('Generating download URL for document ID:', document.id);
      const result = await generateDocumentDownloadURL(document.id.toString());
      if (result.success && result.download_url) {
        window.open(result.download_url, '_blank');
        toast.success('Document opened for viewing');
      } else {
        toast.error('Failed to generate download URL');
      }
    } catch (err) {
      console.error('Download error:', err);
      toast.error('Failed to open document');
    }
  };



  const formatDocType = (docType: string) => {
    return docType.split('_').map(word => 
      word.charAt(0).toUpperCase() + word.slice(1)
    ).join(' ');
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString();
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  if (loading) {
    if (showOnlyScore) {
      return (
        <div className="flex justify-center mb-6">
          <div className="w-20 h-20 rounded-full bg-gray-300 animate-pulse flex items-center justify-center">
            <div className="text-center">
              <div className="text-xl font-bold text-gray-500">...</div>
              <div className="text-xs text-gray-500">RIBL</div>
            </div>
          </div>
        </div>
      );
    }
    return (
      <div className="flex items-center justify-center py-8">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <span className="ml-2">Loading documents...</span>
      </div>
    );
  }

  if (error && !showOnlyScore) {
    return (
      <div className="bg-red-50 border border-red-200 rounded-md p-4">
        <p className="text-red-600">{error}</p>
        <button 
          onClick={loadDocuments}
          className="mt-2 text-red-600 hover:text-red-800 underline"
        >
          Try again
        </button>
      </div>
    );
  }

  const getRIBLScore = () => {
    const totalPoints = documents.length * 2;
    return Math.min(totalPoints, 10);
  };

  const renderRIBLScorePieChart = (score: number) => {
    const getScoreColor = (score: number) => {
      if (score <= 3) return 'bg-red-500';
      if (score <= 6) return 'bg-yellow-500';
      return 'bg-green-500';
    };

    return (
      <div className="flex justify-center mb-6">
        <div className={`w-20 h-20 rounded-full flex items-center justify-center ${getScoreColor(score)} shadow-lg`}>
          <div className="text-center">
            <div className="text-xl font-bold text-white">{score}</div>
            <div className="text-xs text-white opacity-90">RIBL</div>
          </div>
        </div>
      </div>
    );
  };

  if (documents.length === 0 && !error) {
    if (showOnlyScore) {
      return (
        <div className="documents-list">
          {renderRIBLScorePieChart(0)}
        </div>
      );
    }
    return (
      <div className="text-center py-8 text-gray-500">
        <p>No documents uploaded yet</p>
      </div>
    );
  }



  const renderRIBLScoreSlider = (score: number) => {
    const percentage = (score / 10) * 100;
    const getScoreColor = (score: number) => {
      if (score <= 3) return 'bg-red-500';
      if (score <= 6) return 'bg-yellow-500';
      return 'bg-green-500';
    };

    return (
      <div className="w-full">
        <div className="flex justify-between items-center mb-1">
          <span className="text-xs font-medium text-gray-700">RIBL Score</span>
          <span className="text-xs font-bold text-gray-900">{score}/10</span>
        </div>
        <div className="w-full bg-gray-200 rounded-full h-2">
          <div 
            className={`h-2 rounded-full transition-all duration-300 ${getScoreColor(score)}`}
            style={{ width: `${percentage}%` }}
          ></div>
        </div>
        <div className="text-xs text-gray-500 mt-1">
          {score <= 3 ? 'Low Trust' : score <= 6 ? 'Medium Trust' : 'High Trust'}
        </div>
      </div>
    );
  };

  // If showOnlyScore is true, only show the RIBL score
  if (showOnlyScore) {
    return (
      <div className="documents-list">
        {renderRIBLScorePieChart(getRIBLScore())}
      </div>
    );
  }

  return (
    <div className="documents-list">
      {/* RIBL Score Chart */}
      {renderRIBLScorePieChart(getRIBLScore())}
      
      <h3 className="text-lg font-semibold mb-4">Uploaded Documents ({documents.length})</h3>
      
      {/* Header Row */}
      <div className="grid grid-cols-3 gap-4 p-3 bg-gray-50 rounded-t-lg border-b">
        <div className="font-medium text-gray-700">Document</div>
        <div className="font-medium text-gray-700 text-center">RIBL Score</div>
        <div className="font-medium text-gray-700 text-center">Download</div>
      </div>
      
      <div className="border border-gray-200 rounded-b-lg">
        {documents.map((document, index) => (
          <div key={document.id} className={`grid grid-cols-3 gap-4 p-4 hover:bg-gray-50 ${index !== documents.length - 1 ? 'border-b border-gray-100' : ''}`}>
            {/* Column 1: Document Info */}
            <div className="flex items-center space-x-3">
              <div className="flex-shrink-0">
                <svg className="h-8 w-8 text-red-600" fill="currentColor" viewBox="0 0 20 20">
                  <path fillRule="evenodd" d="M4 4a2 2 0 012-2h4.586A2 2 0 0112 2.586L15.414 6A2 2 0 0116 7.414V16a2 2 0 01-2 2H6a2 2 0 01-2-2V4zm2 6a1 1 0 011-1h6a1 1 0 110 2H7a1 1 0 01-1-1zm1 3a1 1 0 100 2h6a1 1 0 100-2H7z" clipRule="evenodd" />
                </svg>
              </div>
              <div className="min-w-0 flex-1">
                <p className="text-sm font-medium text-gray-900 truncate">
                  {document.file_url.split('/').pop() || 'Document'}
                </p>
                <p className="text-sm text-gray-500">
                  {formatDocType(document.doc_type)}
                </p>
                <p className="text-xs text-gray-400">
                  {formatDate(document.created_at)}
                </p>
              </div>
            </div>
            
            {/* Column 2: RIBL Score Slider */}
            <div className="flex items-center justify-center px-4">
              {renderRIBLScoreSlider(getRIBLScore())}
            </div>
            
            {/* Column 3: Download Button */}
            <div className="flex items-center justify-center">
              <button
                onClick={() => handleDownload(document)}
                className="inline-flex items-center px-3 py-2 border border-gray-300 shadow-sm text-sm rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                title="Download Document"
              >
                <svg className="h-4 w-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                </svg>
                Download
              </button>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
};

export default DocumentsList;