import React, { useState } from 'react';
import { toast } from 'sonner';

interface Property {
  id: number;
  listing_type: 'sell' | 'rent' | 'urgent-sale' | 'bidding';
  title: string;
  owner_id?: number;
}

interface EnquiryFormProps {
  property: Property;
  onClose: () => void;
}

export function EnquiryForm({ property, onClose }: EnquiryFormProps) {
  const [formData, setFormData] = useState({
    tenant_name: '',
    tenant_email: '',
    tenant_phone: '',
    preferred_move_in: '',
    rental_duration: '',
    interested_in_visit: false,
    message: "I'm interested in this property..."
  });
  const [loading, setLoading] = useState(false);

  const isRental = property.listing_type === 'rent';

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      
      const enquiryData = {
        property_id: property.id,
        enquiry_for: isRental ? 'RENTAL' : 'SALE',
        tenant_name: formData.tenant_name,
        tenant_email: formData.tenant_email,
        tenant_phone: formData.tenant_phone,
        preferred_move_in: isRental ? formData.preferred_move_in : undefined,
        rental_duration: isRental ? formData.rental_duration : undefined,
        interested_in_visit: formData.interested_in_visit,
        message: formData.message
      };

      const response = await fetch('/api/property-management/enquiries', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        },
        body: JSON.stringify(enquiryData)
      });

      if (response.ok) {
        toast.success(`${isRental ? 'Rental' : 'Sale'} enquiry submitted successfully!`);
        onClose();
      } else {
        const error = await response.json();
        toast.error(error.detail || 'Failed to submit enquiry');
      }
    } catch (error) {
      console.error('Enquiry submission failed:', error);
      toast.error('Failed to submit enquiry');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4">
        <div className="flex justify-between items-center mb-4">
          <h2 className="text-xl font-semibold">
            {isRental ? 'Rental' : 'Purchase'} Enquiry
          </h2>
          <button
            onClick={onClose}
            className="text-gray-500 hover:text-gray-700"
          >
            ✕
          </button>
        </div>

        <div className="mb-4 p-3 bg-gray-50 rounded-lg">
          <h3 className="font-medium text-sm text-gray-700">Property</h3>
          <p className="text-gray-900">{property.title}</p>
          <span className={`inline-block px-2 py-1 rounded text-xs font-medium mt-2 ${
            isRental ? 'bg-green-100 text-green-800' : 'bg-blue-100 text-blue-800'
          }`}>
            {isRental ? 'For Rent' : 'For Sale'}
          </span>
        </div>

        <form onSubmit={handleSubmit} className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Full Name *
            </label>
            <input
              type="text"
              required
              value={formData.tenant_name}
              onChange={(e) => setFormData({...formData, tenant_name: e.target.value})}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Email *
            </label>
            <input
              type="email"
              required
              value={formData.tenant_email}
              onChange={(e) => setFormData({...formData, tenant_email: e.target.value})}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Phone Number *
            </label>
            <input
              type="tel"
              required
              value={formData.tenant_phone}
              onChange={(e) => setFormData({...formData, tenant_phone: e.target.value})}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            />
          </div>

          {/* Rental-specific fields */}
          {isRental && (
            <>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Preferred Move-in Date
                </label>
                <input
                  type="date"
                  value={formData.preferred_move_in}
                  onChange={(e) => setFormData({...formData, preferred_move_in: e.target.value})}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Rental Duration
                </label>
                <select
                  value={formData.rental_duration}
                  onChange={(e) => setFormData({...formData, rental_duration: e.target.value})}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="">Select duration</option>
                  <option value="6 months">6 months</option>
                  <option value="11 months">11 months</option>
                  <option value="1 year">1 year</option>
                  <option value="2 years">2 years</option>
                  <option value="3+ years">3+ years</option>
                </select>
              </div>
            </>
          )}

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Message
            </label>
            <textarea
              rows={3}
              value={formData.message}
              onChange={(e) => setFormData({...formData, message: e.target.value})}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            />
          </div>

          <div className="flex items-center">
            <input
              type="checkbox"
              id="interested_in_visit"
              checked={formData.interested_in_visit}
              onChange={(e) => setFormData({...formData, interested_in_visit: e.target.checked})}
              className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
            />
            <label htmlFor="interested_in_visit" className="ml-2 text-sm text-gray-700">
              I'm interested in scheduling a visit
            </label>
          </div>

          <div className="flex gap-3 pt-4">
            <button
              type="button"
              onClick={onClose}
              className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
            >
              Cancel
            </button>
            <button
              type="submit"
              disabled={loading}
              className={`flex-1 px-4 py-2 rounded-lg text-white transition-colors ${
                isRental 
                  ? 'bg-green-600 hover:bg-green-700' 
                  : 'bg-blue-600 hover:bg-blue-700'
              } ${loading ? 'opacity-50 cursor-not-allowed' : ''}`}
            >
              {loading ? 'Submitting...' : `Submit ${isRental ? 'Rental' : 'Purchase'} Enquiry`}
            </button>
          </div>
        </form>
      </div>
    </div>
  );
}