import React, { useState } from 'react';
import { useEventBus } from '../context/EventBusContext';
import { eventBusService } from '../services/eventBusService';
import { Button } from './ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { Send, Wifi, WifiOff, TestTube, Bell, Home, Calendar, FileCheck, Gavel } from 'lucide-react';

interface EventBusTestProps {
  userId: string;
}

export const EventBusTest: React.FC<EventBusTestProps> = ({ userId }) => {
  const { isConnected, connectionStatus, sendMessage } = useEventBus();
  const [isLoading, setIsLoading] = useState(false);
  const [lastResponse, setLastResponse] = useState<string>('');

  const handleTestEvent = async (eventType: string, eventData: any) => {
    setIsLoading(true);
    try {
      let response;
      
      switch (eventType) {
        case 'test_notification':
          response = await eventBusService.sendTestEvent(userId, 'This is a test notification from Event Bus!');
          break;
          
        case 'property_approved':
          response = await eventBusService.publishPropertyStatusEvent(
            userId,
            'prop-123',
            'Beautiful 3BHK Apartment',
            'approved'
          );
          break;
          
        case 'property_rejected':
          response = await eventBusService.publishPropertyStatusEvent(
            userId,
            'prop-124',
            'Luxury Villa in Gurgaon',
            'rejected',
            'Incomplete documentation'
          );
          break;
          
        case 'visit_approved':
          response = await eventBusService.publishVisitStatusEvent(
            userId,
            'visit-456',
            'Modern Office Space',
            'approved'
          );
          break;
          
        case 'auction_bid':
          response = await eventBusService.publishAuctionBidEvent(
            'prop-789',
            'Premium Penthouse',
            5000000,
            'John Doe',
            [userId]
          );
          break;
          
        case 'auction_won':
          response = await eventBusService.publishAuctionWinnerEvent(
            userId,
            'prop-789',
            'Premium Penthouse',
            5500000
          );
          break;
          
        case 'inquiry_received':
          response = await eventBusService.publishInquiryEvent(
            userId,
            'inq-101',
            'Cozy 2BHK Flat',
            'Jane Smith'
          );
          break;
          
        case 'document_verified':
          response = await eventBusService.publishDocumentVerificationEvent(
            userId,
            'doc-202',
            'Property Title Deed',
            'verified'
          );
          break;
          
        case 'websocket_test':
          const success = sendMessage({
            type: 'test_event',
            data: { message: 'Direct WebSocket test message' }
          });
          response = { status: success ? 'success' : 'failed', message: 'WebSocket message sent' };
          break;
          
        default:
          response = { status: 'error', message: 'Unknown event type' };
      }
      
      setLastResponse(JSON.stringify(response, null, 2));
    } catch (error) {
      setLastResponse(`Error: ${error}`);
    } finally {
      setIsLoading(false);
    }
  };

  const getStatusColor = () => {
    switch (connectionStatus) {
      case 'connected': return 'bg-green-500';
      case 'connecting': return 'bg-yellow-500';
      case 'error': return 'bg-red-500';
      default: return 'bg-gray-500';
    }
  };

  const testEvents = [
    {
      id: 'test_notification',
      title: 'Test Notification',
      description: 'Send a basic test notification',
      icon: <Bell className="w-4 h-4" />,
      color: 'bg-blue-500'
    },
    {
      id: 'property_approved',
      title: 'Property Approved',
      description: 'Simulate property approval notification',
      icon: <Home className="w-4 h-4" />,
      color: 'bg-green-500'
    },
    {
      id: 'property_rejected',
      title: 'Property Rejected',
      description: 'Simulate property rejection notification',
      icon: <Home className="w-4 h-4" />,
      color: 'bg-red-500'
    },
    {
      id: 'visit_approved',
      title: 'Visit Approved',
      description: 'Simulate visit approval notification',
      icon: <Calendar className="w-4 h-4" />,
      color: 'bg-green-500'
    },
    {
      id: 'auction_bid',
      title: 'Auction Bid',
      description: 'Simulate new auction bid notification',
      icon: <Gavel className="w-4 h-4" />,
      color: 'bg-purple-500'
    },
    {
      id: 'auction_won',
      title: 'Auction Won',
      description: 'Simulate auction winner notification',
      icon: <Gavel className="w-4 h-4" />,
      color: 'bg-yellow-500'
    },
    {
      id: 'inquiry_received',
      title: 'Inquiry Received',
      description: 'Simulate new inquiry notification',
      icon: <Send className="w-4 h-4" />,
      color: 'bg-blue-500'
    },
    {
      id: 'document_verified',
      title: 'Document Verified',
      description: 'Simulate document verification notification',
      icon: <FileCheck className="w-4 h-4" />,
      color: 'bg-green-500'
    },
    {
      id: 'websocket_test',
      title: 'WebSocket Test',
      description: 'Send direct WebSocket message',
      icon: <TestTube className="w-4 h-4" />,
      color: 'bg-indigo-500'
    }
  ];

  return (
    <div className="max-w-4xl mx-auto p-6 space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center space-x-2">
            <TestTube className="w-5 h-5" />
            <span>Event Bus Testing Panel</span>
          </CardTitle>
          <CardDescription>
            Test real-time events and notifications using the Event Bus system
          </CardDescription>
        </CardHeader>
        <CardContent>
          {/* Connection Status */}
          <div className="flex items-center justify-between mb-6 p-4 bg-gray-50 rounded-lg">
            <div className="flex items-center space-x-3">
              <div className="flex items-center space-x-2">
                {isConnected ? (
                  <Wifi className="w-5 h-5 text-green-500" />
                ) : (
                  <WifiOff className="w-5 h-5 text-red-500" />
                )}
                <span className="font-medium">WebSocket Status:</span>
              </div>
              <Badge variant="outline" className={`${getStatusColor()} text-white`}>
                {connectionStatus}
              </Badge>
            </div>
            <div className="text-sm text-gray-600">
              User ID: <code className="bg-gray-200 px-2 py-1 rounded">{userId}</code>
            </div>
          </div>

          {/* Test Event Buttons */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 mb-6">
            {testEvents.map((event) => (
              <Card key={event.id} className="hover:shadow-md transition-shadow">
                <CardContent className="p-4">
                  <div className="flex items-start space-x-3">
                    <div className={`p-2 rounded-lg ${event.color} text-white`}>
                      {event.icon}
                    </div>
                    <div className="flex-1">
                      <h4 className="font-medium text-sm">{event.title}</h4>
                      <p className="text-xs text-gray-600 mb-3">{event.description}</p>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => handleTestEvent(event.id, {})}
                        disabled={isLoading || !isConnected}
                        className="w-full"
                      >
                        {isLoading ? 'Sending...' : 'Test'}
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Response Display */}
          {lastResponse && (
            <div className="mt-6">
              <h4 className="font-medium mb-2">Last Response:</h4>
              <pre className="bg-gray-100 p-4 rounded-lg text-sm overflow-x-auto">
                {lastResponse}
              </pre>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
};