import React, { useState, useEffect } from 'react';
import { Flag, Sparkles } from 'lucide-react';

interface FeatureFlag {
  name: string;
  description: string;
  category: string;
  is_enabled: boolean;
}

export function FeatureFlagsSection() {
  const [flags, setFlags] = useState<FeatureFlag[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    console.log('FeatureFlagsSection mounted');
    fetchUserFlags();
  }, []);

  const fetchUserFlags = async () => {
    try {
      const token = localStorage.getItem('access_token');
      if (!token) {
        console.log('No access token found');
        setLoading(false);
        return;
      }
      
      console.log('Fetching user flags...');
      const res = await fetch('http://localhost:8090/api/admin/feature-flags/my-flags/', {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      
      console.log('Response status:', res.status);
      const data = await res.json();
      console.log('User flags data:', data);
      
      setFlags(data.flags || []);
    } catch (error) {
      console.error('Error fetching user flags:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) return null;
  
  console.log('Rendering FeatureFlagsSection with flags:', flags);
  
  if (flags.length === 0) return null;

  return (
    <div className="bg-gradient-to-r from-purple-50 to-blue-50 border border-purple-200 rounded-xl p-6 mb-6">
      <div className="flex items-center gap-3 mb-4">
        <div className="bg-purple-600 p-2 rounded-lg">
          <Sparkles className="w-5 h-5 text-white" />
        </div>
        <div>
          <h3 className="text-lg font-bold text-gray-900">Exclusive Features</h3>
          <p className="text-sm text-gray-600">Special features enabled for you</p>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
        {flags.map((flag) => (
          <div
            key={flag.name}
            className="bg-white border border-purple-200 rounded-lg p-4 hover:shadow-md transition-all"
          >
            <div className="flex items-start gap-3">
              <div className="bg-purple-100 p-2 rounded-lg">
                <Flag className="w-4 h-4 text-purple-600" />
              </div>
              <div className="flex-1">
                <h4 className="font-semibold text-gray-900 text-sm">
                  {flag.name.replace(/_/g, ' ').toUpperCase()}
                </h4>
                <p className="text-xs text-gray-600 mt-1">{flag.description}</p>
                <span className="inline-block mt-2 px-2 py-1 text-xs font-bold rounded-full bg-green-100 text-green-700">
                  Active
                </span>
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}
