import React, { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import { api } from "../utils/api";
import { PropertyCard } from "./PropertyCard";
import { getPropertyMedia, generateDownloadURL } from '../api/mediaUpload';
import { ScrollFadeIn, ScrollStagger, ScrollStaggerItem } from './ui/ScrollAnimation';

interface Property {
  id: number;
  title: string;
  location: string;
  beds: string | number;
  baths: string | number;
  area: string;
  price: string;
  imageUrl: string;
  listingType: string;
  urgent_pricing?: number;
  sale_price?: number;
  priceNegotiable?: boolean;
  loanAvailable?: boolean;
}

export function FeaturedProperties() {
  const navigate = useNavigate();
  const [allProperties, setAllProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);
  const [clickedHearts, setClickedHearts] = useState<Set<number>>(new Set());

  useEffect(() => {
    const loadTrendingProperties = async () => {
      try {
        setLoading(true);
        
        // Get user's location
        const sellProperties = await api.fetchSellProperties();
        const formattedProperties = await Promise.all(
          sellProperties.slice(0, 3).map(async (property: any) => {
            let imageUrl = property.primary_image || property.photo;
            
            // Keep database URLs as-is (they're already correct)
            // No conversion needed - just pass through
            
            // Try to get real image from Azure Blob storage
            if (!imageUrl) {
              try {
                const mediaData = await getPropertyMedia(property.id.toString());
                if (mediaData?.images && mediaData.images.length > 0) {
                  const primaryImage = mediaData.images.find((img: any) => img.is_primary) || mediaData.images[0];
                  const downloadResponse = await generateDownloadURL(primaryImage.id);
                  imageUrl = downloadResponse?.download_url || downloadResponse?.url || downloadResponse;
                }
              } catch (error) {
                console.log('Could not fetch media for property', property.id);
              }
            }
            
            return {
              id: property.id,
              title: property.title,
              location: `${property.locality}, ${property.city}`,
              beds: property.bedrooms || 'N/A',
              baths: property.bathrooms || 'N/A', 
              area: property.built_up_area ? `${property.built_up_area} sq ft` : 'N/A',
              price: formatPrice(parseFloat(
                property.listing_type === 'urgent-sale' && property.urgent_pricing 
                  ? property.urgent_pricing 
                  : property.sale_price
              )),
              imageUrl,
              listingType: property.listing_type,
              priceNegotiable: property.price_negotiable || false,
              loanAvailable: property.loan_available || false
            };
          })
        );
        
        setAllProperties(formattedProperties);
      } catch (error) {
        console.log('Error loading properties:', error);
        setAllProperties([]);
      } finally {
        setLoading(false);
      }
    };
    loadTrendingProperties();
  }, []);

  const formatPrice = (price: number) => {
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(2)} Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(2)} L`;
    return `₹${price.toLocaleString()}`;
  };

  const handleFavorite = async (propertyId: number) => {
    setClickedHearts(prev => new Set(prev).add(propertyId));
    
    const token = localStorage.getItem('access_token');
    if (token) {
      try {
        await fetch('http://localhost:8000/api/add-to-wishlist/', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`
          },
          body: JSON.stringify({ property_id: propertyId })
        });
      } catch (error) {
        console.error('Error adding to wishlist:', error);
      }
    } else {
      localStorage.setItem('pending_wishlist', propertyId.toString());
      setTimeout(() => navigate('/login'), 300);
    }
  };

  const refreshProperties = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/properties/trending', {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        const result = await response.json();
        const trendingProperties = result.properties || [];
        
        const formattedProperties = await Promise.all(
          trendingProperties.slice(0, 3).map(async (property: any) => {
            let imageUrl = property.primary_image || property.photo;
            
            // Convert database URLs to relative URLs to avoid CSP issues
            if (imageUrl && !imageUrl.startsWith('http')) {
              imageUrl = imageUrl.startsWith('/') ? imageUrl : '/' + imageUrl;
            }
            
            // Try to get real image from Azure Blob storage
            if (!imageUrl) {
              try {
                const mediaData = await getPropertyMedia(property.id.toString());
                if (mediaData?.images && mediaData.images.length > 0) {
                  const primaryImage = mediaData.images.find((img: any) => img.is_primary) || mediaData.images[0];
                  const downloadResponse = await generateDownloadURL(primaryImage.id);
                  imageUrl = downloadResponse?.download_url || downloadResponse?.url || downloadResponse;
                }
              } catch (error) {
                console.log('Could not fetch media for property', property.id);
              }
            }
            
            return {
              id: property.id,
              title: property.title,
              location: `${property.locality}, ${property.city}`,
              beds: property.bedrooms || 'N/A',
              baths: property.bathrooms || 'N/A', 
              area: property.built_up_area ? `${property.built_up_area} sq ft` : 'N/A',
              price: formatPrice(parseFloat(
                property.listing_type === 'urgent-sale' && property.urgent_pricing 
                  ? property.urgent_pricing 
                  : property.sale_price
              )),
              imageUrl,
              listingType: property.listing_type,
              priceNegotiable: property.price_negotiable || false,
              loanAvailable: property.loan_available || false
            };
          })
        );
        
        setAllProperties(formattedProperties);
      } else {
        setAllProperties([]);
      }
    } catch (error) {
      console.log('Error refreshing properties:', error);
      setAllProperties([]);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div style={{ backgroundColor: '#eff6ff', width: '100%', paddingTop: '2.5rem', paddingBottom: '2.5rem' }}>
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 featured-properties">
      {/* Featured Properties Heading */}
      <ScrollFadeIn>
        <div className="text-center mb-12">
          <div className="flex items-center justify-center gap-3 mb-3">
            <svg className="w-10 h-10 text-blue-600" fill="currentColor" viewBox="0 0 20 20">
              <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
            </svg>
            <h2 className="text-4xl font-bold text-gray-900">Featured Properties</h2>
          </div>
          <div className="w-24 h-1 bg-gradient-to-r from-blue-400 via-blue-600 to-blue-400 mx-auto mb-4 rounded-full"></div>
          <p className="text-lg text-gray-600 max-w-3xl mx-auto">Discover our verified, handpicked selection of premium properties designed for modern living.</p>
        </div>
      </ScrollFadeIn>
      
      <ScrollStagger className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 justify-items-center min-h-[500px]" staggerDelay={0.15}>
        {loading ? (
          <div className="col-span-full text-center py-8">
            <p className="text-gray-500">Loading properties...</p>
          </div>
        ) : allProperties.length > 0 ? allProperties.map((property: Property) => (
          <ScrollStaggerItem key={property.id}>
            <PropertyCard
            key={property.id}
            id={property.id}
            title={property.title}
            location={property.location}
            beds={property.beds}
            baths={property.baths}
            area={property.area}
            price={property.price}
            imageUrl={property.imageUrl}
            onFavorite={handleFavorite}
            isFavorited={clickedHearts.has(property.id)}
            priceNegotiable={property.priceNegotiable}
            loanAvailable={property.loanAvailable}
            propertyData={{
              id: property.id.toString(),
              title: property.title,
              location: property.location,
              beds: typeof property.beds === 'string' ? parseInt(property.beds) || 0 : property.beds,
              baths: typeof property.baths === 'string' ? parseInt(property.baths) || 0 : property.baths,
              area: property.area,
              price: property.price,
              image: property.imageUrl || '',
              images: property.imageUrl ? [property.imageUrl] : [],
              verified: true,
              riblScore: 'A+',
              description: '',
              amenities: [],
              nearbyPlaces: [],
              priceHistory: [],
              agent: {
                name: '',
                rating: 0,
                experience: '',
                phone: '',
                image: ''
              },
              specifications: {
                propertyType: 'apartment',
                facing: '',
                floor: '',
                totalFloors: '',
                age: '',
                furnishing: '',
                parking: 0
              },
              legalInfo: {
                approvals: [],
                rera: '',
                ownership: ''
              }
            }}
          />
          </ScrollStaggerItem>
        )) : (
          <div className="col-span-full text-center py-8">
            <p className="text-gray-500">No properties available for sale at the moment.</p>
          </div>
        )}
      </ScrollStagger>
    </div>
    </div>
  );
}