import React, { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import { getProperties } from "../api/properties";

// Static fallback data - always available
const staticProperties = [
  {
    id: 1,
    title: "Sattva Vasanta Skye",
    location: "Devanahalli, North Bangalore",
    beds: 3,
    baths: 2,
    area: "1,200 sq ft",
    price: "₹1.2 Cr",
    imageUrl: "https://images.unsplash.com/photo-1564013799919-ab600027ffc6?w=400&h=300&fit=crop",
    isStatic: true
  },
  {
    id: 2,
    title: "Prestige Lakeside Habitat",
    location: "Whitefield, Bangalore",
    beds: 4,
    baths: 3,
    area: "2,800 sq ft",
    price: "₹2.5 Cr",
    imageUrl: "https://images.unsplash.com/photo-1570129477492-45c003edd2be?w=400&h=300&fit=crop",
    isStatic: true
  }
];

interface RealProperty {
  id: number;
  title: string;
  locality: string;
  city: string;
  sale_price: number;
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
  photo?: string;
  property_type: string;
}

export function FeaturedPropertiesSafe() {
  const navigate = useNavigate();
  const [allProperties, setAllProperties] = useState(staticProperties);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    const loadRealProperties = async () => {
      try {
        setLoading(true);
        const data = await getProperties();
        const sellProperties = data.properties
          .filter((property: RealProperty) => property.listing_type === 'sell')
          .map((property: RealProperty) => ({
            id: property.id,
            title: property.title,
            location: `${property.locality || ''}${property.locality ? ', ' : ''}${property.city || ''}`,
            beds: property.bedrooms || 'N/A',
            baths: property.bathrooms || 'N/A',
            area: property.built_up_area ? `${property.built_up_area} sq ft` : 'N/A',
            price: formatPrice(property.sale_price),
            imageUrl: property.photo || "https://images.unsplash.com/photo-1564013799919-ab600027ffc6?w=400&h=300&fit=crop",
            isStatic: false
          }));

        // Combine static and real properties
        setAllProperties([...staticProperties, ...sellProperties]);
        console.log('✅ Successfully loaded real properties:', sellProperties.length);
      } catch (error) {
        console.log('⚠️ Error loading real properties, using static data only:', error);
      } finally {
        setLoading(false);
      }
    };

    loadRealProperties();
  }, []);

  const formatPrice = (price: number) => {
    if (price >= 10000000) {
      return `₹${(price / 10000000).toFixed(1)} Cr`;
    } else if (price >= 100000) {
      return `₹${(price / 100000).toFixed(0)} L`;
    } else {
      return `₹${price.toLocaleString()}`;
    }
  };

  const handleViewMore = (id: number) => {
    navigate(`/property/${id}`);
  };

  return (
    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 bg-gray-50">
      <h2 className="text-3xl font-bold mb-8 text-center text-gray-800">
        Featured Properties
        {loading && <span className="text-sm text-gray-500 ml-2">(Loading...)</span>}
      </h2>
      <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 gap-8">
        {allProperties.map((property) => (
          <div key={property.id} className="bg-white rounded-xl shadow-lg overflow-hidden relative transform transition-all duration-300 hover:scale-105 hover:shadow-2xl">
            <div className="relative">
              <img src={property.imageUrl} alt={property.title} className="w-full h-48 object-cover transition-transform duration-300 hover:scale-110" />
              <div className="absolute top-3 left-3 bg-teal-500 text-white text-xs font-semibold px-3 py-1 rounded-full shadow-md">
                NAL Verified
              </div>
              <div className="absolute top-3 right-3 bg-white text-xs font-semibold px-3 py-1 rounded-full shadow-md border">
                {property.isStatic ? 'DEMO' : 'REAL'}
              </div>
              <button className="absolute bottom-3 right-3 bg-white p-2 rounded-full shadow-lg hover:bg-red-50 hover:shadow-xl transition-all duration-200" aria-label="Favorite">
                <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5 text-gray-600 hover:text-red-500 transition-colors duration-200" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                </svg>
              </button>
            </div>
            <div className="p-5">
              <h3 className="text-lg font-bold text-gray-800 mb-1">{property.title}</h3>
              <p className="text-gray-500 text-sm mb-3">{property.location}</p>
              <div className="flex items-center justify-between mt-3 text-gray-600 text-sm mb-4">
                <div className="flex items-center space-x-1">
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                  </svg>
                  <span>{property.beds} Beds</span>
                </div>
                <div className="flex items-center space-x-1">
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                  </svg>
                  <span>{property.baths} Baths</span>
                </div>
                <div className="flex items-center space-x-1">
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                  </svg>
                  <span>{property.area}</span>
                </div>
              </div>
              <div className="text-2xl font-bold text-blue-600 mb-4">{property.price}</div>
              <button onClick={() => handleViewMore(property.id)} className="w-full bg-gradient-to-r from-orange-500 to-orange-600 text-white py-4 rounded-lg font-bold text-lg hover:from-orange-600 hover:to-orange-700 transition-all duration-300 shadow-lg hover:shadow-xl transform hover:-translate-y-1 border-2 border-orange-400 flex justify-center items-center space-x-2">
                <span>View More</span>
                <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2}>
                  <path strokeLinecap="round" strokeLinejoin="round" d="M9 5l7 7-7 7" />
                </svg>
              </button>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}