import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Calculator, CheckCircle, Ruler, Wallet, ArrowRight, FileText } from "lucide-react";
import { useState } from "react";
import { useNavigate } from "react-router-dom";
import { EMICalculatorModal } from "./EMICalculatorModal";
import { ScrollFadeIn, ScrollStagger, ScrollStaggerItem } from './ui/ScrollAnimation';

export function FinancialTools() {
  const [isEMIModalOpen, setIsEMIModalOpen] = useState(false);
  const navigate = useNavigate();
  const tools = [
    {
      title: "Budget Calculator",
      description: "Determine your affordable property price range based on income and expenses to avoid overcommitting financially",
      icon: Wallet,
      color: "bg-purple-600",
      features: ["Max EMI", "Max Loan", "Property Budget"],
      cta: "Check Budget",
      purpose: "Helps buyers estimate realistic affordability before property search, preventing financial overcommitment"
    },
    {
      title: "Loan Calculator",
      description: "Calculate monthly EMI and understand loan cost breakdown for better financial planning",
      icon: Calculator,
      color: "bg-[#0056D2]",
      features: ["EMI", "Total Interest", "Payment Breakdown"],
      cta: "Calculate Loan",
      purpose: "Shows exact EMI amounts and principal vs interest breakdown to help plan loan tenure and payments"
    },
    {
      title: "Loan Eligibility",
      description: "Assess maximum loan amount banks may offer based on your repayment capacity and DTI ratio",
      icon: CheckCircle,
      color: "bg-[#00BFA6]",
      features: ["Eligible Amount", "Max EMI", "DTI"],
      cta: "Check Eligibility",
      purpose: "Prevents loan rejections by showing eligibility before approaching lenders, evaluates financial health"
    },
    {
      title: "Area Converter",
      description: "Convert property measurements between different units for accurate comparison",
      icon: Ruler,
      color: "bg-orange-500",
      features: ["Sqft", "Sqm", "Acre/Ha"],
      cta: "Convert Area",
      purpose: "Essential for comparing properties listed in different units and construction planning"
    },
  ];

  return (
    <section className="py-16 bg-[#F5F5F5]">
      <div className="w-full px-4 sm:px-6 lg:px-8">
        <ScrollFadeIn>
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Financial Tools</h2>
            <p className="text-gray-600 max-w-2xl mx-auto">
              Simplify your property financing with our comprehensive financial tools and services
            </p>
          </div>
        </ScrollFadeIn>

        <ScrollStagger className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6" staggerDelay={0.1}>
          {tools.map((tool, index) => (
            <ScrollStaggerItem key={index}>
              <Card 
              key={index} 
              className="relative overflow-hidden group hover:shadow-xl transition-all duration-300 h-full flex flex-col"
            >
              
              <CardHeader className="text-center pb-4">
                <div className={`${tool.color} w-16 h-16 rounded-full flex items-center justify-center mx-auto mb-4`}>
                  <tool.icon className="w-8 h-8 text-white" />
                </div>
                <CardTitle className="text-xl">{tool.title}</CardTitle>
              </CardHeader>
              
              <CardContent className="text-center flex-1 flex flex-col">
                <p className="text-gray-600 mb-6">{tool.description}</p>
                
                <div className="space-y-3 mb-6 flex-1">
                  {tool.features.map((feature, featureIndex) => (
                    <div key={featureIndex} className="flex items-center justify-center space-x-2">
                      <CheckCircle className="w-4 h-4 text-[#00BFA6]" />
                      <span className="text-sm text-gray-700">{feature}</span>
                    </div>
                  ))}
                </div>
                
                <Button 
                  onClick={() => {
                    if (tool.title === "Budget Calculator") {
                      navigate('/budget-calculator');
                    } else if (tool.title === "Loan Calculator") {
                      navigate('/emi-calculator');
                    } else if (tool.title === "Loan Eligibility") {
                      navigate('/loan-eligibility');
                    } else if (tool.title === "Area Converter") {
                      navigate('/area-converter');
                    }
                  }}
                  className="w-full group bg-[#0056D2] hover:bg-[#0056D2]/90 text-white"
                >
                  {tool.cta}
                  <ArrowRight className="w-4 h-4 ml-2 group-hover:translate-x-1 transition-transform" />
                </Button>
              </CardContent>
            </Card>
            </ScrollStaggerItem>
          ))}
        </ScrollStagger>


      </div>
      
      <EMICalculatorModal 
        isOpen={isEMIModalOpen} 
        onClose={() => setIsEMIModalOpen(false)} 
      />
    </section>
  );
}