import React, { useState, useEffect } from 'react';
import { Waves, Shield, AlertTriangle, Droplets, Clock, TrendingUp } from 'lucide-react';

interface FloodRiskData {
  riskScore: number;
  severity: 'High' | 'Medium' | 'Low';
  confidence: number;
  events: any[];
  recommendation: string;
}

interface FloodRiskCardProps {
  lat: number;
  lon: number;
}

export const FloodRiskCard: React.FC<FloodRiskCardProps> = ({ lat, lon }) => {
  const [floodData, setFloodData] = useState<FloodRiskData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchFloodRisk();
  }, [lat, lon]);

  const fetchFloodRisk = async () => {
    try {
      setLoading(true);
      setError(null);
      
      const response = await fetch(`http://localhost:8090/api/flood/risk?lat=${lat}&lon=${lon}`);
      
      if (!response.ok) {
        throw new Error('Failed to fetch flood risk data');
      }
      
      const data = await response.json();
      setFloodData(data);
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Unknown error occurred');
    } finally {
      setLoading(false);
    }
  };

  const getSeverityConfig = (severity: string) => {
    switch (severity.toLowerCase()) {
      case 'high':
        return {
          color: 'text-red-600',
          bg: 'bg-red-100',
          icon: AlertTriangle,
          gradient: 'from-red-50 to-rose-50'
        };
      case 'medium':
        return {
          color: 'text-orange-600',
          bg: 'bg-orange-100',
          icon: AlertTriangle,
          gradient: 'from-orange-50 to-amber-50'
        };
      default:
        return {
          color: 'text-green-600',
          bg: 'bg-green-100',
          icon: Shield,
          gradient: 'from-green-50 to-emerald-50'
        };
    }
  };

  if (loading) {
    return (
      <div className="bg-white rounded-2xl shadow-xl p-6 border border-gray-100">
        <div className="animate-pulse">
          <div className="bg-gradient-to-br from-cyan-50 to-blue-50 rounded-2xl p-6 mb-6">
            <div className="flex items-center justify-between mb-4">
              <div className="flex-1">
                <div className="h-8 bg-blue-200 rounded-lg w-64 mb-4"></div>
                <div className="flex items-center gap-4">
                  <div className="h-12 bg-blue-200 rounded-lg w-20"></div>
                  <div className="h-8 bg-blue-200 rounded-full w-24"></div>
                </div>
              </div>
              <div className="w-16 h-16 bg-blue-200 rounded-full"></div>
            </div>
          </div>
          <div className="grid grid-cols-3 gap-4">
            <div className="h-20 bg-gray-200 rounded-xl"></div>
            <div className="h-20 bg-gray-200 rounded-xl"></div>
            <div className="h-20 bg-gray-200 rounded-xl"></div>
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="bg-white rounded-2xl shadow-xl p-6 border border-red-200">
        <div className="bg-gradient-to-br from-red-50 to-rose-50 rounded-2xl p-6">
          <div className="flex items-center justify-between">
            <div className="flex-1">
              <div className="flex items-center gap-3 mb-3">
                <AlertTriangle className="h-8 w-8 text-red-600" />
                <h3 className="text-2xl font-bold tracking-tight text-red-800">Flood Risk Assessment</h3>
              </div>
              <p className="text-red-600 font-medium">{error}</p>
            </div>
            <Waves className="h-16 w-16 text-red-400 opacity-50" />
          </div>
        </div>
      </div>
    );
  }

  if (!floodData) return null;

  const severityConfig = getSeverityConfig(floodData.severity);
  const IconComponent = severityConfig.icon;

  return (
    <div className="bg-white rounded-2xl shadow-xl p-6 space-y-6 transition-all duration-300">
      {/* Header Card */}
      <div className={`bg-gradient-to-br ${severityConfig.gradient} rounded-2xl p-6 border ${severityConfig.color.replace('text-', 'border-').replace('-600', '-200')}`}>
        <div className="flex items-center justify-between">
          <div className="flex-1">
            <div className="flex items-center gap-3 mb-4">
              <Droplets className="h-8 w-8 text-blue-600" />
              <h2 className="text-2xl font-bold tracking-tight text-gray-900">Flood Risk Assessment</h2>
            </div>
            <div className="flex items-center gap-6 mb-4">
              <div className="text-5xl font-bold text-gray-900">
                {floodData.riskScore}
              </div>
              <div className="flex flex-col gap-2">
                <span className="text-lg text-gray-600">/10</span>
                <div className={`px-4 py-1.5 rounded-full text-sm font-semibold ${severityConfig.bg} ${severityConfig.color} shadow-md`}>
                  {floodData.severity} Risk
                </div>
              </div>
            </div>
            <p className="text-gray-700 text-sm font-medium italic leading-relaxed">
              {floodData.recommendation}
            </p>
          </div>
          <div className="hidden md:flex flex-col items-center gap-2">
            <IconComponent className={`h-16 w-16 ${severityConfig.color} opacity-80`} />
            <Waves className="h-10 w-10 text-blue-500 opacity-60" />
          </div>
        </div>
      </div>

      {/* Info Grid */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div className="bg-gradient-to-br from-blue-50 to-cyan-50 rounded-xl shadow-md p-4 border border-blue-100 hover:shadow-lg transition-all duration-300">
          <div className="flex items-center gap-3 mb-2">
            <div className="p-2 bg-blue-500 rounded-full">
              <AlertTriangle className="w-4 h-4 text-white" />
            </div>
            <span className="text-sm font-semibold text-blue-700">Flood Alerts</span>
          </div>
          <div className="text-2xl font-bold text-blue-800">{floodData.events.length}</div>
        </div>
        
        <div className="bg-gradient-to-br from-emerald-50 to-green-50 rounded-xl shadow-md p-4 border border-emerald-100 hover:shadow-lg transition-all duration-300">
          <div className="flex items-center gap-3 mb-2">
            <div className="p-2 bg-emerald-500 rounded-full">
              <TrendingUp className="w-4 h-4 text-white" />
            </div>
            <span className="text-sm font-semibold text-emerald-700">Confidence</span>
          </div>
          <div className="text-2xl font-bold text-emerald-800">{floodData.confidence}%</div>
        </div>
        
        <div className="bg-gradient-to-br from-purple-50 to-indigo-50 rounded-xl shadow-md p-4 border border-purple-100 hover:shadow-lg transition-all duration-300">
          <div className="flex items-center gap-3 mb-2">
            <div className="p-2 bg-purple-500 rounded-full">
              <Clock className="w-4 h-4 text-white" />
            </div>
            <span className="text-sm font-semibold text-purple-700">Last Updated</span>
          </div>
          <div className="text-sm font-bold text-purple-800">Real-time</div>
        </div>
      </div>

      {/* Footer */}
      <div className="bg-gray-50 rounded-2xl p-4 border border-gray-100">
        <div className="flex items-center justify-center gap-2">
          <Waves className="w-4 h-4 text-blue-500" />
          <p className="text-xs text-gray-600 font-medium">
            Data provided by Ambee Weather Intelligence
          </p>
        </div>
      </div>
    </div>
  );
};