import React, { useEffect, useRef, useState } from 'react';
import { Droplets } from 'lucide-react';

interface FloodZone {
  state: string;
  district: string;
  lat: number;
  lon: number;
  flood_risk: number;
  intensity: number;
  risk_score: number;
  distance_km?: number;
}

interface FloodRiskMapProps {
  propertyLat: number;
  propertyLon: number;
  propertyName: string;
}

declare global {
  interface Window {
    google: any;
  }
}

export const FloodRiskMap: React.FC<FloodRiskMapProps> = ({
  propertyLat,
  propertyLon,
  propertyName
}) => {
  const mapRef = useRef<HTMLDivElement>(null);
  const [floodZones, setFloodZones] = useState<FloodZone[]>([]);
  const [loading, setLoading] = useState(true);
  const [floodRiskScore, setFloodRiskScore] = useState<number>(0);
  const [radius, setRadius] = useState<number>(200);

  useEffect(() => {
    fetchFloodData();
  }, [propertyLat, propertyLon, radius]);

  const fetchFloodData = async () => {
    try {
      setLoading(true);
      const response = await fetch(
        `http://localhost:8090/api/flood/nearby?lat=${propertyLat}&lon=${propertyLon}&radius=${radius}`
      );
      
      if (response.ok) {
        const data = await response.json();
        setFloodZones(data.zones || []);
        setFloodRiskScore(data.flood_risk_score || 0);
      }
    } catch (error) {
      console.error('Error fetching flood data:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (!mapRef.current || !window.google || loading) return;

    const map = new window.google.maps.Map(mapRef.current, {
      center: { lat: propertyLat, lng: propertyLon },
      zoom: 8,
      mapTypeId: 'roadmap',
    });

    // Property marker
    new window.google.maps.Marker({
      position: { lat: propertyLat, lng: propertyLon },
      map: map,
      title: propertyName,
      icon: {
        url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
          <svg width="32" height="32" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg">
            <circle cx="16" cy="16" r="12" fill="#2563eb" stroke="white" stroke-width="3"/>
            <text x="16" y="20" text-anchor="middle" fill="white" font-size="16">🏠</text>
          </svg>
        `),
        scaledSize: new window.google.maps.Size(32, 32),
      }
    });

    // Create heatmap data
    const heatmapData = floodZones.map(zone => ({
      location: new window.google.maps.LatLng(zone.lat, zone.lon),
      weight: zone.intensity / 15
    }));

    // Add heatmap layer
    if (heatmapData.length > 0) {
      const heatmap = new window.google.maps.visualization.HeatmapLayer({
        data: heatmapData,
        radius: 35,
        opacity: 0.6,
        maxIntensity: 1,
        gradient: [
          'rgba(0, 255, 0, 0)',
          'rgba(135, 206, 250, 1)',
          'rgba(30, 144, 255, 1)',
          'rgba(0, 0, 255, 1)'
        ]
      });
      heatmap.setMap(map);
    }

    // Add zone markers
    floodZones.forEach((zone) => {
      const color = zone.intensity >= 12 ? '#1e40af' : 
                    zone.intensity >= 9 ? '#3b82f6' : '#60a5fa';
      
      const marker = new window.google.maps.Marker({
        position: { lat: zone.lat, lng: zone.lon },
        map: map,
        icon: {
          path: window.google.maps.SymbolPath.CIRCLE,
          scale: 8,
          fillColor: color,
          fillOpacity: 0.7,
          strokeColor: 'white',
          strokeWeight: 2,
        }
      });

      const infoWindow = new window.google.maps.InfoWindow({
        content: `
          <div style="padding: 8px;">
            <h4 style="margin: 0 0 4px 0; font-weight: bold;">${zone.district}, ${zone.state}</h4>
            <p style="margin: 2px 0; font-size: 12px;">Flood Risk: ${zone.flood_risk}</p>
            <p style="margin: 2px 0; font-size: 12px;">Intensity: ${zone.intensity}/15</p>
            ${zone.distance_km ? `<p style="margin: 2px 0; font-size: 12px;">Distance: ${zone.distance_km}km</p>` : ''}
          </div>
        `
      });

      marker.addListener('click', () => {
        infoWindow.open(map, marker);
      });
    });

  }, [propertyLat, propertyLon, floodZones, loading]);

  const getRiskLevel = (score: number) => {
    if (score >= 8) return { label: 'Very Low Risk', color: 'text-green-600', bg: 'bg-green-100' };
    if (score >= 6) return { label: 'Low Risk', color: 'text-blue-600', bg: 'bg-blue-100' };
    if (score >= 4) return { label: 'Moderate Risk', color: 'text-yellow-600', bg: 'bg-yellow-100' };
    return { label: 'High Risk', color: 'text-red-600', bg: 'bg-red-100' };
  };

  const risk = getRiskLevel(floodRiskScore);
  const lowRisk = floodZones.filter(z => z.intensity <= 8).length;
  const mediumRisk = floodZones.filter(z => z.intensity > 8 && z.intensity <= 11).length;
  const highRisk = floodZones.filter(z => z.intensity > 11).length;

  return (
    <div className="space-y-4">
      {/* Flood Risk Score Card */}
      <div className="bg-gradient-to-r from-blue-50 to-cyan-100 rounded-xl p-6 shadow-lg">
        <div className="flex items-center justify-between">
          <div>
            <h3 className="text-xl font-bold text-gray-800 mb-2">Flood Risk Score</h3>
            <div className="flex items-center gap-3">
              <div className="text-4xl font-bold text-gray-800">{floodRiskScore.toFixed(1)}/10</div>
              <div className={`px-4 py-2 rounded-full ${risk.bg}`}>
                <span className={`font-semibold ${risk.color}`}>{risk.label}</span>
              </div>
            </div>
            <p className="text-sm text-gray-600 mt-2">Based on NDMA flood vulnerability data</p>
          </div>
          <Droplets className="h-16 w-16 text-blue-500" />
        </div>
      </div>

      {/* Flood Statistics */}
      <div className="grid grid-cols-3 gap-4">
        <div className="bg-white rounded-lg p-4 shadow-sm border border-gray-100">
          <div className="flex items-center gap-2 mb-2">
            <div className="w-3 h-3 bg-blue-300 rounded-full"></div>
            <span className="text-sm font-medium text-gray-600">Low Risk Districts</span>
          </div>
          <div className="text-2xl font-bold text-gray-800">{lowRisk}</div>
          <p className="text-xs text-gray-500 mt-1">Intensity ≤ 8</p>
        </div>
        
        <div className="bg-white rounded-lg p-4 shadow-sm border border-gray-100">
          <div className="flex items-center gap-2 mb-2">
            <div className="w-3 h-3 bg-blue-500 rounded-full"></div>
            <span className="text-sm font-medium text-gray-600">Medium Risk Districts</span>
          </div>
          <div className="text-2xl font-bold text-gray-800">{mediumRisk}</div>
          <p className="text-xs text-gray-500 mt-1">Intensity 9-11</p>
        </div>
        
        <div className="bg-white rounded-lg p-4 shadow-sm border border-gray-100">
          <div className="flex items-center gap-2 mb-2">
            <div className="w-3 h-3 bg-blue-800 rounded-full"></div>
            <span className="text-sm font-medium text-gray-600">High Risk Districts</span>
          </div>
          <div className="text-2xl font-bold text-gray-800">{highRisk}</div>
          <p className="text-xs text-gray-500 mt-1">Intensity ≥ 12</p>
        </div>
      </div>

      {/* Radius Filter */}
      <div className="bg-white rounded-lg p-4 shadow-sm border border-gray-100">
        <div className="flex items-center gap-2 mb-3">
          <span className="text-sm font-medium text-gray-700">Show districts within:</span>
          {[50, 100, 200, 500].map((dist) => (
            <button
              key={dist}
              onClick={() => setRadius(dist)}
              className={`px-4 py-2 rounded-full text-sm font-medium transition-all ${
                radius === dist
                  ? 'bg-blue-500 text-white shadow-md'
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
              }`}
            >
              {dist}km
            </button>
          ))}
        </div>
      </div>

      {/* Map */}
      <div className="bg-white rounded-lg p-4 shadow-sm border border-gray-100">
        <h3 className="text-lg font-semibold mb-3">Flood Risk Heatmap</h3>
        <div
          ref={mapRef}
          className="w-full h-96 rounded-lg border"
          style={{ minHeight: '400px' }}
        />
        <div className="mt-3 flex items-center justify-center gap-6 text-sm">
          <div className="flex items-center gap-2">
            <div className="w-4 h-4 bg-blue-600 rounded-full"></div>
            <span>Property Location</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-4 h-4 bg-blue-300 rounded-full"></div>
            <span>Low Risk</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-4 h-4 bg-blue-500 rounded-full"></div>
            <span>Medium Risk</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-4 h-4 bg-blue-800 rounded-full"></div>
            <span>High Risk</span>
          </div>
        </div>
      </div>

      {/* Disclaimer */}
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
        <p className="text-sm text-blue-800">
          <strong>Note:</strong> Flood risk scores are calculated using inverse distance weighting from nearby flood-prone districts. 
          Data is based on NDMA flood vulnerability index and CWC flood zones.
        </p>
      </div>
    </div>
  );
};
