import { useState, useEffect } from "react";
import { useNavigate, useSearchParams } from "react-router-dom";
import { toast } from "sonner";
import { sendForgotPasswordEmail, validateResetToken, resetPassword } from '../api/auth';
import '../styles/animated-auth.css';

interface ForgotPasswordProps {
  userType?: 'buyer' | 'seller' | 'agent';
}

export function ForgotPassword({ userType = 'buyer' }: ForgotPasswordProps) {
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const token = searchParams.get('token');
  
  // States
  const [step, setStep] = useState<'email' | 'reset'>('email');
  const [email, setEmail] = useState('');
  const [newPassword, setNewPassword] = useState('');
  const [confirmPassword, setConfirmPassword] = useState('');
  const [loading, setLoading] = useState(false);
  const [success, setSuccess] = useState(false);
  const [error, setError] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const [passwordValidation, setPasswordValidation] = useState({ isValid: false, errors: [], strength: 0 });
  const [showPasswordHelp, setShowPasswordHelp] = useState(false);
  const [tokenValidated, setTokenValidated] = useState(false);
  const [maskedEmail, setMaskedEmail] = useState('');

  // Initialize step based on token presence
  useEffect(() => {
    if (token) {
      setStep('reset');
      validateToken();
    }
  }, [token]);

  const validateEmail = (email: string): boolean => {
    const emailRegex = /^[a-zA-Z0-9._%+-]+@[a-zA-Z.-]+\.[a-zA-Z]{2,}$/;
    if (!emailRegex.test(email)) return false;
    
    const allowedDomains = ['gmail.com', 'yahoo.com', 'outlook.com', 'hotmail.com', 'live.com', 'icloud.com', 'proton.me', 'protonmail.com', 'zoho.com', 'gmx.com', 'mail.com', 'yandex.com', 'rediffmail.com', 'aol.com', 'msn.com', 'mit.edu', 'stanford.edu', 'harvard.edu', 'iitd.ac.in', 'iith.ac.in', 'vit.ac.in', 'ox.ac.uk', 'utoronto.ca', 'alstonair.com', 'sitams.in'];
    const domain = email.split('@')[1]?.toLowerCase();
    return allowedDomains.includes(domain);
  };

  const validatePassword = (password: string, email?: string): { isValid: boolean; errors: string[]; strength: number } => {
    const errors: string[] = [];
    let strength = 0;
    
    if (password.length < 8) errors.push('At least 8 characters required');
    if (password.length > 128) errors.push('Maximum 128 characters allowed');
    else if (password.length >= 8) strength += 1;
    
    if (!/[a-z]/.test(password)) errors.push('At least one lowercase letter required');
    else strength += 1;
    
    if (!/[A-Z]/.test(password)) errors.push('At least one uppercase letter required');
    else strength += 1;
    
    if (!/\d/.test(password)) errors.push('At least one number required');
    else strength += 1;
    
    if (!/[!@#$%^&*(),.?":{}|<>]/.test(password)) errors.push('At least one special character required');
    else strength += 1;
    
    const commonPasswords = ['password', '123456', '123456789', 'qwerty', 'abc123', 'password123', '12345678', 'admin', 'letmein', 'welcome'];
    if (commonPasswords.includes(password.toLowerCase())) {
      errors.push('Password is too common');
      strength = 0;
    }
    
    if (email && password.toLowerCase().includes(email.split('@')[0].toLowerCase())) {
      errors.push('Password cannot contain your email');
      strength = Math.max(0, strength - 1);
    }
    
    return { isValid: errors.length === 0, errors, strength };
  };

  const getPasswordStrengthText = (strength: number): { text: string; color: string } => {
    if (strength <= 1) return { text: 'Very Weak', color: '#ef4444' };
    if (strength === 2) return { text: 'Weak', color: '#f97316' };
    if (strength === 3) return { text: 'Fair', color: '#eab308' };
    if (strength === 4) return { text: 'Good', color: '#22c55e' };
    return { text: 'Strong', color: '#16a34a' };
  };

  const validateToken = async () => {
    if (!token) return;
    
    setLoading(true);
    try {
      const data = await validateResetToken(token, userType);
      
      if (data.success) {
        setTokenValidated(true);
        setMaskedEmail(data.email || '');
      } else {
        toast.error(data.message || 'Invalid or expired reset link');
        navigate(`/${userType === 'buyer' ? 'login' : userType}`);
      }
    } catch (error: any) {
      toast.error(error.message || 'Failed to validate reset link');
      navigate(`/${userType === 'buyer' ? 'login' : userType}`);
    } finally {
      setLoading(false);
    }
  };

  const handleSendResetEmail = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!validateEmail(email)) {
      setError(true);
      setTimeout(() => {
        toast.error('Please enter a valid email address');
        setError(false);
      }, 500);
      return;
    }

    setLoading(true);
    setError(false);
    setSuccess(false);

    try {
      const data = await sendForgotPasswordEmail(email, userType);
      
      if (data.success) {
        setSuccess(true);
        setTimeout(() => {
          toast.success('Password reset link sent to your email!');
        }, 600);
      } else {
        setError(true);
        setTimeout(() => {
          toast.error(data.message || 'Failed to send reset email');
          setError(false);
        }, 500);
      }
    } catch (error: any) {
      setError(true);
      setTimeout(() => {
        toast.error(error.message || 'Failed to send reset email. Please try again.');
        setError(false);
      }, 500);
    } finally {
      setLoading(false);
    }
  };

  const handleResetPassword = async (e: React.FormEvent) => {
    e.preventDefault();
    
    const passwordCheck = validatePassword(newPassword, maskedEmail);
    if (!passwordCheck.isValid) {
      setError(true);
      setTimeout(() => {
        toast.error(passwordCheck.errors[0] || 'Invalid password');
        setError(false);
      }, 500);
      return;
    }

    if (newPassword !== confirmPassword) {
      setError(true);
      setTimeout(() => {
        toast.error('Passwords do not match');
        setError(false);
      }, 500);
      return;
    }

    setLoading(true);
    setError(false);
    setSuccess(false);

    try {
      const data = await resetPassword(token!, newPassword, confirmPassword, userType);
      
      if (data.success) {
        setSuccess(true);
        toast.success('Password reset successfully!');
        setTimeout(() => {
          navigate(`/${userType === 'buyer' ? 'login' : userType}`);
        }, 1500);
      } else {
        setError(true);
        toast.error(data.message || 'Failed to reset password');
        setTimeout(() => setError(false), 500);
      }
    } catch (error: any) {
      setError(true);
      setTimeout(() => {
        toast.error(error.message || 'Failed to reset password. Please try again.');
        setError(false);
      }, 500);
    } finally {
      setLoading(false);
    }
  };

  const getUserTypeTitle = () => {
    switch (userType) {
      case 'seller': return 'Seller';
      case 'agent': return 'Agent';
      default: return 'User';
    }
  };

  const getBackLink = () => {
    switch (userType) {
      case 'seller': return '/seller';
      case 'agent': return '/agent';
      default: return '/login';
    }
  };

  return (
    <div className="seller-auth-container" data-type={userType}>
      <div className="seller-auth-wrapper">
        {/* Left side illustration */}
        <div className="illustration-side">
          <div className="character-illustration"></div>
        </div>
        
        {/* Right side form */}
        <div className="form-side">
          <div className="container">
            <div className="form-box Login">
              {step === 'email' ? (
                <>
                  <h2>Forgot Password</h2>
                  <p style={{ color: '#6b7280', marginBottom: '24px', fontSize: '14px' }}>
                    Enter your email address and we'll send you a link to reset your password.
                  </p>
                  
                  <form onSubmit={handleSendResetEmail}>
                    <div className="input-box">
                      <label>Email Address</label>
                      <input 
                        type="email" 
                        value={email}
                        onChange={(e) => setEmail(e.target.value)}
                        className={error ? 'error' : success ? 'success' : ''}
                        placeholder="Enter your email address"
                        required 
                      />
                    </div>

                    <button 
                      className={`btn ${loading ? 'loading' : ''} ${success ? 'success' : ''}`} 
                      type="submit" 
                      disabled={loading || success}
                    >
                      {loading ? "Sending..." : success ? "Email Sent!" : "Send Reset Link"}
                    </button>

                    <div className="regi-link">
                      <p>
                        <button 
                          type="button"
                          onClick={() => navigate(getBackLink())}
                          style={{
                            background: 'none',
                            border: 'none',
                            color: '#667eea',
                            textDecoration: 'underline',
                            cursor: 'pointer',
                            fontSize: 'inherit'
                          }}
                        >
                          Back to Login
                        </button>
                      </p>
                    </div>
                  </form>
                </>
              ) : (
                <>
                  <h2>Reset Password</h2>
                  <p style={{ color: '#6b7280', marginBottom: '24px', fontSize: '14px' }}>
                    {maskedEmail && `Resetting password for ${maskedEmail}`}
                  </p>
                  
                  {tokenValidated ? (
                    <form onSubmit={handleResetPassword}>
                      <div className="input-box">
                        <label>New Password</label>
                        <div style={{ position: 'relative' }}>
                          <input 
                            type={showPassword ? 'text' : 'password'}
                            value={newPassword}
                            onChange={(e) => {
                              setNewPassword(e.target.value);
                              const validation = validatePassword(e.target.value, maskedEmail);
                              setPasswordValidation(validation);
                            }}
                            onFocus={() => setShowPasswordHelp(true)}
                            onBlur={() => setShowPasswordHelp(false)}
                            className={error ? 'error' : success ? 'success' : ''}
                            placeholder="Enter new password"
                            required 
                          />
                          <button
                            type="button"
                            onClick={() => setShowPassword(!showPassword)}
                            style={{
                              position: 'absolute',
                              right: '12px',
                              top: '50%',
                              transform: 'translateY(-50%)',
                              background: 'none',
                              border: 'none',
                              cursor: 'pointer',
                              color: '#6b7280'
                            }}
                          >
                            {showPassword ? (
                              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2">
                                <path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24"/>
                                <line x1="1" y1="1" x2="23" y2="23"/>
                              </svg>
                            ) : (
                              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2">
                                <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/>
                                <circle cx="12" cy="12" r="3"/>
                              </svg>
                            )}
                          </button>
                        </div>
                        {newPassword && (
                          <div className="password-strength" style={{ marginTop: '8px' }}>
                            <div style={{ display: 'flex', alignItems: 'center', gap: '8px' }}>
                              <div style={{ flex: 1, height: '4px', backgroundColor: '#e5e7eb', borderRadius: '2px' }}>
                                <div 
                                  style={{ 
                                    width: `${(passwordValidation.strength / 5) * 100}%`, 
                                    height: '100%', 
                                    backgroundColor: getPasswordStrengthText(passwordValidation.strength).color,
                                    borderRadius: '2px',
                                    transition: 'all 0.3s ease'
                                  }}
                                />
                              </div>
                              <span style={{ 
                                fontSize: '12px', 
                                color: getPasswordStrengthText(passwordValidation.strength).color,
                                fontWeight: '500'
                              }}>
                                {getPasswordStrengthText(passwordValidation.strength).text}
                              </span>
                            </div>
                            {(showPasswordHelp || passwordValidation.errors.length > 0) && (
                              <div style={{ marginTop: '4px' }}>
                                {passwordValidation.errors.map((error, index) => (
                                  <div key={index} style={{ fontSize: '11px', color: '#ef4444', marginTop: '2px' }}>
                                    • {error}
                                  </div>
                                ))}
                              </div>
                            )}
                          </div>
                        )}
                      </div>

                      <div className="input-box">
                        <label>Confirm New Password</label>
                        <div style={{ position: 'relative' }}>
                          <input 
                            type={showConfirmPassword ? 'text' : 'password'}
                            value={confirmPassword}
                            onChange={(e) => setConfirmPassword(e.target.value)}
                            className={error ? 'error' : success ? 'success' : ''}
                            placeholder="Confirm new password"
                            required 
                          />
                          <button
                            type="button"
                            onClick={() => setShowConfirmPassword(!showConfirmPassword)}
                            style={{
                              position: 'absolute',
                              right: '12px',
                              top: '50%',
                              transform: 'translateY(-50%)',
                              background: 'none',
                              border: 'none',
                              cursor: 'pointer',
                              color: '#6b7280'
                            }}
                          >
                            {showConfirmPassword ? (
                              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2">
                                <path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24"/>
                                <line x1="1" y1="1" x2="23" y2="23"/>
                              </svg>
                            ) : (
                              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2">
                                <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/>
                                <circle cx="12" cy="12" r="3"/>
                              </svg>
                            )}
                          </button>
                        </div>
                      </div>

                      <button 
                        className={`btn ${loading ? 'loading' : ''} ${success ? 'success' : ''}`} 
                        type="submit" 
                        disabled={loading || success}
                      >
                        {loading ? "Resetting..." : success ? "Password Reset!" : "Reset Password"}
                      </button>

                      <div className="regi-link">
                        <p>
                          <button 
                            type="button"
                            onClick={() => navigate(getBackLink())}
                            style={{
                              background: 'none',
                              border: 'none',
                              color: '#667eea',
                              textDecoration: 'underline',
                              cursor: 'pointer',
                              fontSize: 'inherit'
                            }}
                          >
                            Back to Login
                          </button>
                        </p>
                      </div>
                    </form>
                  ) : (
                    <div style={{ textAlign: 'center', padding: '40px 0' }}>
                      <div style={{ fontSize: '16px', color: '#6b7280' }}>
                        Validating reset link...
                      </div>
                    </div>
                  )}
                </>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}