import React, { useEffect, useRef, useState } from 'react';

interface Property {
  id: number;
  title: string;
  location: string;
  price: string;
  latitude?: number;
  longitude?: number;
  imageUrl?: string;
  listing_type?: string;
  area?: string;
}

interface GoogleMapProps {
  properties?: Property[];
  center?: {lat: number, lng: number} | null;
  onPropertySelect?: (property: Property) => void;
  address?: string;
  coordinates?: {lat: number, lng: number};
  onLocationSelect?: (location: {lat: number, lng: number}) => void;
  height?: string;
}

declare global {
  interface Window {
    google: any;
  }
}

export const GoogleMap: React.FC<GoogleMapProps> = ({ 
  properties = [], 
  center, 
  onPropertySelect, 
  address, 
  coordinates, 
  onLocationSelect, 
  height = '400px' 
}) => {
  // Add global function for property navigation
  React.useEffect(() => {
    (window as any).openPropertyDetail = (propertyId: number) => {
      window.location.href = `/property/${propertyId}`;
    };
    
    return () => {
      delete (window as any).openPropertyDetail;
    };
  }, []);
  const mapRef = useRef<HTMLDivElement>(null);
  const [map, setMap] = useState<any>(null);
  const [mapType, setMapType] = useState<'roadmap' | 'satellite'>('roadmap');
  const markersRef = useRef<any[]>([]);

  useEffect(() => {
    console.log('GoogleMap useEffect - window.google:', !!window.google, 'mapRef.current:', !!mapRef.current);
    
    if (window.google && mapRef.current) {
      console.log('Initializing map immediately');
      initializeMap();
    } else {
      console.log('Waiting for Google Maps to load...');
      // Wait for Google Maps to load
      const checkGoogleMaps = setInterval(() => {
        console.log('Checking Google Maps...', !!window.google, !!mapRef.current);
        if (window.google && mapRef.current) {
          console.log('Google Maps loaded, initializing...');
          initializeMap();
          clearInterval(checkGoogleMaps);
        }
      }, 100);
      
      // Cleanup interval after 10 seconds
      setTimeout(() => {
        clearInterval(checkGoogleMaps);
        console.log('Timeout: Google Maps failed to load');
      }, 10000);
    }
  }, []);

  useEffect(() => {
    if (map) {
      updateMarkers();
    }
  }, [properties, map]);

  useEffect(() => {
    if (map && center) {
      map.setCenter(center);
      map.setZoom(12);
    }
  }, [center, map]);

  useEffect(() => {
    if (map) {
      map.setMapTypeId(mapType);
    }
  }, [mapType, map]);

  const initializeMap = () => {
    try {
      console.log('Creating Google Map instance...');
      const defaultCenter = { lat: 12.9716, lng: 77.5946 }; // Bangalore
      
      const mapInstance = new window.google.maps.Map(mapRef.current, {
        zoom: 11,
        center: defaultCenter,
        mapTypeId: mapType,
        styles: [
          {
            featureType: 'poi',
            elementType: 'labels',
            stylers: [{ visibility: 'off' }]
          }
        ],
        mapTypeControl: false,
        streetViewControl: false,
        fullscreenControl: false,
      });

      console.log('Map instance created successfully:', mapInstance);
      setMap(mapInstance);
    } catch (error) {
      console.error('Error initializing Google Map:', error);
    }
  };

  // Color mapping for different listing types
  const getMarkerColor = (listingType?: string): string => {
    switch (listingType) {
      case 'sell': return '#10B981'; // Green
      case 'rent': return '#3B82F6'; // Blue  
      case 'lease': return '#8B5CF6'; // Purple
      case 'urgent-sale': return '#EF4444'; // Red
      case 'bidding': return '#F59E0B'; // Orange
      default: return '#0056D2'; // Default blue
    }
  };



  const updateMarkers = () => {
    // Clear existing markers
    markersRef.current.forEach(marker => marker.setMap(null));
    markersRef.current = [];

    if (!properties || properties.length === 0) return;

    // Add new markers
    properties.forEach(property => {
      if (property.latitude && property.longitude) {
        const markerColor = getMarkerColor(property.listing_type);
        
        const marker = new window.google.maps.Marker({
          position: { lat: property.latitude, lng: property.longitude },
          map: map,
          title: property.title,
          icon: {
            url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
              <svg width="40" height="50" viewBox="0 0 40 50" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M20 0C8.95 0 0 8.95 0 20C0 35 20 50 20 50S40 35 40 20C40 8.95 31.05 0 20 0Z" fill="${markerColor}"/>
                <circle cx="20" cy="20" r="8" fill="white"/>
                <text x="20" y="25" text-anchor="middle" fill="${markerColor}" font-size="10" font-weight="bold">₹</text>
              </svg>
            `),
            scaledSize: new window.google.maps.Size(30, 38),
            anchor: new window.google.maps.Point(15, 38)
          }
        });

        // Create info window with minimal property card
        const infoWindow = new window.google.maps.InfoWindow({
          content: `
            <div style="padding: 6px 8px; width: 160px; border-radius: 4px; cursor: pointer; background: white; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;" onclick="window.openPropertyDetail(${property.id})">
              <div style="font-weight: 600; color: #1f2937; margin-bottom: 2px; font-size: 13px; line-height: 1.1;">${property.title}</div>
              <div style="color: #2563eb; font-weight: 700; font-size: 14px; margin-bottom: 1px;">${property.price}</div>
              <div style="color: #6b7280; font-size: 11px;">${property.location}</div>
            </div>
          `
        });

        marker.addListener('click', () => {
          // Close all other info windows
          markersRef.current.forEach(m => {
            if (m.infoWindow) {
              m.infoWindow.close();
            }
          });
          
          infoWindow.open(map, marker);
          
          if (onPropertySelect) {
            onPropertySelect(property);
          }
        });
        
        // Close info window when map is clicked
        map.addListener('click', () => {
          infoWindow.close();
        });

        marker.infoWindow = infoWindow;
        markersRef.current.push(marker);
      }
    });

    // Fit map to show all markers only if no center is specified
    if (markersRef.current.length > 0 && !center) {
      const bounds = new window.google.maps.LatLngBounds();
      markersRef.current.forEach(marker => {
        bounds.extend(marker.getPosition());
      });
      map.fitBounds(bounds);
      
      // Don't zoom in too much for single markers
      const listener = window.google.maps.event.addListener(map, 'idle', () => {
        if (map.getZoom() > 15) map.setZoom(15);
        window.google.maps.event.removeListener(listener);
      });
    }
  };

  return (
    <div className="relative w-full h-full">
      {/* Map Type Toggle */}
      <div className="absolute top-4 left-4 z-10 flex gap-2">
        <button
          onClick={() => setMapType('roadmap')}
          className={`px-3 py-1 rounded-lg shadow-md text-sm font-medium transition-colors ${
            mapType === 'roadmap' 
              ? 'bg-white text-gray-900' 
              : 'bg-gray-100 text-gray-600 hover:bg-white'
          }`}
        >
          Map
        </button>
        <button
          onClick={() => setMapType('satellite')}
          className={`px-3 py-1 rounded-lg shadow-md text-sm font-medium transition-colors ${
            mapType === 'satellite' 
              ? 'bg-white text-gray-900' 
              : 'bg-gray-100 text-gray-600 hover:bg-white'
          }`}
        >
          Satellite
        </button>
      </div>

      {/* Color Legend */}
      <div className="absolute top-4 right-4 z-10 bg-white rounded-lg shadow-md p-3">
        <h4 className="text-sm font-semibold text-gray-800 mb-2">Property Types</h4>
        <div className="space-y-1">
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full" style={{backgroundColor: '#10B981'}}></div>
            <span className="text-xs text-gray-700">For Sale</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full" style={{backgroundColor: '#3B82F6'}}></div>
            <span className="text-xs text-gray-700">For Rent</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full" style={{backgroundColor: '#8B5CF6'}}></div>
            <span className="text-xs text-gray-700">Lease</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full" style={{backgroundColor: '#EF4444'}}></div>
            <span className="text-xs text-gray-700">Urgent Sale</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full" style={{backgroundColor: '#F59E0B'}}></div>
            <span className="text-xs text-gray-700">Bidding</span>
          </div>
        </div>
      </div>

      {/* Map Container */}
      <div 
        ref={mapRef} 
        className="w-full h-full bg-gray-200" 
        style={{ minHeight: height, width: '100%' }}
      />

      {/* Loading/Error State */}
      {!map && (
        <div className="absolute inset-0 bg-gradient-to-br from-blue-100 to-green-100 flex items-center justify-center">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-[#0056D2] mx-auto mb-4"></div>
            <p className="text-blue-600 font-medium">Loading Map...</p>
            {!window.google && (
              <div className="mt-4">
                <p className="text-red-600 text-sm">Google Maps API not loaded</p>
                <p className="text-gray-600 text-xs mt-1">Check console for errors</p>
              </div>
            )}
          </div>
        </div>
      )}
      
      {/* Fallback when Google Maps fails */}
      {!map && (
        <div className="absolute inset-0 bg-gradient-to-br from-blue-50 to-indigo-100 flex items-center justify-center">
          <div className="text-center p-8">
            <div className="text-6xl mb-4">🗺️</div>
            <h3 className="text-lg font-semibold text-gray-700 mb-2">Map View</h3>
            <p className="text-gray-600 text-sm mb-4">Interactive map with property locations</p>
            <div className="bg-white rounded-lg p-4 shadow-sm">
              <p className="text-gray-700 font-medium mb-2">Properties Found: {properties?.length || 0}</p>
              {properties?.slice(0, 3).map((property, index) => (
                <div key={property.id} className="text-left text-sm text-gray-600 mb-1">
                  • {property.title} - {property.location}
                </div>
              ))}
              {(properties?.length || 0) > 3 && (
                <p className="text-xs text-gray-500 mt-2">+{(properties?.length || 0) - 3} more properties</p>
              )}
            </div>
          </div>
        </div>
      )}
    </div>
  );
};
