import React, { useState } from 'react';
import axios from 'axios';

interface GrievanceFormProps {
  userId: number;
  userType: string;
  onSuccess: () => void;
}

const GrievanceForm: React.FC<GrievanceFormProps> = ({ userId, userType, onSuccess }) => {
  const [formData, setFormData] = useState({
    title: '',
    description: '',
    grievance_type: 'property_dispute',
    incident_date: '',
    respondent_id: '',
    respondent_type: '',
    property_id: ''
  });
  const [files, setFiles] = useState<File[]>([]);
  const [loading, setLoading] = useState(false);

  const grievanceTypes = [
    { value: 'property_dispute', label: 'Property Dispute' },
    { value: 'contract_breach', label: 'Contract Breach' },
    { value: 'misrepresentation', label: 'Property Misrepresentation' },
    { value: 'payment_dispute', label: 'Payment Dispute' },
    { value: 'agent_misconduct', label: 'Agent Misconduct' },
    { value: 'maintenance_issue', label: 'Maintenance Issue' },
    { value: 'neighbor_dispute', label: 'Neighbor Dispute' },
    { value: 'legal_issue', label: 'Legal Issue' },
    { value: 'other', label: 'Other' }
  ];

  const userTypes = [
    { value: 'buyer', label: 'Buyer' },
    { value: 'seller', label: 'Seller' },
    { value: 'agent', label: 'Agent' },
    { value: 'landlord', label: 'Landlord' },
    { value: 'tenant', label: 'Tenant' }
  ];

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    const submitData = new FormData();
    submitData.append('complainant_id', String(userId));
    submitData.append('complainant_type', userType);
    submitData.append('title', formData.title);
    submitData.append('description', formData.description);
    submitData.append('grievance_type', formData.grievance_type);
    
    if (formData.incident_date) submitData.append('incident_date', formData.incident_date);
    if (formData.respondent_id) submitData.append('respondent_id', formData.respondent_id);
    if (formData.respondent_type) submitData.append('respondent_type', formData.respondent_type);
    if (formData.property_id) submitData.append('property_id', formData.property_id);

    files.forEach(file => {
      submitData.append('evidence', file);
    });

    try {
      const response = await axios.post('/api/grievances/submit', submitData, {
        headers: { 'Content-Type': 'multipart/form-data' }
      });
      alert(`Grievance submitted successfully! Case ID: ${response.data.case_id}`);
      onSuccess();
    } catch (error) {
      console.error('Error submitting grievance:', error);
      alert('Failed to submit grievance');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="bg-white rounded-lg shadow p-6">
      <h2 className="text-2xl font-bold mb-6">Submit New Grievance</h2>
      <form onSubmit={handleSubmit} className="space-y-4">
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-1">
            Grievance Type *
          </label>
          <select
            value={formData.grievance_type}
            onChange={(e) => setFormData({ ...formData, grievance_type: e.target.value })}
            className="w-full border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-blue-500"
            required
          >
            {grievanceTypes.map(type => (
              <option key={type.value} value={type.value}>{type.label}</option>
            ))}
          </select>
        </div>

        <div>
          <label className="block text-sm font-medium text-gray-700 mb-1">
            Title *
          </label>
          <input
            type="text"
            value={formData.title}
            onChange={(e) => setFormData({ ...formData, title: e.target.value })}
            className="w-full border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-blue-500"
            placeholder="Brief summary of the issue"
            required
          />
        </div>

        <div>
          <label className="block text-sm font-medium text-gray-700 mb-1">
            Description *
          </label>
          <textarea
            value={formData.description}
            onChange={(e) => setFormData({ ...formData, description: e.target.value })}
            className="w-full border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-blue-500"
            rows={5}
            placeholder="Detailed description of the grievance"
            required
          />
        </div>

        <div className="grid grid-cols-2 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Incident Date
            </label>
            <input
              type="datetime-local"
              value={formData.incident_date}
              onChange={(e) => setFormData({ ...formData, incident_date: e.target.value })}
              className="w-full border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-blue-500"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Property ID (Optional)
            </label>
            <input
              type="number"
              value={formData.property_id}
              onChange={(e) => setFormData({ ...formData, property_id: e.target.value })}
              className="w-full border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-blue-500"
              placeholder="Related property ID"
            />
          </div>
        </div>

        <div className="grid grid-cols-2 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Respondent ID (Optional)
            </label>
            <input
              type="number"
              value={formData.respondent_id}
              onChange={(e) => setFormData({ ...formData, respondent_id: e.target.value })}
              className="w-full border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-blue-500"
              placeholder="Person being complained against"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Respondent Type
            </label>
            <select
              value={formData.respondent_type}
              onChange={(e) => setFormData({ ...formData, respondent_type: e.target.value })}
              className="w-full border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-blue-500"
            >
              <option value="">Select type</option>
              {userTypes.map(type => (
                <option key={type.value} value={type.value}>{type.label}</option>
              ))}
            </select>
          </div>
        </div>

        <div>
          <label className="block text-sm font-medium text-gray-700 mb-1">
            Evidence Files (Optional)
          </label>
          <input
            type="file"
            multiple
            onChange={(e) => setFiles(Array.from(e.target.files || []))}
            className="w-full border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-blue-500"
            accept=".pdf,.doc,.docx,.jpg,.jpeg,.png"
          />
          {files.length > 0 && (
            <p className="text-sm text-gray-500 mt-1">{files.length} file(s) selected</p>
          )}
        </div>

        <div className="flex justify-end space-x-3">
          <button
            type="submit"
            disabled={loading}
            className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 disabled:bg-gray-400"
          >
            {loading ? 'Submitting...' : 'Submit Grievance'}
          </button>
        </div>
      </form>
    </div>
  );
};

export default GrievanceForm;
