import { Button } from "./ui/button";
import { ImpersonateButton } from "./admin-dashboard/ImpersonateButton";
import { User, ChevronDown, Crown, Check, Heart, Eye, Bookmark, FileText, Settings, LogOut, Bell, Building, Calendar, X, LayoutDashboard } from "lucide-react";
import { useState, useEffect, useRef } from "react";
import { Link, useLocation, useNavigate } from "react-router-dom";
import { LoginTypeModal } from "./LoginTypeModal";
import logo from "../assets/logo.jpg";
import { getProfile } from "../api/profile";
import { isAdmin, getPrimaryRole } from "../utils/auth";
import { getAdminNotifications, markNotificationAsRead, getUserNotifications, deleteNotification, markUserNotificationAsRead, markAllUserNotificationsAsRead, deleteAllUserNotifications, Notification } from "../api/notifications";
import { getCampaignsByUserId } from "../api/campaigns";
import { getCurrentUserUuid } from "../utils/userUtils";
import { LoadingDots } from "./LoadingDots";
import { useLoading } from "../context/LoadingContext";

// HTML entity decoding function
const decodeHtmlEntities = (text: string): string => {
  if (!text) return '';

  const textarea = document.createElement('textarea');
  textarea.innerHTML = text;
  const decoded = textarea.value;

  return decoded
    .replace(/&#39;/g, "'")
    .replace(/&quot;/g, '"')
    .replace(/&amp;/g, '&')
    .replace(/&lt;/g, '<')
    .replace(/&gt;/g, '>')
    .replace(/&nbsp;/g, ' ');
};

interface User {
  fullName?: string;
  email?: string;
}

function UserNotificationBell({ userId }: { userId: string }) {
  console.log('UserNotificationBell rendered with userId:', userId);
  console.log('User object from localStorage:', JSON.parse(localStorage.getItem('user') || '{}'));
  const [isOpen, setIsOpen] = useState(false);
  const [notifications, setNotifications] = useState<Notification[]>([]);
  const [unreadCount, setUnreadCount] = useState(0);
  const [isLoading, setIsLoading] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const navigate = useNavigate();

  const fetchNotifications = async (forceRefresh = false) => {
    if (!userId || userId === 'user') {
      console.log('Invalid userId, skipping notification fetch:', userId);
      setNotifications([]);
      setUnreadCount(0);
      return;
    }

    try {
      setIsLoading(true);
      console.log('🔔 Fetching user notifications for userId:', userId);
      const data = await getUserNotifications(userId);
      console.log('🔔 User notification data received:', data);
      console.log('🔔 User notifications count:', data.notifications?.length || 0);
      console.log('🔔 User unread count:', data.unread_count || 0);
      let mergedNotifications = [...(data.notifications || [])];
      let extraUnread = 0;

      const currentUserUuid = getCurrentUserUuid();
      if (currentUserUuid) {
        try {
          const campaignResponse = await getCampaignsByUserId(currentUserUuid, { status: 'approved' });
          const approvedCampaigns = campaignResponse?.campaigns || [];
          const approvedCount = approvedCampaigns.length;

          const lastSeenKey = `campaignApprovedLastSeen:${currentUserUuid}`;
          const lastSeenCount = Number(localStorage.getItem(lastSeenKey) || '0');
          const isRead = approvedCount <= lastSeenCount;

          if (approvedCount > 0) {
            mergedNotifications.unshift({
              id: `campaign-approved-${currentUserUuid}`,
              type: 'admin_action',
              title: 'Campaigns Approved',
              message: `You have ${approvedCount} approved campaign${approvedCount > 1 ? 's' : ''}. Start tracking performance now.`,
              time: new Date().toISOString(),
              read: isRead,
              data: { count: approvedCount, status: 'approved' }
            });
          }

          if (!isRead) {
            extraUnread = 1;
          }
        } catch (error) {
          console.error('❌ Error fetching campaign approvals:', error);
        }
      }

      setNotifications(mergedNotifications);
      setUnreadCount((data.unread_count || 0) + extraUnread);
    } catch (error) {
      console.error('❌ Error fetching user notifications:', error);
      setNotifications([]);
      setUnreadCount(0);
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    if (userId) {
      fetchNotifications();
      const interval = setInterval(() => fetchNotifications(true), 60000);

      // Refresh notifications when user comes back to the page
      const handleFocus = () => fetchNotifications(true);
      window.addEventListener('focus', handleFocus);

      // Listen for refresh notifications event (when admin completes review)
      const handleRefreshNotifications = () => {
        console.log('Refreshing user notifications due to admin action');
        fetchNotifications(true);
      };
      window.addEventListener('refreshNotifications', handleRefreshNotifications);

      return () => {
        clearInterval(interval);
        window.removeEventListener('focus', handleFocus);
        window.removeEventListener('refreshNotifications', handleRefreshNotifications);
      };
    }
  }, [userId]);

  useEffect(() => {
    if (isOpen && userId) {
      fetchNotifications(true);
    }
  }, [isOpen]);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setIsOpen(false);
      }
    };

    if (isOpen) {
      document.addEventListener('mousedown', handleClickOutside);
      return () => document.removeEventListener('mousedown', handleClickOutside);
    }
  }, [isOpen]);

  return (
    <div className="relative" ref={dropdownRef}>
      <button
        onClick={(e) => {
          e.preventDefault();
          e.stopPropagation();
          setIsOpen(!isOpen);
        }}
        className="relative inline-flex items-center justify-center w-10 h-10 p-1 rounded-full hover:bg-gray-100 transition"
      >
        <Bell className="w-5 h-5 text-gray-700" />
        {unreadCount > 0 && (
          <span className="absolute top-0 right-0 flex items-center justify-center h-5 min-w-[20px] px-1.5 text-[8px] font-semibold text-white bg-red-600 rounded-full border-2 border-white">
            {unreadCount > 99 ? '99+' : unreadCount}
          </span>
        )}
      </button>

      {isOpen && (
        <div className="absolute right-0 mt-2 bg-white rounded-lg shadow-xl border border-gray-200 z-[9999]" style={{ width: '250px', maxWidth: '250px' }}>
          <div className="px-4 py-3 border-b border-gray-100">
            <div className="flex items-center justify-between">
              <h3 className="text-base font-semibold text-gray-900">Notifications</h3>
              <div className="flex items-center gap-2">
                {notifications.length > 0 && (
                  <button
                    onClick={async () => {
                      try {
                        // Update local state immediately
                        setNotifications([]);
                        setUnreadCount(0);

                        // Delete all notifications from backend
                        await deleteAllUserNotifications(userId);
                      } catch (error) {
                        console.error('Failed to clear all notifications:', error);
                        // Refresh notifications if API call fails
                        fetchNotifications();
                      }
                    }}
                    className="text-xs text-blue-600 hover:text-blue-700 font-medium"
                  >
                    Clear All
                  </button>
                )}
                <button
                  onClick={() => setIsOpen(false)}
                  className="p-1 text-gray-400 hover:text-gray-600 rounded-full transition-colors"
                >
                  <X className="w-4 h-4" />
                </button>
              </div>
            </div>
            <p className="text-sm text-gray-500 mt-1">
              {unreadCount === 0 ? "You're all caught up!" : `${unreadCount} unread notification${unreadCount === 1 ? '' : 's'}`}
            </p>
          </div>

          <div className="overflow-y-auto" style={{ height: '300px' }}>
            {isLoading ? (
              <div className="p-4 flex justify-center">
                <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-blue-600"></div>
              </div>
            ) : notifications.length === 0 ? (
              <div className="p-8 text-center">
                <Bell className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                <p className="text-gray-500 text-sm">No notifications yet</p>
                <p className="text-gray-400 text-xs mt-1">We'll notify you about property updates</p>
              </div>
            ) : (
              <div className="divide-y divide-gray-50">
                {notifications.map((notification) => {
                  let timeStr = 'Just now';
                  try {
                    const date = new Date(notification.time);
                    if (!isNaN(date.getTime())) {
                      timeStr = new Intl.DateTimeFormat('en-US', {
                        month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit'
                      }).format(date);
                    }
                  } catch (e) {
                    timeStr = 'Just now';
                  }

                  return (
                    <div
                      key={notification.id}
                      className="p-4 hover:bg-gray-50 transition-colors cursor-pointer"
                      onClick={async (e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        try {
                          if (notification.id.startsWith('campaign-approved-')) {
                            const currentUserUuid = getCurrentUserUuid();
                            if (currentUserUuid && notification.data?.count) {
                              localStorage.setItem(
                                `campaignApprovedLastSeen:${currentUserUuid}`,
                                String(notification.data.count)
                              );
                            }
                          }
                          if (!notification.read) {
                            // Update local state immediately for better UX
                            setNotifications(prev =>
                              prev.map(n => n.id === notification.id ? { ...n, read: true } : n)
                            );
                            setUnreadCount(prev => Math.max(0, prev - 1));
                            
                            // Mark as read in backend (skip synthetic campaign notification)
                            try {
                              if (!notification.id.startsWith('campaign-approved-')) {
                                await markUserNotificationAsRead(notification.id);
                              }
                            } catch (apiError) {
                              console.error('Failed to mark notification as read:', apiError);
                              // Revert local state if API call fails
                              setNotifications(prev =>
                                prev.map(n => n.id === notification.id ? { ...n, read: false } : n)
                              );
                              setUnreadCount(prev => prev + 1);
                            }
                          }
                          
                          if (notification.id.startsWith('campaign-approved-')) {
                            navigate('/seller/dashboard/campaigns');
                            setIsOpen(false);
                            return;
                          }

                          // Direct navigation to property
                          const propertyId = notification.data?.propertyId || notification.data?.property_id;
                          const visitId = notification.data?.visitId || notification.data?.visit_id;

                          if (propertyId) {
                            navigate(`/property/${propertyId}`);
                          } else if (visitId) {
                            navigate('/user/dashboard');
                          } else {
                            navigate('/user/dashboard');
                          }
                          setIsOpen(false);
                        } catch (error) {
                          console.error('Error handling notification click:', error);
                          setIsOpen(false);
                        }
                      }}
                    >
                      <div className="flex items-start justify-between gap-3">
                        <div className="flex-1 min-w-0">
                          <div className="flex items-center gap-2 mb-1">
                            <h4 className="text-sm font-semibold text-gray-900 break-words">
                              {notification.title ? notification.title.replace(/&#39;/g, "'").replace(/&quot;/g, '"') : 'Notification'}
                            </h4>
                            {!notification.read && (
                              <div className="w-2 h-2 bg-blue-500 rounded-full flex-shrink-0"></div>
                            )}
                          </div>
                          <p className="text-sm text-gray-600 mb-2 break-words whitespace-pre-wrap">
                            {notification.message ? notification.message.replace(/&#39;/g, "'").replace(/&quot;/g, '"') : 'No message'}
                          </p>
                          <div className="flex items-center gap-2">
                            <span className="text-xs text-gray-400">{timeStr}</span>
                            <span className="text-xs px-2 py-0.5 rounded-full font-medium bg-blue-100 text-blue-700">
                              {notification.type}
                            </span>
                          </div>
                        </div>
                        <button
                          onClick={async (e) => {
                            e.stopPropagation();
                            try {
                              // Update local state immediately
                              const wasUnread = !notification.read;
                              setNotifications(prev => prev.filter(n => n.id !== notification.id));
                              if (wasUnread) {
                                setUnreadCount(prev => Math.max(0, prev - 1));
                              }

                              // Delete from backend
                              await deleteNotification(notification.id);
                            } catch (error) {
                              console.error('Failed to delete notification:', error);
                              // Refresh notifications if API call fails
                              fetchNotifications();
                            }
                          }}
                          className="p-1 text-gray-400 hover:text-gray-600 rounded-full transition-all"
                        >
                          <X className="w-3 h-3" />
                        </button>
                      </div>
                    </div>
                  );
                })}
              </div>
            )}
          </div>
        </div>
      )}
    </div>
  );
}

function NotificationBell() {
  const [isOpen, setIsOpen] = useState(false);
  const [notifications, setNotifications] = useState<Notification[]>([]);
  const [totalCount, setTotalCount] = useState(0);
  const [loadingNotification, setLoadingNotification] = useState<number | null>(null);
  const [showFullScreenLoader, setShowFullScreenLoader] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const cacheRef = useRef<{ data: any; timestamp: number } | null>(null);
  const navigate = useNavigate();

  const fetchNotifications = async (useCache = true) => {
    if (useCache && cacheRef.current) {
      const age = Date.now() - cacheRef.current.timestamp;
      if (age < 10000) {
        setNotifications(cacheRef.current.data.notifications);
        setTotalCount(cacheRef.current.data.unread_count);
        return;
      }
    }

    try {
      setIsLoading(true);
      console.log('🔔 Fetching admin notifications...');
      const data = await getAdminNotifications();
      console.log('🔔 Admin notification data received:', data);
      console.log('🔔 Notifications count:', data.notifications?.length || 0);
      console.log('🔔 Unread count:', data.unread_count || 0);
      setNotifications(data.notifications);
      setTotalCount(data.unread_count);

      cacheRef.current = {
        data,
        timestamp: Date.now()
      };
    } catch (error) {
      console.error('❌ Error fetching notifications:', error);
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    fetchNotifications(false);
    const interval = setInterval(() => fetchNotifications(true), 60000);

    // Listen for refresh notifications event
    const handleRefreshNotifications = () => {
      console.log('Refreshing notifications due to admin action');
      fetchNotifications(false);
    };

    window.addEventListener('refreshNotifications', handleRefreshNotifications);

    return () => {
      clearInterval(interval);
      window.removeEventListener('refreshNotifications', handleRefreshNotifications);
    };
  }, []);

  useEffect(() => {
    if (isOpen) {
      fetchNotifications(false);
    }
  }, [isOpen]);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setIsOpen(false);
      }
    };

    if (isOpen) {
      document.addEventListener('mousedown', handleClickOutside);
      return () => document.removeEventListener('mousedown', handleClickOutside);
    }
  }, [isOpen]);

  return (
    <>
      <LoadingDots isVisible={showFullScreenLoader} />

      <div className="relative" ref={dropdownRef}>
        <button
          onClick={() => setIsOpen(!isOpen)}
          aria-label={totalCount > 0 ? `Notifications, ${totalCount} unread` : 'Notifications'}
          aria-expanded={isOpen}
          title="Notifications"
          className="relative inline-flex items-center justify-center w-10 h-10 p-1 rounded-full hover:bg-gray-100 transition focus:outline-none focus-visible:ring-2 focus-visible:ring-offset-2 focus-visible:ring-blue-500 overflow-visible"
        >
          <Bell className="w-5 h-5 text-gray-700 z-10" />
          {totalCount > 0 && (
            <span
              aria-live="polite"
              className="absolute top-0 right-0 z-20 flex items-center justify-center h-5 min-w-[20px] px-1.5 text-[8px] font-semibold text-white bg-red-600 rounded-full border-2 border-white shadow-sm"
              style={{ transform: 'translate(0%, -0%)' }}
            >
              {totalCount > 99 ? '99+' : totalCount}
            </span>
          )}
        </button>

        {isOpen && (
          <div
            role="dialog"
            aria-label="Notifications"
            className="absolute right-0 mt-2 w-96 bg-white rounded-2xl shadow-xl border border-gray-100 z-50 overflow-hidden"
          >
            <div className="px-6 py-4 bg-gradient-to-r from-blue-50 to-indigo-50 border-b border-gray-100">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <div className="w-8 h-8 bg-blue-500 rounded-lg flex items-center justify-center">
                    <Bell className="w-4 h-4 text-white" />
                  </div>
                  <div>
                    <h3 className="text-lg font-semibold text-gray-900">Notifications</h3>
                    <p className="text-xs text-gray-600">
                      {totalCount === 0 ? "All caught up!" : `${totalCount} new`}
                    </p>
                  </div>
                </div>
                <div className="flex items-center gap-2">
                  {notifications.length > 0 && (
                    <button
                      onClick={async () => {
                        try {
                          // Update local state immediately
                          setNotifications([]);
                          setTotalCount(0);
                          cacheRef.current = null;

                          // Call backend to delete all notifications
                          const { deleteAllAdminNotifications } = await import('../api/notifications');
                          await deleteAllAdminNotifications();
                        } catch (error) {
                          console.error('Failed to clear all notifications:', error);
                          // Refresh notifications if API call fails
                          fetchNotifications(false);
                        }
                      }}
                      className="text-xs text-blue-600 hover:text-blue-700 font-medium px-2 py-1 rounded hover:bg-white/50 transition-all"
                    >
                      Clear All
                    </button>
                  )}
                  <button
                    onClick={() => setIsOpen(false)}
                    className="p-2 text-gray-400 hover:text-gray-600 hover:bg-white/50 rounded-lg transition-all"
                  >
                    <X className="w-4 h-4" />
                  </button>
                </div>
              </div>
            </div>

            <div className="max-h-80 overflow-y-auto">
              {isLoading && notifications.length === 0 ? (
                <div className="p-8 flex justify-center">
                  <div className="animate-spin rounded-full h-6 w-6 border-2 border-blue-500 border-t-transparent"></div>
                </div>
              ) : (!notifications || notifications.length === 0) ? (
                <div className="p-8 text-center">
                  <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-3">
                    <Bell className="w-6 h-6 text-gray-400" />
                  </div>
                  <p className="text-sm text-gray-500">No notifications yet</p>
                </div>
              ) : (
                <div className="divide-y divide-gray-50">
                  {notifications.map((notification) => {
                    const timeStr = new Intl.DateTimeFormat('en-US', {
                      month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit'
                    }).format(new Date(notification.time));

                    const handleClick = async () => {
                      try {
                        setLoadingNotification(Number(notification.id));
                        setShowFullScreenLoader(true);
                        setIsOpen(false);

                        setNotifications(prev =>
                          prev.map(n => n.id === notification.id ? { ...n, read: true } : n)
                        );
                        setTotalCount(prev => Math.max(0, prev - 1));

                        markNotificationAsRead(notification.id).catch(console.error);
                        await new Promise(resolve => setTimeout(resolve, 400));

                        // Enhanced navigation logic for admin notifications
                        if (notification.data?.propertyId) {
                          // If we have a specific property ID, navigate to property management with that property
                          navigate('/admin/dashboard', {
                            state: {
                              activeSection: 'PropertyManagement',
                              activeTab: 'pending',
                              propertyId: notification.data.propertyId
                            }
                          });
                        } else if (notification.type === 'property' || notification.type === 'property_approval') {
                          navigate('/admin/dashboard', { state: { activeSection: 'PropertyManagement', activeTab: 'pending' } });
                        } else if (notification.type === 'visit' || notification.type === 'visit_approval') {
                          navigate('/admin/dashboard', { state: { activeSection: 'VisitSchedules', activeTab: 'pending' } });
                        } else if (notification.message.toLowerCase().includes('visit')) {
                          navigate('/admin/dashboard', { state: { activeSection: 'VisitSchedules', activeTab: 'pending' } });
                        } else if (notification.message.toLowerCase().includes('property')) {
                          navigate('/admin/dashboard', { state: { activeSection: 'PropertyManagement', activeTab: 'pending' } });
                        } else {
                          navigate('/admin/dashboard');
                        }

                        setTimeout(() => setShowFullScreenLoader(false), 100);
                        cacheRef.current = null;
                      } catch (err) {
                        console.error('Notification action error', err);
                        setShowFullScreenLoader(false);
                      } finally {
                        setLoadingNotification(null);
                      }
                    };

                    return (
                      <div key={notification.id} className="relative">
                        <button
                          onClick={handleClick}
                          disabled={loadingNotification === Number(notification.id)}
                          className="w-full p-4 text-left hover:bg-gray-50 transition-colors disabled:opacity-50"
                        >
                          <div className="flex items-start justify-between gap-3">
                            <div className="flex-1 min-w-0">
                              <div className="flex items-center gap-2 mb-1">
                                <h4 className="text-sm font-semibold text-gray-900 truncate">
                                  {notification.title}
                                </h4>
                                {!notification.read && (
                                  <div className="w-2 h-2 bg-blue-500 rounded-full flex-shrink-0"></div>
                                )}
                              </div>
                              <p className="text-sm text-gray-600 line-clamp-2 mb-2">
                                {notification.message}
                              </p>
                              <div className="flex items-center gap-2">
                                <span className="text-xs text-gray-400">{timeStr}</span>
                                {notification.type === 'visit' && (
                                  <span className="text-xs px-2 py-0.5 rounded-full font-medium bg-orange-100 text-orange-700">
                                    Visit
                                  </span>
                                )}
                                {notification.type === 'admin_action' && (
                                  <span className="text-xs px-2 py-0.5 rounded-full font-medium bg-green-100 text-green-700">
                                    Confirmation
                                  </span>
                                )}
                              </div>
                            </div>
                            {notification.type === 'admin_action' && (
                              <button
                                onClick={async (e) => {
                                  e.stopPropagation();
                                  try {
                                    // Update local state immediately
                                    const wasUnread = !notification.read;
                                    setNotifications(prev => prev.filter(n => n.id !== notification.id));
                                    if (wasUnread) {
                                      setTotalCount(prev => Math.max(0, prev - 1));
                                    }
                                    cacheRef.current = null;

                                    // Delete from backend
                                    const { deleteAdminNotification } = await import('../api/notifications');
                                    await deleteAdminNotification(notification.id);
                                  } catch (error) {
                                    console.error('Failed to delete notification:', error);
                                    // Refresh notifications if API call fails
                                    fetchNotifications(false);
                                  }
                                }}
                                className="p-1 text-gray-400 hover:text-gray-600 rounded-full transition-all"
                              >
                                <X className="w-3 h-3" />
                              </button>
                            )}
                          </div>
                        </button>
                      </div>
                    );
                  })}
                </div>
              )}
            </div>
          </div>
        )}
      </div>
    </>
  );
}

export function Header() {
  const navigate = useNavigate();
  const location = useLocation();
  const [showPremiumDropdown, setShowPremiumDropdown] = useState(false);
  const [showProfileDropdown, setShowProfileDropdown] = useState(false);
  const [showPropertiesDropdown, setShowPropertiesDropdown] = useState(false);
  const [showServicesDropdown, setShowServicesDropdown] = useState(false);
  const [showAdvisoryDropdown, setShowAdvisoryDropdown] = useState(false);
  const [showSolutionsDropdown, setShowSolutionsDropdown] = useState(false);
  const [showLoginTypeModal, setShowLoginTypeModal] = useState(false);
  const [loginModalContext, setLoginModalContext] = useState<'login' | 'post-property'>('login');
  const [userProfile, setUserProfile] = useState<any>(null);
  const [isLoggingOut, setIsLoggingOut] = useState(false);
  const { isLoginLoading, setLoginLoading } = useLoading();

  // Check if we're in admin dashboard
  const isAdminDashboard = location.pathname.startsWith('/admin/dashboard');

  useEffect(() => {
    const loadUserProfile = async () => {
      const accessToken = localStorage.getItem('access_token');
      const agentToken = localStorage.getItem('agentToken');
      const sellerToken = localStorage.getItem('sellerToken');

      // Skip profile loading for agents/sellers
      if (agentToken || sellerToken) {
        return;
      }

      if (accessToken) {
        try {
          const profile = await getProfile();
          setUserProfile(profile);
          localStorage.setItem('user', JSON.stringify(profile));
        } catch (error) {
          console.error('Failed to load user profile:', error);
        }
      }
    };

    loadUserProfile();

    const handleStorageChange = () => {
      loadUserProfile();
    };

    const handleProfileUpdate = () => {
      loadUserProfile();
    };

    window.addEventListener('storage', handleStorageChange);
    window.addEventListener('profileUpdated', handleProfileUpdate);
    return () => {
      window.removeEventListener('storage', handleStorageChange);
      window.removeEventListener('profileUpdated', handleProfileUpdate);
    };
  }, []);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      const target = event.target as Element;
      if (!target.closest('.profile-dropdown-container')) {
        setShowProfileDropdown(false);
      }
    };

    if (showProfileDropdown) {
      document.addEventListener('mousedown', handleClickOutside);
      return () => document.removeEventListener('mousedown', handleClickOutside);
    }
  }, [showProfileDropdown]);

  const accessToken = localStorage.getItem('access_token');
  const refreshToken = localStorage.getItem('refresh_token');
  const user = userProfile || JSON.parse(localStorage.getItem('user') || '{}');
  const adminLoggedIn = localStorage.getItem('adminLoggedIn');
  const adminName = localStorage.getItem('adminName');

  // Check for agent/seller tokens
  const agentToken = localStorage.getItem('agentToken');
  const sellerToken = localStorage.getItem('sellerToken');
  const agentEmail = localStorage.getItem('agentEmail');
  const sellerEmail = localStorage.getItem('sellerEmail');

  const isLoggedIn = accessToken || adminLoggedIn || agentToken || sellerToken;

  // Debug user object
  console.log('🔍 Header Debug Info:');
  console.log('User object:', user);
  console.log('Is logged in:', isLoggedIn);
  console.log('Is admin:', isAdmin());
  console.log('Access token exists:', !!accessToken);
  console.log('Admin logged in:', !!adminLoggedIn);
  console.log('Should show admin notifications:', isAdmin());
  console.log('Should show user notifications:', !isAdmin() && isLoggedIn);

  const getDashboardPath = () => {
    if (adminLoggedIn) return '/admin/dashboard';
    if (agentToken) return '/agent/dashboard';
    if (sellerToken) return '/seller/dashboard';

    // Check user role from various sources
    const userRole = user.role || user.user_role || getDisplayRole().toLowerCase();

    if (userRole === 'buyer') return '/buyer/dashboard';
    if (userRole === 'seller') return '/seller/dashboard';
    if (userRole === 'agent') return '/agent/dashboard';

    // Default based on tokens or fallback to user dashboard
    return isLoggedIn ? '/user/dashboard' : '/login';
  };

  const getUserDisplayName = () => {
    if (adminLoggedIn) return adminName || 'Admin';
    if (agentToken) return agentEmail || 'Agent';
    if (sellerToken) return sellerEmail || 'Seller';

    // For regular users, try to get email from various sources
    const userEmail = localStorage.getItem('user_email') ||
      localStorage.getItem('login_email') ||
      user.email;

    if (userEmail) {
      return userEmail.split('@')[0];
    }

    return 'User';
  };

  const getUserEmail = () => {
    if (adminLoggedIn) return 'admin@nal.com';
    if (agentToken) return agentEmail;
    if (sellerToken) return sellerEmail;

    return localStorage.getItem('user_email') ||
      localStorage.getItem('login_email') ||
      user.email ||
      'user@nal.com';
  };

  const getUserInitial = () => {
    const displayName = getUserDisplayName();
    return displayName.charAt(0).toUpperCase();
  };

  const getDisplayRole = () => {
    if (agentToken) return 'Agent';
    if (sellerToken) return 'Seller';
    if (isAdmin()) return 'Admin';
    return getPrimaryRole();
  };

  const handleLogout = async () => {
    setIsLoggingOut(true);
    try {
      const refreshToken = localStorage.getItem('refresh_token');
      if (refreshToken) {
        await fetch('/api/auth/logout', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            ...(accessToken && { 'Authorization': `Bearer ${accessToken}` })
          },
          body: JSON.stringify({ refresh_token: refreshToken })
        });
      }
      await new Promise(resolve => setTimeout(resolve, 1000));
    } catch (error) {
      console.error('Logout API error:', error);
    } finally {
      localStorage.clear();
      sessionStorage.clear();

      // Navigate to homepage
      window.location.href = '/';

      setTimeout(() => {
        setIsLoggingOut(false);
      }, 1000);
    }
  };

  const handleSamePageClick = (path: string) => {
    if (location.pathname === path) {
      window.scrollTo({ top: 0, behavior: 'smooth' });
    }
  };

  const handleLoginClick = () => {
    setLoginModalContext('login');
    setShowLoginTypeModal(true);
  };

  // Admin Dashboard Header (simplified) - positioned to account for sidebar
  if (isAdminDashboard) {
    return (
      <>
        <LoadingDots isVisible={isLoggingOut || isLoginLoading} />
        <header className="bg-white shadow-sm border-b border-gray-100 fixed top-0 right-0 z-40" style={{ left: '256px' }}>
          <div className="w-full px-4 sm:px-6 lg:px-8">
            <div className="flex items-center justify-between h-16">
              <div className="flex-shrink-0">
                <h2 className="text-xl font-semibold text-gray-900">Welcome back, {(() => {
                  const accessToken = localStorage.getItem('access_token');
                  const userEmail = localStorage.getItem('user_email');
                  if (accessToken) {
                    try {
                      const payload = JSON.parse(atob(accessToken.split('.')[1]));
                      if (payload.is_admin === true && userEmail) {
                        return userEmail.split('@')[0];
                      }
                    } catch (e) { }
                  }
                  return getUserDisplayName();
                })()}</h2>
              </div>

              <div className="flex items-center space-x-4">
                <ImpersonateButton />
                {isLoggedIn && (
                  <div className="relative profile-dropdown-container">
                    <button
                      onClick={() => setShowProfileDropdown(!showProfileDropdown)}
                      className="flex items-center gap-2 px-3 py-2 rounded-lg hover:bg-gray-50 transition-all duration-200"
                    >
                      <div className="w-8 h-8 bg-blue-500 rounded-full flex items-center justify-center shadow-sm">
                        <span className="text-white text-sm font-medium">{getUserInitial()}</span>
                      </div>
                      <div className="flex flex-col items-start">
                        <span className="text-sm font-medium text-gray-900">{(() => {
                          const accessToken = localStorage.getItem('access_token');
                          if (accessToken) {
                            try {
                              const payload = JSON.parse(atob(accessToken.split('.')[1]));
                              if (payload.is_admin === true) {
                                return localStorage.getItem('user_email') || 'msaikushith718@gmail.com';
                              }
                            } catch (e) { }
                          }
                          return getUserEmail();
                        })()}</span>
                        <span className="text-xs text-gray-500">{(() => {
                          const accessToken = localStorage.getItem('access_token');
                          if (accessToken) {
                            try {
                              const payload = JSON.parse(atob(accessToken.split('.')[1]));
                              if (payload.is_admin === true) {
                                return 'Admin';
                              }
                            } catch (e) { }
                          }
                          return getDisplayRole();
                        })()}</span>
                      </div>
                      <ChevronDown className="w-4 h-4 text-gray-400" />
                    </button>

                    {showProfileDropdown && (
                      <div className="absolute top-full right-0 mt-2 w-64 bg-white border border-gray-200 rounded-xl shadow-xl z-[99999]" style={{ backgroundColor: 'white', opacity: 1 }}>
                        <div className="p-2">
                          <Link to="/profile" className="flex items-center gap-3 px-3 py-2 text-sm text-gray-700 hover:bg-gray-50 rounded-md transition-colors">
                            <User className="w-4 h-4" />
                            View Profile
                          </Link>
                          <Link to="/" className="flex items-center gap-3 px-3 py-2 text-sm text-gray-700 hover:bg-gray-50 rounded-md transition-colors">
                            <Settings className="w-4 h-4" />
                            Settings
                          </Link>
                          <div className="border-t border-gray-100 mt-2 pt-2 relative z-[999999]">
                            <button
                              type="button"
                              onClick={handleLogout}
                              disabled={isLoggingOut}
                              className="w-full flex items-center gap-3 px-3 py-2 text-sm text-gray-700 hover:bg-gray-50 rounded-md transition-colors disabled:opacity-50 relative z-[999999] bg-white"
                            >
                              <LogOut className="w-4 h-4" />
                              Logout
                            </button>
                          </div>
                        </div>
                      </div>
                    )}
                  </div>
                )}
              </div>
            </div>
          </div>
        </header>
      </>
    );
  }

  // Regular Header (full navigation)
  return (
    <>
      <LoadingDots isVisible={isLoggingOut || isLoginLoading} />
      <header className="bg-white shadow-sm border-b border-gray-100 sticky top-0 z-50">
        <div className="w-full px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between h-16">
            <div className="flex-shrink-0">
              <Link
                to="/"
                className="flex items-center"
                onClick={(e) => {
                  // Check if user is admin and prevent redirect loop
                  const accessToken = localStorage.getItem('access_token');
                  if (accessToken) {
                    try {
                      const payload = JSON.parse(atob(accessToken.split('.')[1]));
                      if (payload.is_admin === true) {
                        // Set flag to allow homepage visit
                        sessionStorage.setItem('admin_visited_homepage', 'true');
                      }
                    } catch (e) { }
                  }
                  handleSamePageClick('/');
                }}
              >
                <img src={logo} alt="NAL India" className="w-12 h-12 rounded-lg object-contain" />
                <div className="ml-2 flex flex-col">
                  <span className="text-xl font-bold text-gray-900">NAL India</span>
                  <span className="text-sm text-gray-400">New Age Land</span>
                </div>
              </Link>
            </div>

          <nav className="flex items-center space-x-6 sm:space-x-6 lg:space-x-8">
            <div className="relative properties-dropdown-container"
              onMouseEnter={() => setShowPropertiesDropdown(true)}
              onMouseLeave={() => setShowPropertiesDropdown(false)}
            >
              <button
                className={`flex items-center px-3 py-2 text-sm font-medium transition-colors relative ${
                  location.pathname === '/properties' || location.pathname === '/heatmap'
                    ? 'text-[#0056D2]' 
                    : 'text-gray-700 hover:text-[#0056D2]'
                    }`}
                >
                  Properties
                  <ChevronDown className="w-4 h-4 ml-1" />
                  {(location.pathname === '/properties' || location.pathname === '/heatmap') && (
                    <div className="absolute bottom-0 left-0 w-full h-1 bg-[#0056D2] rounded-full"></div>
                  )}
                </button>

                {showPropertiesDropdown && (
                  <div className="absolute top-full left-0 mt-1 w-48 bg-white border border-gray-200 rounded-lg shadow-lg z-50">
                    <div className="py-2">
                      <Link to="/properties" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/properties'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowPropertiesDropdown(false)}>All Properties</Link>
                      <Link to="/heatmap" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/heatmap'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowPropertiesDropdown(false)}>Heatmap</Link>
                    </div>
                  </div>
                )}
            </div>
            
            
            <div className="relative services-dropdown-container"
              onMouseEnter={() => setShowServicesDropdown(true)}
              onMouseLeave={() => setShowServicesDropdown(false)}
            >
              <button
                className={`flex items-center px-3 py-2 text-sm font-medium transition-colors relative ${
                  location.pathname === '/services' || location.pathname === '/docverification' || location.pathname.startsWith('/services/')
                    ? 'text-[#0056D2]' 
                    : 'text-gray-700 hover:text-[#0056D2]'
                    }`}
                >
                  Services
                  <ChevronDown className="w-4 h-4 ml-1" />
                  {(location.pathname === '/services' || location.pathname === '/docverification' || location.pathname.startsWith('/services/')) && (
                    <div className="absolute bottom-0 left-0 w-full h-1 bg-[#0056D2] rounded-full"></div>
                  )}
                </button>

                {showServicesDropdown && (
                  <div className="absolute top-full left-0 mt-1 w-64 bg-white border border-gray-200 rounded-lg shadow-lg z-50 max-h-96 overflow-y-auto">
                    <div className="py-2">
                      <Link to="/docverification" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/docverification'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>Instant Document Verification</Link>
                      <Link to="/services/digital-vault" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/services/digital-vault'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>Digital Vault</Link>
                      <Link to="/services/legal-loan-assistance" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/services/legal-loan-assistance'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>Legal & Loan Assistance</Link>
                      <Link to="/services/esign-estamp" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/services/esign-estamp'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>E-sign & E-stamp Services</Link>
                      <Link to="/services/property-valuation" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/services/property-valuation'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>Property Valuation</Link>
                      <Link to="/services/market-analysis" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/services/market-analysis'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>Market Analysis</Link>
                      <Link to="/top-markets" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/top-markets'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>Top Performing Markets</Link>
                      <Link to="/micro-markets" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/micro-markets'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>Micro Markets Dashboard</Link>
                      <Link to="/services/title-search" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/services/title-search'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>Title Search & Property History</Link>
                      <Link to="/services/digital-marketing" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/services/digital-marketing' 
                        ? 'text-[#0056D2] font-medium' 
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                      }`} onClick={() => setShowServicesDropdown(false)}>Digital Marketing Services</Link>
                      <Link to="/services/rera-compliance" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/services/rera-compliance'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>RERA Compliance Assistance</Link>
                      <Link to="/services/property-management" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/services/property-management'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>Property Management Services</Link>
                      <Link to="/services/home-insurance" className={`block px-4 py-2 text-sm transition-colors ${location.pathname === '/services/home-insurance'
                        ? 'text-[#0056D2] font-medium'
                        : 'text-gray-700 hover:bg-gray-50 hover:text-[#0056D2]'
                        }`} onClick={() => setShowServicesDropdown(false)}>Home Insurance Assistance</Link>
                    </div>
                  </div>
                )}
            </div>
            <div className="relative advisory-dropdown-container"
              onMouseEnter={() => setShowAdvisoryDropdown(true)}
              onMouseLeave={() => setShowAdvisoryDropdown(false)}
            >
              <button
                className={`flex items-center px-3 py-2 text-sm font-medium transition-colors relative ${
                  location.pathname.startsWith('/advisory')
                    ? 'text-[#0056D2]'
                    : 'text-gray-700 hover:text-[#0056D2]'
                }`}
              >
                Advisory Report
                <ChevronDown className="w-4 h-4 ml-1" />
                {location.pathname.startsWith('/advisory') && (
                  <div className="absolute bottom-0 left-0 w-full h-1 bg-[#0056D2] rounded-full"></div>
                )}
              </button>

              {showAdvisoryDropdown && (
                <div className="absolute top-full left-0 mt-1 w-72 bg-white border border-gray-200 rounded-lg shadow-lg z-50 max-h-96 overflow-y-auto">
                  <div className="py-2">
                    <Link to="/advisory/investment-insights" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>Property Investment Insights</Link>
                    <Link to="/advisory/locality-intelligence" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>Locality Intelligence Report</Link>
                    <Link to="/advisory/risk-compliance" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>Risk & Compliance Check</Link>
                    <Link to="/advisory/property-dossier" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>Custom Property Dossier</Link>
                    <Link to="/advisory/nri-assistance" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>NRI Assistance Desk</Link>
                    <Link to="/advisory/market-trends" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>Market Trends & Price Forecast</Link>
                    <Link to="/rates-trends" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>Rates and Trends</Link>
                    <Link to="/advisory/builder-reputation" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>Builder Reputation & Track Record</Link>
                    <Link to="/advisory/benchmark-reports" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>Comparative Property Benchmark</Link>
                    <Link to="/advisory/legal-health" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>Legal Health & Title Status</Link>
                    <Link to="/advisory/rental-yield" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>Rental Yield Analysis</Link>
                    <Link to="/advisory/custom-report" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowAdvisoryDropdown(false)}>Custom Advisory Report</Link>
                  </div>
                </div>
              )}
            </div>
            <div className="relative solutions-dropdown-container"
              onMouseEnter={() => setShowSolutionsDropdown(true)}
              onMouseLeave={() => setShowSolutionsDropdown(false)}
            >
              <button
                className={`flex items-center px-3 py-2 text-sm font-medium transition-colors relative ${
                  location.pathname.startsWith('/solutions')
                    ? 'text-[#0056D2]' 
                    : 'text-gray-700 hover:text-[#0056D2]'
                }`}
              >
                Solutions
                <ChevronDown className="w-4 h-4 ml-1" />
                {location.pathname.startsWith('/solutions') && (
                  <div className="absolute bottom-0 left-0 w-full h-1 bg-[#0056D2] rounded-full"></div>
                )}
              </button>
              
              {showSolutionsDropdown && (
                <div className="absolute top-full left-0 mt-1 w-72 bg-white border border-gray-200 rounded-lg shadow-lg z-50 max-h-96 overflow-y-auto">
                  <div className="py-2">
                    <Link to="/solutions/asset-management" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowSolutionsDropdown(false)}>Asset Management</Link>
                    <Link to="/solutions/smart-land-maps" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowSolutionsDropdown(false)}>Smart Land Maps</Link>
                    <Link to="/valuation-reports" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowSolutionsDropdown(false)}>Get Valuation Reports</Link>
                    <Link to="/solutions/marketing-services" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowSolutionsDropdown(false)}>Marketing Services</Link>
                    <Link to="/solutions/leads-prospects" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowSolutionsDropdown(false)}>Leads & Prospects</Link>
                    <Link to="/solutions/investors" className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0056D2] transition-colors" onClick={() => setShowSolutionsDropdown(false)}>Investors</Link>
                  </div>
                </div>
              )}
            </div>

            
            <div className="relative premium-dropdown-container"
              onMouseEnter={() => setShowPremiumDropdown(true)}
              onMouseLeave={() => setShowPremiumDropdown(false)}
            >
              <button
                className="flex items-center text-gray-700 hover:text-[#0056D2] px-3 py-2 text-sm font-medium transition-colors"
              >
                NAL Premium
                <ChevronDown className="w-4 h-4 ml-1" />
              </button>
              
              {showPremiumDropdown && (
                <div className="absolute top-full left-0 mt-1 w-72 bg-white border border-gray-200 rounded-lg shadow-lg z-50">
                  <div className="p-4">
                    <div className="flex items-center gap-2 mb-3">
                      <Crown className="w-5 h-5 text-yellow-500" />
                      <h3 className="text-base font-bold text-gray-900">NAL Premium</h3>
                    </div>
                    <p className="text-xs text-gray-600 mb-3">Pay Zero Commission | Save Time & Money</p>
                    
                    <div className="grid grid-cols-2 gap-2 mb-3">
                      <div className="flex items-center gap-1">
                        <Check className="w-3 h-3 text-green-500" />
                        <span className="text-xs text-gray-700">Contact 30 Owners</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <Check className="w-3 h-3 text-green-500" />
                        <span className="text-xs text-gray-700">Premium Properties</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <Check className="w-3 h-3 text-green-500" />
                        <span className="text-xs text-gray-700">Priority Support</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <Check className="w-3 h-3 text-green-500" />
                        <span className="text-xs text-gray-700">Advanced Filters</span>
                      </div>
                    </div>
                    
                    <Link to="/premium">
                      <Button className="w-full bg-[#0056D2] hover:bg-[#0056D2]/90 text-white text-sm h-8">
                        Join Now
                      </Button>
                    </Link>
                  </div>
                </div>
              )}
            </div>
          </nav>

          <div className="flex items-center space-x-4">
            <Button 
              onClick={() => {
                if (!isLoggedIn) {
                  setLoginModalContext('post-property');
                  setShowLoginTypeModal(true);
                } else if (agentToken || sellerToken) {
                  navigate('/list-property');
                } else {
                  alert('Only sellers and agents can post properties');
                }
              }}
              className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white"
            >
              Post Property
            </Button>
            
            {isAdmin() && <NotificationBell />}
            
            {/* Notification Bell for Users */}
            {!isAdmin() && isLoggedIn && (() => {
              // Try to get user ID from multiple sources
              let actualUserId = user.user_id || user.id;
              
              // If not found in user object, try to decode from JWT token
              if (!actualUserId && accessToken) {
                try {
                  const payload = JSON.parse(atob(accessToken.split('.')[1]));
                  actualUserId = payload.user_id;
                } catch (e) {
                  console.error('Failed to decode JWT token:', e);
                }
              }
              
              // Fallback to email if still no user ID
              if (!actualUserId) {
                actualUserId = getUserEmail();
              }
              
              console.log('Final userId for notifications:', actualUserId);
              
              return actualUserId && actualUserId !== 'user' ? (
                <UserNotificationBell userId={actualUserId} />
              ) : null;
            })()}
            
            <LoginTypeModal 
              isOpen={showLoginTypeModal} 
              onClose={() => setShowLoginTypeModal(false)} 
              context={loginModalContext}
            />
            
            {!isLoggedIn ? (
              <Button 
                variant="ghost" 
                size="sm"
                onClick={handleLoginClick}
              >
                <User className="w-4 h-4 mr-2" />
                Login
              </Button>
            ) : (
              <div className="relative profile-dropdown-container">
                <button
                  onClick={() => setShowProfileDropdown(!showProfileDropdown)}
                  className="flex items-center gap-2 px-3 py-2 rounded-lg hover:bg-gray-50 transition-all duration-200"
                >
                  <div className="w-8 h-8 bg-blue-500 rounded-full flex items-center justify-center shadow-sm">
                    <span className="text-white text-sm font-medium">{getUserInitial()}</span>
                  </div>
                  <div className="flex flex-col items-start">
                    <span className="text-sm font-medium text-gray-900">{getUserEmail()}</span>
                    <span className="text-xs text-gray-500">{getDisplayRole()}</span>
                  </div>
                  <ChevronDown className="w-4 h-4 text-gray-400" />
                </button>

                {showProfileDropdown && (
                  <div className="absolute top-full right-0 mt-2 w-64 bg-white border border-gray-200 rounded-xl shadow-xl z-50">
                    <div className="p-2">
                      {!isAdmin() && (
                        <Link
                          to={getDashboardPath()}
                          className="flex items-center gap-3 px-3 py-2 text-sm text-gray-700 hover:bg-gray-50 rounded-md transition-colors"
                          onClick={() => setShowProfileDropdown(false)}
                        >
                          <LayoutDashboard className="w-4 h-4" />
                          Dashboard
                        </Link>
                      )}
                      <Link
                        to="/profile"
                        className="flex items-center gap-3 px-3 py-2 text-sm text-gray-700 hover:bg-gray-50 rounded-md transition-colors"
                        onClick={() => setShowProfileDropdown(false)}
                      >
                        <User className="w-4 h-4" />
                        View Profile
                      </Link>
                      {isAdmin() && (
                        <Link to="/admin/dashboard" className="flex items-center gap-3 px-3 py-2 text-sm text-red-700 hover:bg-red-50 rounded-md transition-colors">
                          <Crown className="w-4 h-4" />
                          Admin Dashboard
                        </Link>
                      )}
                      <div className="border-t border-gray-100 mt-2 pt-2">
                        <button
                          type="button"
                          onClick={handleLogout}
                          disabled={isLoggingOut}
                          className="w-full flex items-center gap-3 px-3 py-2 text-sm text-gray-700 hover:bg-gray-50 rounded-md transition-colors disabled:opacity-50"
                        >
                          <LogOut className="w-4 h-4" />
                          Logout
                        </button>
                      </div>
                    </div>
                  </div>
                )}
              </div>
            )}
          </div>
          </div>
        </div>
      </header>
    </>
  );
}
