import { useState } from "react";
import { useNavigate } from "react-router-dom";
import { Button } from "./ui/button";
import { Search, MapPin, SlidersHorizontal, X } from "lucide-react";
import { toast } from "sonner";
import { INDIAN_CITIES, CITY_LOCALITIES, POPULAR_CITIES, PROPERTY_CATEGORIES, PROPERTY_TYPES, PROPERTY_SUBTYPES } from "../data/cities";

export function HeroSection() {
  const navigate = useNavigate();
  const [showFilters, setShowFilters] = useState(false);
  const [selectedCity, setSelectedCity] = useState('');
  const [selectedLocality, setSelectedLocality] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('');
  const [selectedPropertyType, setSelectedPropertyType] = useState('');
  const [selectedSubType, setSelectedSubType] = useState('');
  const [minPrice, setMinPrice] = useState(0);
  const [maxPrice, setMaxPrice] = useState(10000000);
  const [belowFMV, setBelowFMV] = useState(false);
  const [urgentSale, setUrgentSale] = useState(false);
  const [priceReduced, setPriceReduced] = useState(false);
  const [citySearch, setCitySearch] = useState('');
  const [localitySearch, setLocalitySearch] = useState('');
  const [showAllCities, setShowAllCities] = useState(false);

  const citiesToShow = INDIAN_CITIES;
  const availableSubTypes = selectedPropertyType ? PROPERTY_SUBTYPES[selectedPropertyType] || [] : [];

  const handleSearch = () => {
    if (!selectedCity) {
      toast.error("Please select a city");
      return;
    }
    navigate(`/properties?city=${selectedCity}${selectedLocality ? `&locality=${selectedLocality}` : ''}`);
  };

  return (
    <section className="relative min-h-screen h-screen flex flex-col justify-center overflow-hidden">
      <div className="absolute inset-0" 
        style={{ background: 'linear-gradient(135deg, #1e3a8a 0%, #2563eb 40%, #3b82f6 70%, #60a5fa 100%)' }} 
      />

      <div className="relative w-full px-4 sm:px-6 lg:px-8 flex-1 flex flex-col justify-center py-8">
        <div className="flex gap-2 text-white font-bold mb-6" style={{ fontFamily: 'Inter, sans-serif', fontSize: '70px' }}>
          <span style={{ width: '20px' }}></span>
          <span className="animate-letter-1">F</span>
          <span className="animate-letter-2">i</span>
          <span className="animate-letter-3">n</span>
          <span className="animate-letter-4">d</span>
          <span style={{ width: '15px' }}></span>
          <span className="animate-letter-5">Y</span>
          <span className="animate-letter-6">o</span>
          <span className="animate-letter-7">u</span>
          <span className="animate-letter-8">r</span>
          <span style={{ width: '15px' }}></span>
          <span className="animate-letter-1">P</span>
          <span className="animate-letter-2">e</span>
          <span className="animate-letter-3">r</span>
          <span className="animate-letter-4">f</span>
          <span className="animate-letter-5">e</span>
          <span className="animate-letter-6">c</span>
          <span className="animate-letter-7">t</span>
          <span style={{ width: '15px' }}></span>
          <span className="animate-letter-8">H</span>
          <span className="animate-letter-1">o</span>
          <span className="animate-letter-2">m</span>
          <span className="animate-letter-3">e</span>
        </div>
        
        <div className="max-w-2xl" style={{ marginLeft: '10%' }}>
          <h1 className="text-white font-bold mb-6 text-center" style={{ fontFamily: 'Inter, sans-serif', fontSize: '70px', lineHeight: '1.1' }}>
            with NAL India
          </h1>
          <p className="text-white mb-8 text-center" style={{ fontFamily: 'Inter, sans-serif', fontSize: '18px' }}>
            Buy, sell, and rent verified properties across India with confidence
          </p>
          
          <div className="bg-white rounded-2xl shadow-2xl p-4 max-w-4xl ml-auto mr-8">
            <div className="flex items-center gap-3">
              <div className="flex-1 flex items-center px-4 bg-gray-50 rounded-xl border border-gray-300">
                <MapPin className="w-5 h-5 text-blue-600 mr-2" />
                <div className="flex-1 flex flex-wrap gap-2 items-center py-2">
                  {selectedCity && (
                    <span className="inline-flex items-center gap-1 bg-blue-100 text-blue-700 px-3 py-1 rounded-full text-sm font-medium">
                      {selectedCity}
                      <X className="w-3 h-3 cursor-pointer" onClick={() => { setSelectedCity(''); setSelectedLocality(''); }} />
                    </span>
                  )}
                  {selectedLocality && (
                    <span className="inline-flex items-center gap-1 bg-green-100 text-green-700 px-3 py-1 rounded-full text-sm font-medium">
                      {selectedLocality}
                      <X className="w-3 h-3 cursor-pointer" onClick={() => setSelectedLocality('')} />
                    </span>
                  )}
                  {selectedCategory && (
                    <span className="inline-flex items-center gap-1 bg-purple-100 text-purple-700 px-3 py-1 rounded-full text-sm font-medium">
                      {selectedCategory}
                      <X className="w-3 h-3 cursor-pointer" onClick={() => setSelectedCategory('')} />
                    </span>
                  )}
                  {selectedPropertyType && (
                    <span className="inline-flex items-center gap-1 bg-orange-100 text-orange-700 px-3 py-1 rounded-full text-sm font-medium">
                      {selectedPropertyType}
                      <X className="w-3 h-3 cursor-pointer" onClick={() => { setSelectedPropertyType(''); setSelectedSubType(''); }} />
                    </span>
                  )}
                  {selectedSubType && (
                    <span className="inline-flex items-center gap-1 bg-pink-100 text-pink-700 px-3 py-1 rounded-full text-sm font-medium">
                      {selectedSubType}
                      <X className="w-3 h-3 cursor-pointer" onClick={() => setSelectedSubType('')} />
                    </span>
                  )}
                  {(minPrice > 0 || maxPrice < 10000000) && (
                    <span className="inline-flex items-center gap-1 bg-indigo-100 text-indigo-700 px-3 py-1 rounded-full text-sm font-medium">
                      ₹{(minPrice/100000).toFixed(1)}L - ₹{(maxPrice/100000).toFixed(1)}L
                      <X className="w-3 h-3 cursor-pointer" onClick={() => { setMinPrice(0); setMaxPrice(10000000); }} />
                    </span>
                  )}
                  {belowFMV && (
                    <span className="inline-flex items-center gap-1 bg-red-100 text-red-700 px-3 py-1 rounded-full text-sm font-medium">
                      Below FMV
                      <X className="w-3 h-3 cursor-pointer" onClick={() => setBelowFMV(false)} />
                    </span>
                  )}
                  {urgentSale && (
                    <span className="inline-flex items-center gap-1 bg-yellow-100 text-yellow-700 px-3 py-1 rounded-full text-sm font-medium">
                      Urgent Sale
                      <X className="w-3 h-3 cursor-pointer" onClick={() => setUrgentSale(false)} />
                    </span>
                  )}
                  {priceReduced && (
                    <span className="inline-flex items-center gap-1 bg-teal-100 text-teal-700 px-3 py-1 rounded-full text-sm font-medium">
                      Price Reduced
                      <X className="w-3 h-3 cursor-pointer" onClick={() => setPriceReduced(false)} />
                    </span>
                  )}
                  {!selectedCity && !selectedLocality && !selectedCategory && !selectedPropertyType && !selectedSubType && minPrice === 0 && maxPrice === 10000000 && !belowFMV && !urgentSale && !priceReduced && (
                    <span className="text-gray-500 text-base">Select filters</span>
                  )}
                </div>
              </div>
              <Button 
                className="inline-flex items-center justify-center gap-2 whitespace-nowrap font-medium bg-gray-100 hover:bg-gray-200 text-gray-700 rounded-xl transition-colors h-12 px-4"
                onClick={() => setShowFilters(!showFilters)}
              >
                <SlidersHorizontal className="w-5 h-5" />
              </Button>
              <Button 
                className="inline-flex items-center justify-center gap-2 whitespace-nowrap font-medium bg-blue-500 hover:bg-blue-600 text-white rounded-xl transition-colors h-12 px-6 text-base"
                onClick={handleSearch}
              >
                <Search className="w-5 h-5" />
                Search
              </Button>
            </div>
            
            {showFilters && (
              <div className="mt-4 p-4 bg-gray-50 rounded-xl border border-gray-200">
                <div className="mb-4">
                  <label className="block text-sm font-medium text-gray-700 mb-2">Select City</label>
                  <input
                    type="text"
                    placeholder="Search city..."
                    value={citySearch}
                    onChange={(e) => setCitySearch(e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg mb-3 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                  <div className="grid grid-cols-4 gap-2 mb-2">
                    {citiesToShow
                      .filter(city => city.name.toLowerCase().includes(citySearch.toLowerCase()))
                      .map(city => (
                        <button
                          key={city.name}
                          onClick={() => {
                            setSelectedCity(city.name);
                            setSelectedLocality('');
                            setCitySearch('');
                          }}
                          className={`px-3 py-2 rounded-lg text-sm font-medium transition-colors ${
                            selectedCity === city.name 
                              ? 'bg-blue-500 text-white' 
                              : 'bg-white text-gray-700 hover:bg-blue-50 border border-gray-200'
                          }`}
                        >
                          {city.name}
                        </button>
                      ))}
                  </div>
                </div>
                
                {selectedCity && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Select Locality in {selectedCity}</label>
                    <input
                      type="text"
                      placeholder="Search locality..."
                      value={localitySearch}
                      onChange={(e) => setLocalitySearch(e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg mb-3 focus:outline-none focus:ring-2 focus:ring-blue-500"
                    />
                    <div className="grid grid-cols-3 gap-2 max-h-40 overflow-y-auto">
                      {CITY_LOCALITIES[selectedCity] ? (
                        CITY_LOCALITIES[selectedCity]
                          .filter(locality => locality.toLowerCase().includes(localitySearch.toLowerCase()))
                          .map(locality => (
                            <button
                              key={locality}
                              onClick={() => {
                                setSelectedLocality(locality);
                                setLocalitySearch('');
                              }}
                              className={`px-3 py-2 rounded-lg text-sm font-medium transition-colors ${
                                selectedLocality === locality 
                                  ? 'bg-green-500 text-white' 
                                  : 'bg-white text-gray-700 hover:bg-green-50 border border-gray-200'
                              }`}
                            >
                              {locality}
                            </button>
                          ))
                      ) : (
                        <div className="col-span-3 text-gray-400 text-sm text-center py-2">No localities available</div>
                      )}
                    </div>
                  </div>
                )}
                
                <div className="mb-4">
                  <label className="block text-sm font-medium text-gray-700 mb-2">Property Category</label>
                  <div className="grid grid-cols-5 gap-2">
                    {PROPERTY_CATEGORIES.map(category => (
                      <button
                        key={category}
                        onClick={() => setSelectedCategory(category)}
                        className={`px-3 py-2 rounded-lg text-sm font-medium transition-colors ${
                          selectedCategory === category 
                            ? 'bg-purple-500 text-white' 
                            : 'bg-white text-gray-700 hover:bg-purple-50 border border-gray-200'
                        }`}
                      >
                        {category}
                      </button>
                    ))}
                  </div>
                </div>
                
                <div className="mb-4">
                  <label className="block text-sm font-medium text-gray-700 mb-2">Property Type</label>
                  <div className="grid grid-cols-4 gap-2">
                    {PROPERTY_TYPES.map(type => (
                      <button
                        key={type}
                        onClick={() => {
                          setSelectedPropertyType(type);
                          setSelectedSubType('');
                        }}
                        className={`px-3 py-2 rounded-lg text-sm font-medium transition-colors ${
                          selectedPropertyType === type 
                            ? 'bg-orange-500 text-white' 
                            : 'bg-white text-gray-700 hover:bg-orange-50 border border-gray-200'
                        }`}
                      >
                        {type}
                      </button>
                    ))}
                  </div>
                </div>
                
                {selectedPropertyType && availableSubTypes.length > 0 && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Sub-Type</label>
                    <div className="grid grid-cols-4 gap-2">
                      {availableSubTypes.map(subType => (
                        <button
                          key={subType}
                          onClick={() => setSelectedSubType(subType)}
                          className={`px-3 py-2 rounded-lg text-sm font-medium transition-colors ${
                            selectedSubType === subType 
                              ? 'bg-pink-500 text-white' 
                              : 'bg-white text-gray-700 hover:bg-pink-50 border border-gray-200'
                          }`}
                        >
                          {subType}
                        </button>
                      ))}
                    </div>
                  </div>
                )}
                
                <div className="mb-4">
                  <label className="block text-sm font-medium text-gray-700 mb-2">Budget Range</label>
                  <div className="px-2">
                    <div className="flex justify-between text-sm text-gray-600 mb-2">
                      <span>₹{(minPrice/100000).toFixed(1)}L</span>
                      <span>₹{(maxPrice/100000).toFixed(1)}L</span>
                    </div>
                    <div className="relative">
                      <input
                        type="range"
                        min="0"
                        max="10000000"
                        step="100000"
                        value={minPrice}
                        onChange={(e) => setMinPrice(Math.min(Number(e.target.value), maxPrice - 100000))}
                        className="absolute w-full h-2 bg-transparent appearance-none pointer-events-none z-20"
                        style={{
                          background: 'transparent',
                          WebkitAppearance: 'none'
                        }}
                      />
                      <input
                        type="range"
                        min="0"
                        max="10000000"
                        step="100000"
                        value={maxPrice}
                        onChange={(e) => setMaxPrice(Math.max(Number(e.target.value), minPrice + 100000))}
                        className="absolute w-full h-2 bg-transparent appearance-none pointer-events-none z-20"
                        style={{
                          background: 'transparent',
                          WebkitAppearance: 'none'
                        }}
                      />
                      <div className="relative h-2 bg-gray-200 rounded-full">
                        <div 
                          className="absolute h-2 bg-blue-500 rounded-full"
                          style={{
                            left: `${(minPrice/10000000)*100}%`,
                            right: `${100-(maxPrice/10000000)*100}%`
                          }}
                        />
                      </div>
                    </div>
                  </div>
                  <style>{`
                    input[type="range"]::-webkit-slider-thumb {
                      -webkit-appearance: none;
                      appearance: none;
                      width: 18px;
                      height: 18px;
                      border-radius: 50%;
                      background: #3b82f6;
                      cursor: pointer;
                      pointer-events: all;
                      position: relative;
                      z-index: 30;
                      border: 2px solid white;
                      box-shadow: 0 2px 4px rgba(0,0,0,0.2);
                    }
                    input[type="range"]::-moz-range-thumb {
                      width: 18px;
                      height: 18px;
                      border-radius: 50%;
                      background: #3b82f6;
                      cursor: pointer;
                      pointer-events: all;
                      border: 2px solid white;
                      box-shadow: 0 2px 4px rgba(0,0,0,0.2);
                    }
                  `}</style>
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Special Filters</label>
                  <div className="flex flex-wrap gap-2">
                    <button
                      onClick={() => setBelowFMV(!belowFMV)}
                      className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                        belowFMV 
                          ? 'bg-red-500 text-white' 
                          : 'bg-white text-gray-700 hover:bg-red-50 border border-gray-200'
                      }`}
                    >
                      Below FMV
                    </button>
                    <button
                      onClick={() => setUrgentSale(!urgentSale)}
                      className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                        urgentSale 
                          ? 'bg-yellow-500 text-white' 
                          : 'bg-white text-gray-700 hover:bg-yellow-50 border border-gray-200'
                      }`}
                    >
                      Urgent Sale
                    </button>
                    <button
                      onClick={() => setPriceReduced(!priceReduced)}
                      className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                        priceReduced 
                          ? 'bg-teal-500 text-white' 
                          : 'bg-white text-gray-700 hover:bg-teal-50 border border-gray-200'
                      }`}
                    >
                      Recently Price Reduced
                    </button>
                  </div>
                </div>
              </div>
            )}
          </div>
          
          <div className="flex gap-6 mt-8 text-white">
            <div className="flex items-center gap-2">
              <div className="w-10 h-10 bg-white/20 rounded-full flex items-center justify-center backdrop-blur-sm">
                <svg className="w-6 h-6" fill="currentColor" viewBox="0 0 20 20"><path fillRule="evenodd" d="M6.267 3.455a3.066 3.066 0 001.745-.723 3.066 3.066 0 013.976 0 3.066 3.066 0 001.745.723 3.066 3.066 0 012.812 2.812c.051.643.304 1.254.723 1.745a3.066 3.066 0 010 3.976 3.066 3.066 0 00-.723 1.745 3.066 3.066 0 01-2.812 2.812 3.066 3.066 0 00-1.745.723 3.066 3.066 0 01-3.976 0 3.066 3.066 0 00-1.745-.723 3.066 3.066 0 01-2.812-2.812 3.066 3.066 0 00-.723-1.745 3.066 3.066 0 010-3.976 3.066 3.066 0 00.723-1.745 3.066 3.066 0 012.812-2.812zm7.44 5.252a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd"/></svg>
              </div>
              <span className="font-semibold">Verified Properties</span>
            </div>
            <div className="flex items-center gap-2">
              <div className="w-10 h-10 bg-white/20 rounded-full flex items-center justify-center backdrop-blur-sm">
                <svg className="w-6 h-6" fill="currentColor" viewBox="0 0 20 20"><path d="M9 6a3 3 0 11-6 0 3 3 0 016 0zM17 6a3 3 0 11-6 0 3 3 0 016 0zM12.93 17c.046-.327.07-.66.07-1a6.97 6.97 0 00-1.5-4.33A5 5 0 0119 16v1h-6.07zM6 11a5 5 0 015 5v1H1v-1a5 5 0 015-5z"/></svg>
              </div>
              <span className="font-semibold">Trusted by 10,000+ Users</span>
            </div>
          </div>
        </div>
        
        <div className="absolute bottom-0 right-0">
          <img 
            src="/Building.png" 
            alt="Building" 
            className="animate-slide-up-building"
            style={{ height: '700px', width: 'auto', objectFit: 'contain', filter: 'drop-shadow(0 10px 30px rgba(0,0,0,0.3))' }}
          />
        </div>
      </div>
    </section>
  );
}
