import React, { useState, useEffect } from "react";
import {
  fetchHyperlocalIndicators,
  HyperlocalIndicators,
} from "../api/hyperlocal";

interface HyperlocalInsightsProps {
  propertyId: string;
  propertyLat?: number;
  propertyLon?: number;
  propertyName?: string;
  radius?: number;
}

export const HyperlocalInsights: React.FC<HyperlocalInsightsProps> = ({
  propertyId,
  propertyLat,
  propertyLon,
  propertyName,
  radius = 5000,
}) => {
  const [data, setData] = useState<HyperlocalIndicators | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const loadHyperlocalData = async () => {
      if (!propertyId) {
        setLoading(false);
        setError("Property ID not available");
        return;
      }

      try {
        setLoading(true);
        const indicators = await fetchHyperlocalIndicators(propertyId, radius);
        setData(indicators);
      } catch (err) {
        console.error(err);
        setError("Failed to load hyperlocal data");
      } finally {
        setLoading(false);
      }
    };

    loadHyperlocalData();
  }, [propertyId, radius]);

  const formatDistance = (meters: number) => {
    if (meters === null || meters === undefined) return "N/A";
    if (meters < 1000) return `${meters} m`;
    return `${(meters / 1000).toFixed(1)} km`;
  };

  // ✅ NEW — hide past expected years
  const formatExpectedYear = (year?: number) => {
    if (!year) return "";

    const currentYear = new Date().getFullYear();

    if (year < currentYear) return ""; // past → hide

    return `(${year})`; // current/future → show
  };

  if (loading) return null;
  if (error) return null;

  // hide entire block if nothing exists
  const hasNoData =
    (!data?.metroLines || data.metroLines.length === 0) &&
    (!data?.infrastructure || data.infrastructure.length === 0) &&
    (!data?.zoning || data.zoning.length === 0);

  if (hasNoData) return null;

  return (
    <div className="bg-white rounded-lg border border-gray-200 p-6 mt-4">
      <h3 className="text-lg font-semibold mb-3 flex items-center">
        <svg
          className="w-5 h-5 mr-2 text-blue-600"
          fill="currentColor"
          viewBox="0 0 20 20"
        >
          <path
            fillRule="evenodd"
            d="M5.05 4.05a7 7 0 119.9 9.9L10 18.9l-4.95-4.95a7 7 0 010-9.9zM10 11a2 2 0 100-4 2 2 0 000 4z"
            clipRule="evenodd"
          />
        </svg>
        Hyperlocal Economic Indicators
      </h3>

      <div className="space-y-4 text-sm">
        {/* 🚇 METRO */}
        {data?.metroLines?.length ? (
          <div>
            <div className="font-medium mb-1">🚇 Nearby Metro</div>
            <ul className="list-disc ml-6 space-y-1">
              {data.metroLines.slice(0, 3).map((m, i) => (
                <li key={i}>
                  {m.name} – {formatDistance(m.distanceMeters)}{" "}
                  {formatExpectedYear(m.expectedYear)}
                </li>
              ))}
            </ul>
          </div>
        ) : null}

        {/* 🛣 INFRASTRUCTURE */}
        {data?.infrastructure?.length ? (
          <div>
            <div className="font-medium mb-1">🛣 Infrastructure Projects</div>
            <ul className="list-disc ml-6 space-y-1">
              {data.infrastructure.slice(0, 3).map((p, i) => (
                <li key={i}>
                  {p.name} – {formatDistance(p.distanceMeters)}{" "}
                  {formatExpectedYear(p.expectedYear)}
                </li>
              ))}
            </ul>
          </div>
        ) : null}

        {/* 📘 ZONING */}
        {data?.zoning?.length ? (
          <div>
            <div className="font-medium mb-1">📘 Zoning Changes</div>
            <ul className="list-disc ml-6 space-y-1">
              {data.zoning.slice(0, 3).map((z, i) => (
                <li key={i}>
                  {z.name} – {formatDistance(z.distanceMeters)}
                </li>
              ))}
            </ul>
          </div>
        ) : null}
      </div>
    </div>
  );
};
