import React, { useState } from 'react';
import { Camera, Upload, Search, Zap, Eye, Palette, Grid, MapPin } from 'lucide-react';
import { toast } from 'sonner';

interface ImageSearchDemoProps {
  onSearchComplete?: (results: any[]) => void;
}

export const ImageSearchDemo: React.FC<ImageSearchDemoProps> = ({ onSearchComplete }) => {
  const [selectedImage, setSelectedImage] = useState<File | null>(null);
  const [imagePreview, setImagePreview] = useState<string | null>(null);
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [analysisResults, setAnalysisResults] = useState<any>(null);
  const [searchResults, setSearchResults] = useState<any[]>([]);

  const handleImageSelect = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    // Validate file
    if (file.size > 10 * 1024 * 1024) {
      toast.error('Image file is too large. Please select an image under 10MB.');
      return;
    }

    if (!file.type.startsWith('image/')) {
      toast.error('Please select a valid image file.');
      return;
    }

    setSelectedImage(file);
    
    // Create preview
    const reader = new FileReader();
    reader.onload = (e) => {
      setImagePreview(e.target?.result as string);
    };
    reader.readAsDataURL(file);
  };

  const analyzeImage = async () => {
    if (!selectedImage) return;

    setIsAnalyzing(true);
    try {
      // Import the image search service
      const { imageSearchService } = await import('../services/imageSearchService');
      
      // Extract image features
      const features = await imageSearchService.extractImageFeatures(selectedImage);
      setAnalysisResults(features);
      
      // Search for similar properties
      const searchResult = await imageSearchService.searchSimilarProperties(selectedImage);
      
      if (searchResult.success && searchResult.similarProperties) {
        setSearchResults(searchResult.similarProperties);
        onSearchComplete?.(searchResult.similarProperties);
        toast.success(`Found ${searchResult.similarProperties.length} similar properties!`);
      } else {
        toast.error('No similar properties found. Try a different image.');
      }
      
    } catch (error) {
      console.error('Image analysis error:', error);
      toast.error('Failed to analyze image. Please try again.');
    } finally {
      setIsAnalyzing(false);
    }
  };

  const resetDemo = () => {
    setSelectedImage(null);
    setImagePreview(null);
    setAnalysisResults(null);
    setSearchResults([]);
  };

  return (
    <div className="bg-white rounded-xl shadow-lg p-6 max-w-4xl mx-auto">
      <div className="text-center mb-6">
        <div className="flex items-center justify-center gap-2 mb-2">
          <Camera className="w-6 h-6 text-blue-600" />
          <h2 className="text-2xl font-bold text-gray-900">Image-Based Property Search</h2>
        </div>
        <p className="text-gray-600">
          Upload a property photo to find visually similar properties using AI-powered image analysis
        </p>
      </div>

      {/* Image Upload Section */}
      {!imagePreview ? (
        <div className="border-2 border-dashed border-gray-300 rounded-lg p-8 text-center hover:border-blue-400 transition-colors">
          <input
            type="file"
            accept="image/*"
            onChange={handleImageSelect}
            className="hidden"
            id="image-upload"
          />
          <label htmlFor="image-upload" className="cursor-pointer">
            <Upload className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <p className="text-lg font-medium text-gray-700 mb-2">
              Click to upload a property image
            </p>
            <p className="text-sm text-gray-500">
              Supports JPG, PNG, WebP (max 10MB)
            </p>
          </label>
        </div>
      ) : (
        <div className="space-y-6">
          {/* Image Preview */}
          <div className="relative">
            <img
              src={imagePreview}
              alt="Selected property"
              className="w-full h-64 object-cover rounded-lg shadow-md"
            />
            <button
              onClick={resetDemo}
              className="absolute top-2 right-2 bg-red-500 text-white px-3 py-1 rounded-full text-sm hover:bg-red-600 transition-colors"
            >
              ✕ Remove
            </button>
          </div>

          {/* Analysis Button */}
          <div className="text-center">
            <button
              onClick={analyzeImage}
              disabled={isAnalyzing}
              className="bg-blue-600 text-white px-6 py-3 rounded-lg font-medium hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center gap-2 mx-auto"
            >
              {isAnalyzing ? (
                <>
                  <div className="animate-spin rounded-full h-4 w-4 border-2 border-white border-t-transparent"></div>
                  Analyzing Image...
                </>
              ) : (
                <>
                  <Search className="w-4 h-4" />
                  Find Similar Properties
                </>
              )}
            </button>
          </div>

          {/* Analysis Results */}
          {analysisResults && (
            <div className="bg-gray-50 rounded-lg p-4">
              <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                <Eye className="w-5 h-5" />
                Image Analysis Results
              </h3>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {/* Dominant Colors */}
                <div className="bg-white rounded-lg p-4">
                  <h4 className="font-medium text-gray-800 mb-2 flex items-center gap-2">
                    <Palette className="w-4 h-4" />
                    Dominant Colors
                  </h4>
                  <div className="flex gap-2">
                    {analysisResults.dominantColors.slice(0, 5).map((color: string, index: number) => (
                      <div
                        key={index}
                        className="w-8 h-8 rounded-full border-2 border-gray-200"
                        style={{ backgroundColor: color }}
                        title={color}
                      />
                    ))}
                  </div>
                </div>

                {/* Image Properties */}
                <div className="bg-white rounded-lg p-4">
                  <h4 className="font-medium text-gray-800 mb-2 flex items-center gap-2">
                    <Zap className="w-4 h-4" />
                    Image Properties
                  </h4>
                  <div className="space-y-1 text-sm">
                    <div>Brightness: {Math.round(analysisResults.brightness)}/255</div>
                    <div>Contrast: {Math.round(analysisResults.contrast)}</div>
                    <div>Texture Features: {analysisResults.textureFeatures.length} detected</div>
                  </div>
                </div>

                {/* Detected Objects */}
                <div className="bg-white rounded-lg p-4 md:col-span-2">
                  <h4 className="font-medium text-gray-800 mb-2 flex items-center gap-2">
                    <Grid className="w-4 h-4" />
                    Detected Features
                  </h4>
                  <div className="flex flex-wrap gap-2">
                    {analysisResults.objectDetection.map((object: string, index: number) => (
                      <span
                        key={index}
                        className="bg-blue-100 text-blue-800 px-2 py-1 rounded-full text-xs font-medium"
                      >
                        {object}
                      </span>
                    ))}
                  </div>
                </div>
              </div>
            </div>
          )}

          {/* Search Results */}
          {searchResults.length > 0 && (
            <div className="bg-green-50 rounded-lg p-4">
              <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                <MapPin className="w-5 h-5" />
                Similar Properties Found ({searchResults.length})
              </h3>
              
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {searchResults.slice(0, 6).map((property: any, index: number) => (
                  <div key={property.id || index} className="bg-white rounded-lg p-4 shadow-sm border">
                    {property.imageUrl && (
                      <img
                        src={property.imageUrl}
                        alt={property.title}
                        className="w-full h-32 object-cover rounded-lg mb-3"
                      />
                    )}
                    <h4 className="font-medium text-gray-900 text-sm mb-1 truncate">
                      {property.title}
                    </h4>
                    <p className="text-xs text-gray-600 mb-2">{property.location}</p>
                    <div className="flex justify-between items-center">
                      <span className="text-sm font-semibold text-blue-600">
                        {property.price}
                      </span>
                      {property.similarityScore && (
                        <span className="text-xs bg-green-100 text-green-800 px-2 py-1 rounded-full">
                          {Math.round(property.similarityScore * 100)}% match
                        </span>
                      )}
                    </div>
                  </div>
                ))}
              </div>
              
              {searchResults.length > 6 && (
                <div className="text-center mt-4">
                  <p className="text-sm text-gray-600">
                    + {searchResults.length - 6} more similar properties found
                  </p>
                </div>
              )}
            </div>
          )}
        </div>
      )}

      {/* How It Works Section */}
      <div className="mt-8 bg-blue-50 rounded-lg p-4">
        <h3 className="text-lg font-semibold text-gray-900 mb-3">How Image Search Works</h3>
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4 text-sm">
          <div className="text-center">
            <div className="bg-blue-100 rounded-full w-8 h-8 flex items-center justify-center mx-auto mb-2">
              <Upload className="w-4 h-4 text-blue-600" />
            </div>
            <p className="font-medium">1. Upload Image</p>
            <p className="text-gray-600">Select a property photo</p>
          </div>
          <div className="text-center">
            <div className="bg-blue-100 rounded-full w-8 h-8 flex items-center justify-center mx-auto mb-2">
              <Eye className="w-4 h-4 text-blue-600" />
            </div>
            <p className="font-medium">2. Analyze Features</p>
            <p className="text-gray-600">Extract visual characteristics</p>
          </div>
          <div className="text-center">
            <div className="bg-blue-100 rounded-full w-8 h-8 flex items-center justify-center mx-auto mb-2">
              <Search className="w-4 h-4 text-blue-600" />
            </div>
            <p className="font-medium">3. Find Matches</p>
            <p className="text-gray-600">Compare with property database</p>
          </div>
          <div className="text-center">
            <div className="bg-blue-100 rounded-full w-8 h-8 flex items-center justify-center mx-auto mb-2">
              <MapPin className="w-4 h-4 text-blue-600" />
            </div>
            <p className="font-medium">4. Show Results</p>
            <p className="text-gray-600">Display similar properties</p>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ImageSearchDemo;