import React, { useState, useEffect } from 'react';
import { Heart, User, Mail, Phone, MapPin, DollarSign, FileText, Tag } from 'lucide-react';
import { getToken } from '../api/auth';

interface InterestedButtonProps {
  propertyId: number;
  onLeadCreated?: () => void;
}

import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "./ui/dropdown-menu";

interface LeadFormData {
  // Basic Details
  buyer_name: string;
  buyer_phone: string;
  buyer_email: string;

  // Lead Classification
  lead_type: 'Buyer' | 'Seller';
  requirement: 'Buy' | 'Rent';
  target_role: 'seller' | 'agent';

  // Location & Budget
  city_location: string;
  budget: string;

  // Preferences
  property_type?: string;
  bhk_preference?: string;
  preferred_areas?: string[];
  area_input?: string;
  timeline?: string;

  // Notes
  notes: string;

  // Source (auto-filled)
  source: string;
}

const InterestedButton: React.FC<InterestedButtonProps> = ({ propertyId, onLeadCreated }) => {
  const [showModal, setShowModal] = useState(false);
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState<LeadFormData>({
    // Basic Details
    buyer_name: '',
    buyer_phone: '',
    buyer_email: '',

    // Lead Classification
    lead_type: 'Buyer',
    requirement: 'Buy',
    target_role: 'seller',

    // Location & Budget
    city_location: '',
    budget: '',

    // Preferences
    property_type: '',
    bhk_preference: '',
    preferred_areas: [],
    area_input: '',
    timeline: '',

    // Notes
    notes: '',

    // Source (auto-filled)
    source: 'Website'
  });

  useEffect(() => {
    if (showModal) {
      document.body.style.overflow = 'hidden';
    } else {
      document.body.style.overflow = 'unset';
    }

    return () => {
      document.body.style.overflow = 'unset';
    };
  }, [showModal]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      const token = getToken() || localStorage.getItem('access_token');
      const headers: Record<string, string> = {
        'Content-Type': 'application/json',
      };

      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }

      // Prepare lead data for backend API
      const leadData = {
        property_id: Number(propertyId),
        name: formData.buyer_name,
        email: formData.buyer_email || `lead${Date.now()}@example.com`,
        phone: formData.buyer_phone,
        message: `${formData.notes}\n\nPreferences:\nProperty Type: ${formData.property_type || 'Any'}\nBHK: ${formData.bhk_preference || 'Any'}\nPreferred Areas: ${formData.preferred_areas?.join(', ') || 'None'}\nTimeline: ${formData.timeline || 'Not specified'}\nRequirement: ${formData.requirement}\nLocation: ${formData.city_location}`,
        budget: formData.budget || 'Not specified',
        preferred_contact: 'call',
        lead_type: 'interested',
        target_role: formData.target_role,
      };

      // Send to backend API
      console.log('Sending lead data:', leadData);
      const response = await fetch('http://localhost:8090/api/leads-management/leads', {
        method: 'POST',
        headers: headers,
        body: JSON.stringify(leadData)
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        console.error('API Error Response:', errorData);

        // Handle validation errors
        if (response.status === 422 && errorData.detail) {
          if (Array.isArray(errorData.detail)) {
            const errorMessages = errorData.detail.map((err: any) => `${err.loc?.join('.')} - ${err.msg}`).join(', ');
            throw new Error(`Validation Error: ${errorMessages}`);
          } else {
            throw new Error(`Validation Error: ${errorData.detail}`);
          }
        }

        throw new Error(errorData.detail || errorData.message || 'Failed to submit interest');
      }

      const result = await response.json();
      console.log('Lead created successfully:', result);

      // Reset form and close modal
      setShowModal(false);
      setFormData({
        buyer_name: '',
        buyer_phone: '',
        buyer_email: '',
        lead_type: 'Buyer',
        requirement: 'Buy',
        target_role: 'seller',
        city_location: '',
        budget: '',
        property_type: '',
        bhk_preference: '',
        preferred_areas: [],
        area_input: '',
        timeline: '',
        notes: '',
        source: 'Website'
      });
      onLeadCreated?.();

      // Show success message
      alert('Thank you for your interest! Your inquiry has been submitted successfully.');
    } catch (error) {
      console.error('Error submitting lead:', error);
      const errorMessage = error instanceof Error ? error.message : 'Failed to submit. Please try again.';
      alert(`Error: ${errorMessage}\n\nPlease check your internet connection and try again.`);
    } finally {
      setLoading(false);
    }
  };

  return (
    <>
      <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <button
            className="flex items-center gap-2 bg-red-600 hover:bg-red-700 text-white px-6 py-3 rounded-lg font-semibold transition-colors"
          >
            <Heart className="w-5 h-5" />
            I'm Interested
          </button>
        </DropdownMenuTrigger>
        <DropdownMenuContent align="end" className="w-48 bg-white shadow-lg rounded-md border p-1 z-50">
          <DropdownMenuItem
            className="cursor-pointer px-3 py-2 hover:bg-gray-100 rounded-sm flex items-center gap-2 outline-none"
            onClick={() => {
              setFormData({ ...formData, target_role: 'seller' });
              setShowModal(true);
            }}
          >
            <User className="w-4 h-4" />
            Contact Seller
          </DropdownMenuItem>
          <DropdownMenuItem
            className="cursor-pointer px-3 py-2 hover:bg-gray-100 rounded-sm flex items-center gap-2 outline-none"
            onClick={() => {
              setFormData({ ...formData, target_role: 'agent' });
              setShowModal(true);
            }}
          >
            <User className="w-4 h-4" />
            Contact Agent
          </DropdownMenuItem>
        </DropdownMenuContent>
      </DropdownMenu>

      {showModal && (
        <div className="fixed inset-0 bg-white z-[9999] flex items-center justify-center p-4">
          <div className="bg-white w-full max-w-3xl h-full max-h-[90vh] flex flex-col shadow-2xl rounded-xl border border-gray-100 animate-in fade-in zoom-in duration-200">
            <div className="p-4 border-b flex items-center justify-between bg-white text-gray-900">
              <div className="flex items-center gap-3">
                <button
                  onClick={() => setShowModal(false)}
                  className="flex items-center gap-1 text-gray-600 hover:text-gray-900 transition-colors"
                >
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 19l-7-7m0 0l7-7m-7 7h18" />
                  </svg>
                  <span className="text-sm font-medium">Back to Property</span>
                </button>
                <div className="h-6 w-px bg-gray-300 mx-2"></div>
                <h3 className="text-lg font-bold text-gray-900">
                  {formData.target_role === 'agent' ? 'Contact Agent' : 'Express Your Interest'}
                </h3>
              </div>
              <button onClick={() => setShowModal(false)} className="text-gray-400 hover:text-gray-600">
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            </div>
            <div className="flex-1 overflow-y-auto p-3">
              <form id="interest-form" onSubmit={handleSubmit} className="space-y-2">
                {/* 1. Basic Details */}
                <div className="border-b pb-2">
                  <h4 className="text-sm font-semibold text-gray-800 mb-2">1. Basic Details</h4>
                  <div className="grid grid-cols-1 gap-2">
                    <div>
                      <label className="flex items-center gap-2 text-sm font-medium text-gray-700 mb-1">
                        <User className="w-4 h-4" />
                        Full Name *
                      </label>
                      <input
                        type="text"
                        required
                        value={formData.buyer_name}
                        onChange={(e) => setFormData({ ...formData, buyer_name: e.target.value })}
                        className="w-full px-2 py-1 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-1 focus:ring-red-500"
                        placeholder="Enter your full name"
                      />
                    </div>

                    <div>
                      <label className="flex items-center gap-2 text-sm font-medium text-gray-700 mb-1">
                        <Phone className="w-4 h-4" />
                        Phone Number *
                      </label>
                      <input
                        type="tel"
                        required
                        value={formData.buyer_phone}
                        onChange={(e) => setFormData({ ...formData, buyer_phone: e.target.value })}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-red-500"
                        placeholder="Enter your phone number"
                      />
                    </div>

                    <div>
                      <label className="flex items-center gap-1 text-xs font-medium text-gray-700 mb-1">
                        <Mail className="w-3 h-3" />
                        Email Address
                      </label>
                      <input
                        type="email"
                        value={formData.buyer_email}
                        onChange={(e) => setFormData({ ...formData, buyer_email: e.target.value })}
                        className="w-full px-2 py-1 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-1 focus:ring-red-500"
                        placeholder="Enter your email (optional)"
                      />
                    </div>
                  </div>
                </div>

                {/* 2. Requirement */}
                <div className="border-b pb-2">
                  <h4 className="text-sm font-semibold text-gray-800 mb-2">2. Requirement</h4>
                  <div>
                    <label className="flex items-center gap-2 text-sm font-medium text-gray-700 mb-1">
                      <Tag className="w-4 h-4" />
                      I want to *
                    </label>
                    <select
                      required
                      value={formData.requirement}
                      onChange={(e) => setFormData({ ...formData, requirement: e.target.value as 'Buy' | 'Rent' })}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-red-500"
                    >
                      <option value="Buy">Buy</option>
                      <option value="Rent">Rent</option>
                    </select>
                  </div>
                </div>

                {/* 3. Location & Budget */}
                <div className="border-b pb-2">
                  <h4 className="text-sm font-semibold text-gray-800 mb-2">3. Location & Budget</h4>
                  <div className="grid grid-cols-1 gap-2">
                    <div>
                      <label className="flex items-center gap-2 text-sm font-medium text-gray-700 mb-1">
                        <MapPin className="w-4 h-4" />
                        City / Location *
                      </label>
                      <input
                        type="text"
                        required
                        value={formData.city_location}
                        onChange={(e) => setFormData({ ...formData, city_location: e.target.value })}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-red-500"
                        placeholder="Preferred location or city"
                      />
                    </div>

                    <div>
                      <label className="flex items-center gap-2 text-sm font-medium text-gray-700 mb-1">
                        <span className="text-sm">₹</span>
                        Budget
                      </label>
                      <input
                        type="text"
                        value={formData.budget}
                        onChange={(e) => setFormData({ ...formData, budget: e.target.value })}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-red-500"
                        placeholder="Expected budget (optional)"
                      />
                    </div>
                  </div>
                </div>

                {/* 4. Preferences */}
                <div className="border-b pb-2">
                  <h4 className="text-sm font-semibold text-gray-800 mb-2">4. Preferences</h4>
                  <div className="grid grid-cols-2 gap-2">
                    <div>
                      <label className="text-sm font-medium text-gray-700 mb-1 block">
                        Property Type
                      </label>
                      <select
                        value={formData.property_type || ''}
                        onChange={(e) => setFormData({ ...formData, property_type: e.target.value })}
                        className="w-full px-2 py-1 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-1 focus:ring-red-500"
                      >
                        <option value="">Any</option>
                        <option value="apartment">Apartment</option>
                        <option value="villa">Villa</option>
                        <option value="house">House</option>
                        <option value="commercial">Commercial</option>
                      </select>
                    </div>

                    <div>
                      <label className="text-sm font-medium text-gray-700 mb-1 block">
                        BHK
                      </label>
                      <select
                        value={formData.bhk_preference || ''}
                        onChange={(e) => setFormData({ ...formData, bhk_preference: e.target.value })}
                        className="w-full px-2 py-1 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-1 focus:ring-red-500"
                      >
                        <option value="">Any</option>
                        <option value="1">1 BHK</option>
                        <option value="2">2 BHK</option>
                        <option value="3">3 BHK</option>
                        <option value="4">4 BHK</option>
                        <option value="5+">5+ BHK</option>
                      </select>
                    </div>

                    <div>
                      <label className="text-sm font-medium text-gray-700 mb-1 block">
                        Preferred Areas
                      </label>
                      <div className="space-y-1">
                        <input
                          type="text"
                          value={formData.area_input || ''}
                          onChange={(e) => setFormData({ ...formData, area_input: e.target.value })}
                          onKeyDown={(e) => {
                            if (e.key === 'Enter' || e.key === ',') {
                              e.preventDefault();
                              const area = formData.area_input?.trim();
                              if (area && !formData.preferred_areas?.includes(area)) {
                                setFormData({
                                  ...formData,
                                  preferred_areas: [...(formData.preferred_areas || []), area],
                                  area_input: ''
                                });
                              }
                            }
                          }}
                          className="w-full px-2 py-1 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-1 focus:ring-red-500"
                          placeholder="Type area and press Enter (e.g., Koramangala)"
                        />
                        {formData.preferred_areas && formData.preferred_areas.length > 0 && (
                          <div className="flex flex-wrap gap-1">
                            {formData.preferred_areas.map((area, index) => (
                              <span
                                key={index}
                                className="inline-flex items-center gap-1 px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded-full"
                              >
                                {area}
                                <button
                                  type="button"
                                  onClick={() => {
                                    setFormData({
                                      ...formData,
                                      preferred_areas: formData.preferred_areas?.filter((_, i) => i !== index)
                                    });
                                  }}
                                  className="text-blue-600 hover:text-blue-800"
                                >
                                  ×
                                </button>
                              </span>
                            ))}
                          </div>
                        )}
                      </div>
                    </div>

                    <div>
                      <label className="text-sm font-medium text-gray-700 mb-1 block">
                        Timeline
                      </label>
                      <select
                        value={formData.timeline || ''}
                        onChange={(e) => setFormData({ ...formData, timeline: e.target.value })}
                        className="w-full px-2 py-1 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-1 focus:ring-red-500"
                      >
                        <option value="">Select timeline</option>
                        <option value="immediate">Immediate</option>
                        <option value="1-3months">1-3 months</option>
                        <option value="3-6months">3-6 months</option>
                        <option value="6months+">6+ months</option>
                      </select>
                    </div>
                  </div>
                </div>

                {/* 5. Notes & Attachments */}
                <div>
                  <h4 className="text-sm font-semibold text-gray-800 mb-2">5. Notes & Attachments</h4>
                  <div>
                    <label className="flex items-center gap-2 text-sm font-medium text-gray-700 mb-1">
                      <FileText className="w-4 h-4" />
                      Additional Notes
                    </label>
                    <textarea
                      value={formData.notes}
                      onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                      className="w-full px-2 py-1 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-1 focus:ring-red-500"
                      rows={2}
                      placeholder="Any specific questions, requirements, or additional information?"
                    />
                  </div>
                </div>

              </form>
            </div>
            <div className="p-3 border-t">
              <div className="flex gap-2">
                <button
                  type="button"
                  onClick={() => setShowModal(false)}
                  className="flex-1 px-3 py-1 text-sm border border-gray-300 text-gray-700 rounded-md hover:bg-gray-50"
                  disabled={loading}
                >
                  Cancel
                </button>
                <button
                  type="submit"
                  form="interest-form"
                  disabled={loading}
                  className="flex-1 px-3 py-1 text-sm bg-red-600 text-white rounded-md hover:bg-red-700 disabled:opacity-50"
                >
                  {loading ? 'Submitting...' : 'Submit Interest'}
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </>
  );
};

export default InterestedButton;