import { Card, CardContent } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Calendar, Clock, ArrowRight, TrendingUp, BookOpen } from "lucide-react";
import { useNavigate } from "react-router-dom";
import { useState, useEffect } from "react";
import { api } from "../utils/api";
import { ScrollFadeIn, ScrollStagger, ScrollStaggerItem } from './ui/ScrollAnimation';

export function KnowledgeHub() {
  const navigate = useNavigate();
  const [blogPosts, setBlogPosts] = useState([]);

  useEffect(() => {
    // Force refresh localStorage and fetch blogs
    localStorage.removeItem('cms-blogs');
    const defaultBlogs = [
      { id: 1, title: 'Top 10 Real Estate Investment Tips for 2026', description: 'Expert advice on property investment strategies, market trends, and profitable opportunities in Indian real estate market', category: 'INVESTMENT', status: 'published', views: 2341, date: 'Jan 15, 2024' },
      { id: 2, title: 'Understanding Property Documentation in India', description: 'Complete guide to property papers, legal requirements, and documentation process for buyers and sellers', category: 'DOCUMENTATION', status: 'published', views: 856, date: 'Jan 10, 2024' },
      { id: 3, title: 'How to Price Your Property Right: A Comprehensive Guide ', description: 'Market analysis and pricing strategies for sellers to maximize returns and attract genuine buyers', category: 'PRICING', status: 'published', views: 1876, date: 'Jan 5, 2024' }
    ];
    localStorage.setItem('cms-blogs', JSON.stringify(defaultBlogs));
    
    const loadBlogs = async () => {
      const adminBlogs = await api.fetchBlogs();
      const formattedBlogs = adminBlogs.map((blog: any) => ({
        id: blog.id,
        title: blog.title,
        excerpt: blog.description,
        image: "https://images.unsplash.com/photo-1664892798972-079f15663b16?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxtb2Rlcm4lMjBsdXh1cnklMjBhcGFydG1lbnQlMjBidWlsZGluZ3xlbnwxfHx8fDE3NTY3OTY5ODB8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
        category: blog.category || "General",
        readTime: "5 min read",
        publishedAt: blog.date || "Recently",
        trending: blog.views > 1000
      }));
      setBlogPosts(formattedBlogs);
    };
    
    loadBlogs();
  }, []);

  return (
    <section className="py-16 bg-[#F5F5F5]">
      <div className="w-full px-4 sm:px-6 lg:px-8">
        <ScrollFadeIn>
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Blogs</h2>
            <p className="text-gray-600 max-w-2xl mx-auto">
              Stay informed with expert insights, market trends, and comprehensive guides from real estate professionals
            </p>
          </div>
        </ScrollFadeIn>

        <ScrollStagger className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6" staggerDelay={0.15}>
          {blogPosts.map((post) => (
            <ScrollStaggerItem key={post.id}>
              <Card key={post.id} className="group overflow-hidden hover:shadow-xl transition-all duration-300">
              <div className="relative">
                <img
                  src={post.image}
                  alt={post.title}
                  className="w-full h-48 object-cover group-hover:scale-105 transition-transform duration-300"
                />
                <div className="absolute top-3 left-3 flex gap-2">
                  <Badge className="bg-[#0056D2] text-white text-xs">{post.category}</Badge>
                  {post.trending && (
                    <Badge className="bg-[#00BFA6] text-white text-xs">
                      <TrendingUp className="w-3 h-3 mr-1" />
                      Trending
                    </Badge>
                  )}
                </div>
              </div>

              <CardContent className="p-4">
                <h3 className="font-semibold text-lg text-gray-900 mb-2 line-clamp-2">{post.title}</h3>
                <p className="text-gray-600 text-sm mb-4 line-clamp-2">{post.excerpt}</p>
                <div className="flex items-center justify-between text-xs text-gray-500 mb-4">
                  <div className="flex items-center">
                    <Clock className="w-3 h-3 mr-1" />
                    {post.readTime}
                  </div>
                  <div className="flex items-center">
                    <Calendar className="w-3 h-3 mr-1" />
                    {post.publishedAt}
                  </div>
                </div>
                <Button variant="outline" size="sm" className="w-full group" onClick={() => navigate(`/blog/${post.id}`)}>
                  Read More
                  <ArrowRight className="w-3 h-3 ml-2 group-hover:translate-x-1 transition-transform" />
                </Button>
              </CardContent>
            </Card>
            </ScrollStaggerItem>
          ))}
        </ScrollStagger>
      </div>
    </section>
  );
}